import React, { Fragment, useEffect, useState } from 'react'
import { View } from 'react-native'
import Icon from 'react-native-vector-icons/MaterialIcons'
import { useSetRecoilState } from 'recoil'
import styled from 'styled-components/native'
import { EndOfSamplingSequence } from '../../../../shared/data/story/eossequences'
import {
    isStringDataFormat,
    RawTokenDataFormats,
    TokenData,
    TokenDataFormat,
    tokenDataFromEncoderType,
} from '../../../../shared/data/story/logitbias'
import { splitStringIntoTokens, splitTokenString, tokenStringToTokens } from '../../../../shared/util/tokens'
import { InputSelected } from '../../../globals/state'
import { BodyMedium400M } from '../../../styles/fonts'
import { EncoderType } from '../../../tokenizer/encoder'
import { hexOpacity } from '../../../util/colors'
import { FlexRow } from '../../common/common.style'

function eosTokenize(text: string, encoderType: EncoderType): TokenData {
    if (text.startsWith('[') && text.endsWith(']')) {
        const tokens = tokenStringToTokens(text)
        return new TokenData(tokens.join(','), tokenDataFromEncoderType(encoderType))
    }
    return new TokenData(text.replace(/\\n/g, '\n'), TokenDataFormat.InterpretedString)
}

export interface EosInfo {
    strings: string[]
    type: TokenDataFormat
}

const STOP_SEQUENCE_LIMIT = 8

export function TokenInput(props: {
    placeholder: string
    encoderType: EncoderType
    onTokenSubmit(eosSequences: EndOfSamplingSequence[]): void
    eosSequences: EndOfSamplingSequence[]
}): JSX.Element {
    const [tokenInput, setTokenInput] = useState('')
    const [eosInfo, setEosInfo] = useState<EosInfo[]>([])

    const setInputSelected = useSetRecoilState(InputSelected)

    useEffect(() => {
        const set = async () => {
            const infos = []
            for (const eos of props.eosSequences) {
                switch (eos.sequence.type) {
                    case TokenDataFormat.GPT2Tokens:
                    case TokenDataFormat.PileNaiTokens:
                    case TokenDataFormat.GenjiTokens:
                        infos.push({
                            strings: splitTokenString(eos.sequence.sequence),
                            type: eos.sequence.type,
                        })
                        break
                    default:
                        const strings = await splitStringIntoTokens(
                            eos.sequence.sequence ?? '',
                            props.encoderType
                        )

                        infos.push({
                            strings: strings.map((s) => s.replace(/\n/g, '\\n')),
                            type: eos.sequence.type,
                        })
                        break
                }
            }
            setEosInfo(infos)
        }

        set()
    }, [props.encoderType, props.eosSequences])

    const setToken = () => {
        if (props.eosSequences.length >= STOP_SEQUENCE_LIMIT) return

        if (tokenInput === '') {
            return
        }
        props.onTokenSubmit([
            ...props.eosSequences,
            new EndOfSamplingSequence(eosTokenize(tokenInput, props.encoderType)),
        ])
        setTokenInput('')
    }

    return (
        <View>
            <InputArea>
                <InputBox
                    editable={props.eosSequences.length < STOP_SEQUENCE_LIMIT}
                    placeholder={props.placeholder}
                    onSubmitEditing={setToken}
                    value={tokenInput}
                    onChangeText={(text: string) => setTokenInput(text)}
                    onFocus={() => setInputSelected(true)}
                />
                <AddButton
                    disabled={props.eosSequences.length >= STOP_SEQUENCE_LIMIT}
                    aria-label="Add stop sequence"
                    onPress={setToken}
                >
                    <PlusIcon />
                </AddButton>
            </InputArea>
            <View>
                {props.eosSequences.map((eos, i) => (
                    <TokenDisplay
                        key={i}
                        encoderType={props.encoderType}
                        onPress={() => {
                            props.onTokenSubmit([
                                ...props.eosSequences.slice(0, i),
                                ...props.eosSequences.slice(i + 1),
                            ])
                        }}
                        eosInfo={eosInfo[i]}
                    />
                ))}
            </View>
            {props.eosSequences.length >= STOP_SEQUENCE_LIMIT && (
                <View style={{ paddingTop: 5 }}>
                    <Faded>Limit of {STOP_SEQUENCE_LIMIT} Stop Sequences reached</Faded>
                </View>
            )}
        </View>
    )
}

//Necessary because input box padding breaks sometimes
const InputArea = styled(FlexRow)`
    background-color: ${(props) => props.theme.colors.bg0};
    padding-left: 10px;
`

const InputBox = styled.TextInput.attrs((props) => ({
    placeholderTextColor: hexOpacity(props.theme.colors.textMain, 0.5),
}))`
    ${BodyMedium400M};
    background-color: ${(props) => props.theme.colors.bg0};
    flex-shrink: 1;
    flex-grow: 1;
    padding-vertical: 10px;
    padding-right: 10px;
`

const AddButton = styled.TouchableOpacity`
    background-color: ${(props) => props.theme.colors.bg3};
    flex-basis: 48px;
    height: 48px;
    align-items: center;
    justify-content: center;
`

const PlusIcon = styled(Icon).attrs(() => ({
    name: 'add',
    size: 24,
}))`
    color: ${(props) => props.theme.colors.textMain};
`

const SmallCrossIcon = styled(Icon).attrs(() => ({
    name: 'close',
    size: 16,
}))`
    color: ${(props) => props.theme.colors.textMain};
`

const WarningText = styled.Text`
    color: ${(props) => props.theme.colors.warning};
`

const Faded = styled.Text`
    opacity: 0.4;
`

const TokenText = styled.Text`
    color: ${(props) => props.theme.colors.textMain};
    font-family: ${(props) => props.theme.fonts.code};
    font-size: 12px;
`

const TokenEntry = styled.View`
    background-color: ${(props) => props.theme.colors.bg3};
    flex-direction: row;
    justify-content: space-between;
    align-items: center;
    margin-vertical: 5px;
    padding-left: 10px;
`

const DeleteEntryButton = styled.Pressable`
    padding: 10px;
    margin: 0px;
`

export function TokenDisplay(props: {
    eosInfo: EosInfo
    onPress: () => void
    encoderType: EncoderType
}): JSX.Element {
    let tokenizerMismatch
    if (props.eosInfo && RawTokenDataFormats.has(props.eosInfo?.type)) {
        tokenizerMismatch = props.eosInfo.type !== tokenDataFromEncoderType(props.encoderType)
    }
    return props.eosInfo ? (
        <TokenEntry aria-label="Clear end of sampling token">
            <TokenText>
                {tokenizerMismatch && <WarningText>! </WarningText>}
                {!isStringDataFormat(props.eosInfo.type) ? <Faded>[</Faded> : <Faded>{'{'}</Faded>}
                {props.eosInfo.strings.map((s, i) => (
                    <Fragment key={i}>
                        {s}
                        {i !== props.eosInfo.strings.length - 1 &&
                            (isStringDataFormat(props.eosInfo.type) ? <Faded>|</Faded> : <Faded>, </Faded>)}
                    </Fragment>
                ))}
                {!isStringDataFormat(props.eosInfo.type) ? <Faded>]</Faded> : <Faded>{'}'}</Faded>}
            </TokenText>
            <DeleteEntryButton
                onPress={props.onPress}
                aria-label={`Delete stop sequence ${
                    isStringDataFormat(props.eosInfo.type)
                        ? props.eosInfo.strings.join('')
                        : props.eosInfo.strings.join(', ')
                }`}
            >
                <SmallCrossIcon />
            </DeleteEntryButton>
        </TokenEntry>
    ) : (
        <View />
    )
}
