import React, {
    forwardRef,
    MutableRefObject,
    useCallback,
    useEffect,
    useImperativeHandle,
    useRef,
    useState,
} from 'react'
import { LayoutRectangle, TextInputTextInputEventData, View } from 'react-native'
import { ScrollView } from 'react-native-gesture-handler'
import styled from 'styled-components/native'
import { UniqueId } from '../../../shared/data/document/util'
import { Change, Document } from '../../../shared/data/document/document'
import { EditorSelection, EditorView, EditorViewHandle } from './editorview'
import { EditorModel } from './editormodel'
import { ParagraphBoundariesOverlay, TestOptions } from './testcomponents'
import { logDebug, logError } from '../../util/browser'
import { SectionId, Section } from '../../../shared/data/document/section'
import { DataOrigin } from '../../../shared/components/editor/glue'
import { BodyLarge400 } from '../../styles/fonts'

export enum EditorFormat {
    None = 0,
    Bold = 1,
    Italic = 2,
    Underline = 3,
    Strikethrough = 4,
}

interface EditorInnerState {
    blocked: boolean
    baseMark: DataOrigin
}

export type EditorHandle = {
    document: Document
    state: EditorInnerState
    focus: (focus?: boolean) => void
    reload: () => void
    generate: (inline?: boolean, selection?: EditorSelection | false) => void
    undo: () => void
    redo: (branch?: UniqueId) => void
    format: (format?: EditorFormat) => void
}

export type EditorId = string
export type EditorProps = {
    editorId: EditorId
    document: Document
    onDocumentChange?: (document: Document, editorId: EditorId) => void
    onReload?: (document: Document, editorId: EditorId) => void
    onRequestGeneration?: (
        onResponse: (text: string) => void,
        text: string,
        start?: number,
        end?: number
    ) => Promise<void>
}

export const Editor = forwardRef<EditorHandle, EditorProps>(function Editor(
    { editorId, document, onDocumentChange, onReload, onRequestGeneration },
    ref
): JSX.Element {
    //Core
    const editorViewRef = useRef<EditorViewHandle>(null)
    const documentRef = useRef(document)
    const baseMarkRef = useRef(DataOrigin.user)
    const [editorModel, setEditorModel] = useState(new EditorModel(editorViewRef, documentRef, baseMarkRef))

    //Garbage
    const testHeightRef = useRef<(newText: string) => void>(() => null)
    const [parOverlayShown, setParOverlayShown] = useState(false)
    const [reload, setReload] = useState(0)
    const [redrawNeeded, setRedrawNeeded] = useState(false)
    const heigthsReloadRef = useRef(() => null) //for the test overlay

    useEffect(() => {
        if (redrawNeeded) setRedrawNeeded(false)
    }, [reload])

    //NEW
    const editorIdRef = useRef(editorId)
    const blockedRef = useRef(false)
    const changeRef = useRef(false)

    const onDocumentChangeRef = useRef(onDocumentChange)
    onDocumentChangeRef.current = onDocumentChange

    const onReloadRef = useRef(onReload)
    onReloadRef.current = onReload

    const onRequestGenerationRef = useRef(onRequestGeneration)
    onRequestGenerationRef.current = onRequestGeneration

    const onModelChanged = useCallback((changes: Map<SectionId, Change<Section>>) => {
        if (changes.size > 0) {
            changeRef.current = true
            documentRef.current.pushChange(changes)
            onDocumentChangeRef.current?.call(
                onDocumentChangeRef.current,
                documentRef.current,
                editorIdRef.current
            )
        }
    }, [])

    const pushHistory = useCallback(() => {
        logDebug('push history')
        if (blockedRef.current) return true
        const pushed = documentRef.current.pushHistory()
        if (pushed || changeRef.current) {
            changeRef.current = false
            onDocumentChangeRef.current?.call(
                onDocumentChangeRef.current,
                documentRef.current,
                editorIdRef.current
            )
        }
    }, [])

    const undo = useCallback(() => {
        if (blockedRef.current) return
        const document = documentRef.current
        const changes = document.popHistory()
        logDebug('<= changes', changes)
        if (!changes) return
        editorModel.applyChanges(changes, true)
        onDocumentChangeRef.current?.call(onDocumentChangeRef.current, document, editorIdRef.current)
    }, [])

    const redo = useCallback((branch?: UniqueId) => {
        if (blockedRef.current) return
        const document = documentRef.current
        const changes = document.descendHistory(branch)
        logDebug('=> changes', changes)
        if (!changes) return
        editorModel.applyChanges(changes)
        onDocumentChangeRef.current?.call(onDocumentChangeRef.current, document, editorIdRef.current)
    }, [])

    const generate = useCallback(
        (inline?: boolean, selection?: EditorSelection | false) => {
            if (blockedRef.current || !editorViewRef.current) return
            pushHistory()
            const documentText = documentRef.current.getText()
            const documentRange = editorViewRef.current.focused
                ? editorViewRef.current.empty
                    ? {
                          start: editorViewRef.current.selection.start,
                          end: editorViewRef.current.selection.end,
                      }
                    : {
                          start: editorViewRef.current.selection.start - 1,
                          end: editorViewRef.current.selection.end - 1,
                      }
                : { start: documentText.length, end: documentText.length }
            if (
                !editorViewRef.current.focused ||
                (editorViewRef.current.selection.start === 0 && editorViewRef.current.selection.end === 0)
            )
                editorViewRef.current.selection = editorViewRef.current.empty
                    ? {
                          start: documentText.length,
                          end: documentText.length,
                      }
                    : {
                          start: documentText.length + 1,
                          end: documentText.length + 1,
                      }
            onRequestGenerationRef.current
                ?.call(
                    onRequestGenerationRef.current,
                    (responseText) => {
                        if (!responseText) return
                        console.log('Selection:', editorViewRef.current?.selection)
                        const artificialInput: TextInputTextInputEventData = {
                            text: responseText,
                            previousText: '',
                            range: editorViewRef.current?.selection ?? { start: 0, end: 0 },
                        }
                        console.log('Artificial Input:', artificialInput)
                        editorModel.processInput(artificialInput, true)
                    },
                    documentText,
                    documentRange.start,
                    documentRange.end
                )
                .catch((error) => {
                    logError(error)
                })
                .finally(() => {
                    pushHistory()
                })
        },
        [pushHistory]
    )

    useImperativeHandle(ref, () => ({
        get document() {
            return document
        },
        set state({ blocked, baseMark }: EditorInnerState) {
            blockedRef.current = blocked
            baseMarkRef.current = baseMark
        },
        focus: (f) => queueMicrotask(() => {}),
        reload: () => queueMicrotask(() => editorModel.loadDocument()),
        generate: (i, s) => queueMicrotask(() => generate(i, s)),
        undo: () => queueMicrotask(() => undo()),
        redo: (s) => queueMicrotask(() => redo(s)),
        format: (s) => queueMicrotask(() => {}),
    }))

    console.log('redrew editor!')

    return (
        <>
            <TestOptions
                editorModel={editorModel}
                heigthsReloadRef={heigthsReloadRef}
                redraw={() => {
                    setReload((prev) => prev + 1)
                    setRedrawNeeded(true)
                }}
                setOverlayVisible={setParOverlayShown}
            />
            <EditorFrame>
                {parOverlayShown && editorViewRef && (
                    <ParagraphBoundariesOverlay
                        paragraphData={editorModel.paragraphs}
                        paragraphOrder={editorModel.order}
                        reloadRef={heigthsReloadRef}
                    />
                )}
                {!redrawNeeded && (
                    <EditorView model={editorModel} ref={editorViewRef} onModelChanged={onModelChanged} />
                )}
                <View style={{ height: 50 }} />
                <HeightTester
                    editorViewRef={editorViewRef}
                    changeTextRef={testHeightRef}
                    onTextChange={(measurements) => console.log(measurements)}
                    visible={parOverlayShown}
                />
            </EditorFrame>
        </>
    )
})

function HeightTester(props: {
    editorViewRef: React.RefObject<EditorViewHandle>
    changeTextRef: MutableRefObject<(newText: string) => void>
    onTextChange: (measurements: LayoutRectangle) => void
    visible: boolean
}): JSX.Element {
    const [text, setText] = useState('')
    props.changeTextRef.current = setText

    return (
        <View
            style={{
                width: props.editorViewRef.current?.width ?? 300,
                marginTop: 10,
                position: 'absolute',
                zIndex: props.visible ? 10 : -100,
            }}
        >
            <View
                style={props.visible ? { opacity: 0.2, backgroundColor: 'red' } : {}}
                onLayout={(layoutEvent) => props.onTextChange(layoutEvent.nativeEvent.layout)}
            >
                <BodyLarge400 style={{ opacity: 0 }}>{text}</BodyLarge400>
            </View>
        </View>
    )
}

const EditorFrame = styled(ScrollView)`
    background-color: ${(props) => props.theme.colors.bg0};
    padding-horizontal: 10px;
`
