import React, { forwardRef, RefObject, useImperativeHandle, useState } from 'react'
import { TouchableOpacity } from 'react-native'
import { HistoryNode } from '../../../shared/data/document/history'
import { BodyLarge400 } from '../../styles/fonts'
import { FlexRow } from '../common/common.style'
import { EditorHandle } from './editor'

interface EditorControlInnerState {
    blocked: boolean
    canUndo: boolean
    canRedo: boolean
    canRetry: boolean
    branches: Array<{ node: HistoryNode; preferred: boolean }>
}
export type EditorControlsHandle = {
    state: EditorControlInnerState
}
interface EditorControlsProps {
    editorRef: RefObject<EditorHandle>
}
export const EditorControls = forwardRef<EditorControlsHandle, EditorControlsProps>(function EditorControls(
    { editorRef },
    ref
): JSX.Element {
    const [redoMenuVisible, setRedoMenuVisible] = useState(false)
    const [redoMenuCursor, setRedoMenuCursor] = useState(-1)

    const showRedoMenu = () => {
        setRedoMenuVisible(true)
    }
    const hideRedoMenu = () => {
        setRedoMenuVisible(false)
        setRedoMenuCursor(-1)
    }

    const [canUndo, setCanUndo] = useState(false)
    const [canRedo, setCanRedo] = useState(false)
    const [canRetry, setCanRetry] = useState(false)
    const [blocked, setBlocked] = useState(false)
    const [branches, setBranches] = useState(new Array<{ node: HistoryNode; preferred: boolean }>())
    useImperativeHandle(ref, () => ({
        set state({ blocked, canUndo, canRedo, canRetry, branches }: EditorControlInnerState) {
            setBlocked(blocked)
            setCanUndo(canUndo)
            setCanRedo(canRedo)
            setCanRetry(canRetry)
            setBranches(branches)
            if (!canRedo) {
                hideRedoMenu()
            }
        },
    }))

    return (
        <FlexRow grow={false} style={{ paddingBottom: 10 }}>
            <TouchableOpacity onPress={() => editorRef.current?.reload()} disabled={blocked}>
                <BodyLarge400 style={{ opacity: blocked ? 0.5 : 1 }}>Force Reload</BodyLarge400>
            </TouchableOpacity>
            <TouchableOpacity onPress={() => editorRef.current?.undo()} disabled={blocked || !canUndo}>
                <BodyLarge400 style={{ opacity: blocked || !canUndo ? 0.5 : 1 }}>Undo</BodyLarge400>
            </TouchableOpacity>
            <TouchableOpacity onPress={() => editorRef.current?.redo()} disabled={blocked || !canRedo}>
                <BodyLarge400 style={{ opacity: blocked || !canRedo ? 0.5 : 1 }}>Redo</BodyLarge400>
            </TouchableOpacity>
            <TouchableOpacity onPress={() => editorRef.current?.generate()} disabled={blocked}>
                <BodyLarge400 style={{ opacity: blocked ? 0.5 : 1 }}>Generate</BodyLarge400>
            </TouchableOpacity>
        </FlexRow>
    )
})
