import React, { useEffect } from 'react'
import { StyleProp, ViewStyle } from 'react-native'
import Animated, {
    Easing,
    useAnimatedStyle,
    useSharedValue,
    withDelay,
    withRepeat,
    withTiming,
} from 'react-native-reanimated'
import { useRecoilValue } from 'recoil'
import styled from 'styled-components/native'
import { SiteTheme } from '../../../shared/globals/state'

export const SpinnerFrame = styled.View<{ visible: boolean }>`
    margin: 4px;
    position: relative;
    opacity: ${(props) => (props.visible ? 0.6 : 0)};
    width: 50px;
    height: 50px;
`

const SpinnerRing = (props: { delay: number; width?: number; height?: number }): JSX.Element => {
    const siteTheme = useRecoilValue(SiteTheme)
    const bounce = useSharedValue(0)

    const bounceStyle: StyleProp<ViewStyle> = {
        borderRadius: 25,
        left: 0,
        top: 0,
        position: 'absolute',
        width: props.height ?? 50,
        height: props.width ?? 50,
        backgroundColor: siteTheme.colors.textMain,
    }

    const animationStyle = useAnimatedStyle(() => ({
        transform: [
            {
                scale: bounce.value,
            },
        ],
    }))

    useEffect(() => {
        bounce.value = withDelay(
            props.delay,
            withRepeat(
                withTiming(1, {
                    duration: 2000,
                    easing: Easing.bezier(0.42, 0.0, 0.58, 1.0).factory(),
                }),
                -1,
                true
            )
        )
    }, [])

    return <Animated.View style={[bounceStyle, animationStyle]} />
}

export default function Spinner(props: {
    visible: boolean
    height?: number
    width?: number
    style?: StyleProp<ViewStyle>
}): JSX.Element {
    return (
        <SpinnerFrame visible={props.visible} style={props.style}>
            <SpinnerRing delay={0} width={props.width} height={props.height} />
            <SpinnerRing delay={500} width={props.width} height={props.height} />
        </SpinnerFrame>
    )
}
