﻿import React, { useState } from 'react'
import { TextInput } from 'react-native-gesture-handler'
import { useRecoilCallback, useRecoilValue } from 'recoil'
import styled from 'styled-components/native'
import { StoryMetadata } from '../../../shared/data/story/storycontainer'
import { GlobalUserContext } from '../../../shared/globals/globals'
import { SelectedStory, StoryUpdate } from '../../../shared/globals/state'
import { BodyMedium400, BodyMedium400M, BodyMedium600M, SoftFade } from '../../styles/fonts'
import { hexOpacity } from '../../util/colors'

export function Tags(): JSX.Element {
    const selectedStory = useRecoilValue(SelectedStory)
    const currentStory = GlobalUserContext.stories.get(selectedStory.id)
    const updateStory = useRecoilCallback(({ set }) => (metadata: StoryMetadata) => {
        set(StoryUpdate(metadata.id), metadata.save())
    })

    return (
        <TagEntry>
            <TagEditor
                onTagSubmit={(tag) => {
                    if (currentStory) {
                        currentStory.tags = [...(currentStory?.tags ?? []), tag]
                        updateStory(currentStory)
                    }
                }}
            />

            <TagDisplay
                tagList={currentStory?.tags ?? []}
                onTagClick={(i) => {
                    if (currentStory) {
                        currentStory.tags = [
                            ...currentStory.tags.slice(0, i),
                            ...currentStory.tags.slice(i + 1),
                        ]
                        updateStory(currentStory)
                    }
                }}
            />
        </TagEntry>
    )
}

function TagEditor(props: { onTagSubmit(tag: string): void }): JSX.Element {
    const [tagInput, setTagInput] = useState('')

    return (
        <StyledTagEditor
            placeholder="Type here and submit to save"
            value={tagInput}
            onChangeText={(text) => setTagInput(text)}
            onSubmitEditing={() => {
                if (tagInput !== '') {
                    props.onTagSubmit(tagInput)
                    setTagInput('')
                }
            }}
        />
    )
}

function TagDisplay(props: { tagList?: string[]; onTagClick(index: number): void }): JSX.Element {
    const tags = props.tagList?.map((tag, index) => {
        return (
            <TagItem key={index} onPress={() => props.onTagClick(index)}>
                <BodyMedium400>{tag}</BodyMedium400>
            </TagItem>
        )
    })
    return tags && tags.length > 0 ? (
        <StyledTagDisplay>
            <DeleteText>Tap a tag to delete it.</DeleteText>
            <TagContainer>{tags}</TagContainer>
        </StyledTagDisplay>
    ) : (
        <></>
    )
}

const StyledTagEditor = styled(TextInput).attrs((props) => ({
    placeholderTextColor: hexOpacity(props.theme.colors.textMain, 0.5),
}))`
    ${BodyMedium400M}
    background-color: ${(props) => props.theme.colors.bg0};
    padding-left: 10px;
`

const TagEntry = styled.View`
    border-color: ${(props) => props.theme.colors.bg3};
    border-width: 1px;
    margin-top: 5px;
    margin-bottom: 30px;
`

const StyledTagDisplay = styled.View`
    padding-top: 5px;
`

const DeleteText = styled.Text`
    ${BodyMedium400M};
    ${SoftFade}
    align-self: flex-end;
    margin-right: 10px;
`

const TagContainer = styled.View`
    flex-direction: row;
    flex-wrap: wrap;
    padding-left: 10px;
    margin-top: 5px;
`

const TagItem = styled.TouchableOpacity`
    background-color: ${(props) => props.theme.colors.bg3};
    padding: 10px;
    margin-right: 10px;
    margin-bottom: 10px;
`
