import { HttpException, HttpStatus, Injectable } from "@nestjs/common";
import { InjectDataSource, InjectRepository } from "@nestjs/typeorm";
import { DataSource, FindOptionsWhere, Repository } from "typeorm";
import { setDatabaseConnection, User, UserType } from "./user.entity";
import { UserDataInput } from "./dto/user-data-input.dto";
import { UserData } from "./user-data.entity";
import { UserData as UserDataDto } from "./dto/user-data.dto";
import { TaskPriorityService } from "./task-priority.service";
import { GoogleRecaptchaValidator } from "@nestlab/google-recaptcha";
import { hashAccessKey, hashEmailAddress } from "../shared/access-key-utils";
import { AuthService } from "./auth.service";
import { nanoid } from "nanoid";
import { CreateUserRequest } from "./dto/create-user.dto";
import {
  SubscriptionService,
  TRIAL_ACTIONS_AMOUNT,
} from "./subscription/subscription.service";
import { CounterMetric, PromService } from "@digikare/nestjs-prom";
import {
  RecoveryFinishRequest,
  RecoveryStartRequest,
} from "./dto/recovery.dto";
import { MailgunService } from "@nextnm/nestjs-mailgun";
import { UpdateKeystoreRequest } from "./dto/update-keystore.dto";
import { UserSubmission } from "./user-submission.entity";
import { UserSubmissionInput } from "./dto/user-submission-input.dto";
import { ChangeAccessKeyRequest } from "./dto/change-access-key.dto";
import { UserSubmissionVote } from "./user-submission-vote.entity";
import {
  EmailVerificationRequest,
  EmailVerificationStartRequest,
} from "./dto/email-verification.dto";
import { DeletionStartRequest } from "./dto/deletion.dto";
import AWS = require("aws-sdk");
import { SubscriptionData } from "./subscription/subscription.entity";
import { isTierABetterThanB } from "src/shared/subscription-tiers";

@Injectable()
export class UserService {
  private readonly _registration_counter: CounterMetric;
  private _s3: AWS.S3;

  constructor(
    @InjectRepository(User)
    private usersRepository: Repository<User>,

    @InjectRepository(UserData)
    private userDataRepository: Repository<UserData>,

    @InjectRepository(UserSubmission)
    private userSubmissionRepository: Repository<UserSubmission>,

    @InjectRepository(UserSubmissionVote)
    private userSubmissionVoteRepository: Repository<UserSubmissionVote>,

    private taskPriorityService: TaskPriorityService,
    private authService: AuthService,
    private subscriptionService: SubscriptionService,

    private readonly recaptchaValidator: GoogleRecaptchaValidator,

    private readonly promService: PromService,
    private readonly mailgunService: MailgunService,

    @InjectDataSource()
    private readonly connection: DataSource,
  ) {
    setDatabaseConnection(connection);

    this._registration_counter = this.promService.getCounter({
      name: "app_registration_count",
      help: "Amount of successful registrations",
    });
    if (process.env.AWS_ENDPOINT_ADDR) {
      this._s3 = new AWS.S3({
        endpoint: new AWS.Endpoint(process.env.AWS_ENDPOINT_ADDR),
        credentials: {
          accessKeyId: process.env.AWS_ACCESS_KEY,
          secretAccessKey: process.env.AWS_SECRET_KEY,
        },
      });
    }
  }

  async checkCaptchaToken(captchaToken: string): Promise<boolean> {
    // The temporary things are the most permanent.
    if (process.env.REGISTRATION_PASSWORD !== undefined) {
      return captchaToken === process.env.REGISTRATION_PASSWORD;
    }

    if (
      !process.env.RECAPTCHA_SECRET_KEY ||
      process.env.RECAPTCHA_SECRET_KEY === "disabled"
    )
      return true;

    // TODO: remove this again
    // NOTE: temporary!
    if (process.env.NODE_ENV !== "production") return true;

    const recaptchaResponse = await this.recaptchaValidator.validate({
      response: captchaToken,
      score: 0.3,
      action: "submit",
    });

    return recaptchaResponse.success;
  }

  checkBadEmail(email: string): boolean {
    const isGovDomain = email.includes(".gov.") || email.endsWith(".gov");

    return isGovDomain;
  }

  checkEmailHash(user: User, email: string): boolean {
    return (
      !user.hashedEmail ||
      user.hashedEmail === hashEmailAddress(email) ||
      user.hashedEmail === hashEmailAddress(email.trim()) ||
      user.hashedEmail === hashEmailAddress(email.toLowerCase().trim()) ||
      user.hashedEmail === hashEmailAddress(email.toLowerCase()) ||
      user.hashedEmail ===
        hashEmailAddress(email.slice(0, 1).toUpperCase() + email.slice(1)) ||
      user.hashedEmail ===
        hashEmailAddress(
          email.trim().slice(0, 1).toUpperCase() + email.trim().slice(1),
        )
    );
  }

  async findUserByEmail(email: string): Promise<User | undefined> {
    let user = undefined;
    user = await this.usersRepository.findOneBy({
      hashedEmail: hashEmailAddress(email),
    });
    if (user) return user;
    user = await this.usersRepository.findOneBy({
      hashedEmail: hashEmailAddress(email.trim()),
    });
    if (user) return user;
    user = await this.usersRepository.findOneBy({
      hashedEmail: hashEmailAddress(email.toLowerCase().trim()),
    });
    if (user) return user;
    user = await this.usersRepository.findOneBy({
      hashedEmail: hashEmailAddress(email.toLowerCase()),
    });
    if (user) return user;
    user = await this.usersRepository.findOneBy({
      hashedEmail: hashEmailAddress(
        email.slice(0, 1).toUpperCase() + email.slice(1),
      ),
    });
    if (user) return user;
    user = await this.usersRepository.findOneBy({
      hashedEmail:
        email.trim().slice(0, 1).toUpperCase() + email.trim().slice(1),
    });
    if (user) return user;
    return undefined;
  }

  async createUser(createUserDto: CreateUserRequest): Promise<string> {
    const accessKeyHash = hashAccessKey(createUserDto.key);
    if (await this.userByAccessKeyHash(accessKeyHash))
      throw new HttpException("Incorrect access key.", HttpStatus.CONFLICT);

    if (createUserDto.email) {
      createUserDto.email = createUserDto.email.toLowerCase();
      createUserDto.email = createUserDto.email.trim();
    }

    if (createUserDto.emailCleartext) {
      createUserDto.emailCleartext = createUserDto.emailCleartext.toLowerCase();
      createUserDto.emailCleartext = createUserDto.emailCleartext.trim();
    }

    if (
      createUserDto.emailCleartext &&
      createUserDto.emailCleartext.length > 0 &&
      (await this.findUser({
        hashedEmail: hashEmailAddress(createUserDto.emailCleartext),
      }))
    )
      throw new HttpException(
        "Account already registered.",
        HttpStatus.CONFLICT,
      );
    else if (
      createUserDto.email &&
      createUserDto.email.length > 0 &&
      (await this.findUser({
        hashedEmail: createUserDto.email,
      }))
    )
      throw new HttpException(
        "Account already registered.",
        HttpStatus.CONFLICT,
      );

    if (
      process.env.REGISTRATION_DISABLED &&
      (!createUserDto.giftkey ||
        !(await this.subscriptionService.checkGiftKey(createUserDto.giftkey)))
    ) {
      throw new HttpException(
        "Registration without gift key disabled.",
        HttpStatus.FORBIDDEN,
      );
    }

    const newUser = new User();

    newUser.accessKey = accessKeyHash;
    if (createUserDto.emailCleartext)
      await this.initEmailVerificationForUser(
        newUser,
        createUserDto.emailCleartext,
      );
    else if (createUserDto.email) newUser.hashedEmail = createUserDto.email;

    newUser.authenticationNonce = nanoid();

    newUser.userType = UserType.RETAIL;

    if (process.env.DEFAULT_SUBSCRIPTION !== undefined) {
      newUser.subscriptionTier = +process.env.DEFAULT_SUBSCRIPTION;
      newUser.subscriptionUntil = new Date(
        (process.env.DEFAULT_SUBSCRIPTION_LENGTH !== undefined
          ? Math.floor(+new Date() / 1000) +
            parseInt(process.env.DEFAULT_SUBSCRIPTION_LENGTH)
          : 2147483647) * 1000,
      );

      await this.taskPriorityService.resetPriority(newUser);
    }

    if (
      createUserDto.giftkey &&
      createUserDto.giftkey != "" &&
      !newUser.hasSubscription()
    ) {
      await this.subscriptionService.bindSubscription(newUser, {
        paymentProcessor: "giftkey",
        subscriptionId: createUserDto.giftkey,
      });
    }

    this._registration_counter.inc();

    await this.usersRepository.save(newUser);

    return this.authService.generateTokenForUser(newUser);
  }

  async verifyEmail(emailVerificationDto: EmailVerificationRequest) {
    const user = await this.findUser({
      emailVerificationToken: hashAccessKey(
        emailVerificationDto.verificationToken,
      ),
    });
    if (!user)
      throw new HttpException("Token not found.", HttpStatus.UNAUTHORIZED);

    user.emailVerified = true;
    user.emailVerificationToken = null;

    // Free trial logic: activate trial upon email verification
    if (!user.trialActivated) {
      user.trialActivated = true;
      user.trialActions = TRIAL_ACTIONS_AMOUNT;
    }

    await this.usersRepository.save(user);
  }

  async initEmailVerificationForUser(user: User, email: string) {
    // if (!await this.mailgunService.validateEmail(email))
    //  throw new HttpException("Incorrect email address.", HttpStatus.CONFLICT);

    // For safe measure
    email = email.trim().toLowerCase();

    if (this.checkBadEmail(email))
      throw new HttpException("Bad email address.", HttpStatus.CONFLICT);

    user.emailVerified = false;
    user.hashedEmail = hashEmailAddress(email);

    const token = nanoid(64);
    user.emailVerificationToken = hashAccessKey(token);
    user.emailVerificationLastSentAt = new Date();

    await this.mailgunService.createEmail(process.env.MAILGUN_DOMAIN, {
      from: `NovelAI <noreply@${process.env.MAILGUN_DOMAIN}>`,
      to: email.trim(),
      subject: "NovelAI - Account Confirmation",
      template: "account_confirmation",
      "h:X-Mailgun-Variables": JSON.stringify({
        confirm_link: "https://novelai.net/confirmation?token=" + token,
        confirmation_token: token,
      }),
    });
  }

  async startEmailVerification(
    emailVerificationDto: EmailVerificationStartRequest,
  ) {
    const EMAIL_SEND_TIMEOUT_HOURS = 24;

    const user = await this.findUserByEmail(emailVerificationDto.email);
    if (!user) return;

    if (user.emailVerified) return;
    /* throw new HttpException("User is already verified.", HttpStatus.CONFLICT); */

    /*if (hashEmailAddress(emailVerificationDto.email) !== user.hashedEmail)
      throw new HttpException(
        "Specified email address does not match the stored hashed value. If you want to verify a different address, you have to change it through changeAccessKey first.",
        HttpStatus.CONFLICT,
      );*/

    const emailVerificationLastUnix = Math.floor(
      +user.emailVerificationLastSentAt / 1000,
    );
    const nowUnix = Math.floor(+new Date() / 1000);
    const timeoutUnix = EMAIL_SEND_TIMEOUT_HOURS * 60 * 60;

    if (nowUnix - emailVerificationLastUnix < timeoutUnix) return;
    /*throw new HttpException(
      `The verification email has already been sent in the past ${EMAIL_SEND_TIMEOUT_HOURS} hours. If you have not received it within 15 minutes since the request, please contact the support at support@novelai.net.`,
      HttpStatus.CONFLICT,
    );*/

    await this.initEmailVerificationForUser(user, emailVerificationDto.email);
    await this.usersRepository.save(user);
  }

  async loginUser(accessKey: string): Promise<string> {
    const user = await this.userByAccessKeyHash(hashAccessKey(accessKey));
    if (!user)
      throw new HttpException("Incorrect access key.", HttpStatus.UNAUTHORIZED);

    return this.authService.generateTokenForUser(user);
  }

  async changeUserAccessKey(
    user: User,
    changeAccessKeyDto: ChangeAccessKeyRequest,
  ) {
    if (user.accessKey !== hashAccessKey(changeAccessKeyDto.currentAccessKey))
      throw new HttpException(
        "Incorrect currentAccessKey.",
        HttpStatus.UNAUTHORIZED,
      );

    const newAccessKeyHash = hashAccessKey(changeAccessKeyDto.newAccessKey);
    if (await this.userByAccessKeyHash(newAccessKeyHash))
      throw new HttpException("Incorrect newAccessKey.", HttpStatus.CONFLICT);

    user.accessKey = newAccessKeyHash;
    user.authenticationNonce = nanoid();

    if (changeAccessKeyDto.newEmail) {
      changeAccessKeyDto.newEmail = changeAccessKeyDto.newEmail.toLowerCase();
      changeAccessKeyDto.newEmail = changeAccessKeyDto.newEmail.trim();

      const emailHash = hashEmailAddress(changeAccessKeyDto.newEmail);

      if (user.hashedEmail && user.hashedEmail === emailHash)
        throw new HttpException(
          "New email address is identical to the old email address.",
          HttpStatus.CONFLICT,
        );

      if (await this.findUser({ hashedEmail: emailHash }))
        throw new HttpException(
          "This email address is already registered.",
          HttpStatus.CONFLICT,
        );

      if (this.checkBadEmail(changeAccessKeyDto.newEmail))
        throw new HttpException("Bad email address.", HttpStatus.CONFLICT);

      // Edge case for legit users and a major vulnerability for potential service disruptors:
      // - Emails are not that cheap
      // - Spamming emails could potentially lead to a domain blacklist by major email providers
      /*if (!user.emailVerified && user.emailVerificationToken !== null)
        throw new HttpException(
          "Please finish verification of the previous email address beforehand. If you are unable to do that, please contact support at support@novelai.net.",
          HttpStatus.CONFLICT,
        );*/

      await this.initEmailVerificationForUser(
        user,
        changeAccessKeyDto.newEmail,
      );
    }

    await this.usersRepository.save(user);

    return this.authService.generateTokenForUser(user);
  }

  async updateKeystore(
    user: User,
    updateKeystoreRequest: UpdateKeystoreRequest,
  ) {
    if (
      !!updateKeystoreRequest.changeIndex &&
      updateKeystoreRequest.changeIndex != user.keystoreChangeIndex
    )
      throw new HttpException(
        `Incorrect change index (expected ${user.keystoreChangeIndex}, got ${updateKeystoreRequest.changeIndex}).`,
        HttpStatus.CONFLICT,
      );

    const newKeystore = updateKeystoreRequest.keystore;

    user.keystore =
      newKeystore.length > 0 ? Buffer.from(newKeystore, "base64") : null;
    user.keystoreChangeIndex++;

    await this.usersRepository.query(
      `UPDATE "user" SET "keystore" = $1, "keystoreChangeIndex" = $2 WHERE "id" = $3;`,
      [user.keystore, user.keystoreChangeIndex, user.id],
    );
  }

  async createUserData(user: User, type: string, userDataInput: UserDataInput) {
    const newData = this.userDataRepository.create();

    newData.type = type;
    newData.meta = userDataInput.meta;
    newData.data = Buffer.from(userDataInput.data, "base64");
    newData.owner = user;
    newData.changeIndex = 1;

    await this.userDataRepository.save(newData);

    return newData;
  }

  async createUserSubmission(
    userSubmissionInput: UserSubmissionInput,
    user?: User,
  ) {
    const existingData = await this.getUserSubmissionBySubmission(
      userSubmissionInput.authorEmail,
      userSubmissionInput.event,
    );
    const newData = existingData || this.userSubmissionRepository.create();

    newData.authorEmail = userSubmissionInput.authorEmail;
    newData.authorName = userSubmissionInput.authorName;
    newData.socials = userSubmissionInput.socials;
    newData.mediums = userSubmissionInput.mediums;
    newData.event = userSubmissionInput.event;
    newData.owner = user;
    newData.dataName = userSubmissionInput.dataName;
    // newData.data = Buffer.from(userSubmissionInput.data, "base64");

    if (!newData.dataKey) newData.dataKey = nanoid();

    try {
      await this._s3
        .putObject({
          Bucket: "novelai-event-submissions",
          Key: newData.dataKey,
          Body: JSON.stringify({
            name: newData.dataName,
            data: userSubmissionInput.data,
          }),
          ContentType: "application/json; charset=utf-8",
        })
        .promise();
    } catch (error: any) {
      console.error(error);
      throw new HttpException(
        "Data upload error.",
        HttpStatus.INTERNAL_SERVER_ERROR,
      );
    }

    await this.userSubmissionRepository.save(newData);

    return newData;
  }

  async getUserSubmissionBySubmission(email: string, event: string) {
    const data = await this.userSubmissionRepository.findOneBy({
      authorEmail: email,
      event: event,
    });
    if (data?.data) (data.data as unknown) = data.data.toString("base64");
    return data;
  }
  async getUserSubmissionByUser(user: User, event: string) {
    const data = await this.userSubmissionRepository.findOneBy({
      owner: { id: user.id },
      event: event,
    });
    if (data?.data) (data.data as unknown) = data.data.toString("base64");
    return data;
  }

  async getUserSubmissionVotes(user: User, event: string) {
    const data = await this.userSubmissionVoteRepository.findBy({
      owner: { id: user.id },
      event,
    });

    return data;
  }

  async voteForSubmission(
    user: User,
    event: string,
    submissionId: string,
    choice: boolean,
  ) {
    const ONE_VOTE_ALLOWED = false;

    const vote = await this.userSubmissionVoteRepository.findOneBy({
      submissionId,
      event,
      owner: { id: user.id },
    });

    if (vote) {
      if (choice == false)
        await this.userSubmissionVoteRepository.delete(vote.id);
      else
        throw new HttpException(
          "You have already voted for that submission.",
          HttpStatus.CONFLICT,
        );

      return;
    }

    if (choice == false)
      throw new HttpException(
        "You haven't yet voted for that submission.",
        HttpStatus.CONFLICT,
      );

    if (
      ONE_VOTE_ALLOWED &&
      (await this.userSubmissionVoteRepository.findOneBy({
        owner: { id: user.id },
      }))
    )
      throw new HttpException(
        "You have already voted for a submission.",
        HttpStatus.CONFLICT,
      );

    const newVote = this.userSubmissionVoteRepository.create({
      submissionId,
      owner: user,
      event,
    });

    await this.userSubmissionVoteRepository.save(newVote);
  }

  objectEntityToObjectEntityDto(data: UserData): UserDataDto {
    return {
      id: data.id,
      type: data.type,
      meta: data.meta,
      data: data.data.toString("base64"),
      lastUpdatedAt: Math.floor(data.lastUpdatedAt.getTime() / 1000),
      changeIndex: data.changeIndex,
    };
  }

  async updateUserData(data: UserData, userDataInput: UserDataInput) {
    if (
      !!userDataInput.changeIndex &&
      data.changeIndex != userDataInput.changeIndex
    )
      throw new HttpException(
        `Incorrect change index (expected ${data.changeIndex}, got ${userDataInput.changeIndex}).`,
        HttpStatus.CONFLICT,
      );

    data.meta = userDataInput.meta;
    data.data = Buffer.from(userDataInput.data, "base64");
    data.changeIndex++;

    await this.userDataRepository.save(data);
  }

  async deleteUserData(data: UserData) {
    await this.userDataRepository.delete(data.id);
  }

  async getUserDatasOfType(user: User, type: string): Promise<UserData[]> {
    return await this.userDataRepository.find({
      where: {
        owner: { id: user.id },
        type: type,
      },
      order: {
        lastUpdatedAt: "DESC",
      },
      take: 1000,
    });
  }

  async getUserData(
    user: User,
    type: string,
    id: string,
  ): Promise<UserData | null> {
    const data = await this.userDataRepository.findOneBy({
      id,
    });

    if (data && (data.ownerId != user.id || data.type != type))
      throw new HttpException("Object not found.", HttpStatus.NOT_FOUND);

    return data;
  }

  async getUserGiftKeys(user: User) {
    return await this.subscriptionService.getUserGiftKeys(user);
  }

  async updateClientSettings(user: User, data: string) {
    user.clientSettings = data;

    await this.usersRepository.query(
      `UPDATE "user" SET "clientSettings" = $1 WHERE "id" = $2;`,
      [data, user.id],
    );
  }

  async deleteUser(user: User) {
    await this.usersRepository.delete({
      id: user.id,
    });
  }

  async startDeletion(
    requestUser: User,
    deletionStartDto: DeletionStartRequest,
  ): Promise<boolean> {
    const requestedEmail = deletionStartDto.email.trim();

    const emailValid = this.checkEmailHash(requestUser, deletionStartDto.email);
    if (!emailValid)
      throw new HttpException("Bad email address.", HttpStatus.BAD_REQUEST);

    const currentUnix = Math.floor(+new Date() / 1000);

    if (
      requestUser.deletionToken &&
      Math.floor(+requestUser.deletionSessionUntil / 1000) > currentUnix
    )
      throw new HttpException(
        "A deletion token was already sent. Please try again in a day.",
        HttpStatus.CONFLICT,
      );

    const token = nanoid(64);
    requestUser.deletionToken = hashAccessKey(token);
    requestUser.deletionSessionUntil = new Date(
      (currentUnix + 1 * 24 * 60 * 60) * 1000,
    );

    await this.usersRepository.save(requestUser);

    await this.mailgunService.createEmail(process.env.MAILGUN_DOMAIN, {
      from: `NovelAI <noreply@${process.env.MAILGUN_DOMAIN}>`,
      to: requestedEmail,
      subject: "NovelAI - Account Deletion",
      template: "account_deletion",
      "h:X-Mailgun-Variables": JSON.stringify({
        deletion_link: "https://novelai.net/deleteaccount?token=" + token,
        deletion_token: token,
      }),
    });

    return true;
  }

  async startRecovery(
    recoveryStartDto: RecoveryStartRequest,
  ): Promise<boolean> {
    const hashedEmail = hashEmailAddress(recoveryStartDto.email.toLowerCase());

    if (recoveryStartDto.email[0].toLowerCase() !== recoveryStartDto.email[0]) {
      // Email begins with an uppercase letter
      const hashedUppercaseEmail = hashEmailAddress(recoveryStartDto.email);
      const uppercaseAccount = await this.findUser({
        hashedEmail: hashedUppercaseEmail,
      });

      if (uppercaseAccount) {
        const lowercaseAccount = await this.findUser({
          hashedEmail,
        });

        let targetAccount: User | null = null;

        if (!lowercaseAccount) {
          targetAccount = uppercaseAccount;
        } else {
          if (
            uppercaseAccount.hasSubscription() &&
            lowercaseAccount.hasSubscription()
          )
            targetAccount = isTierABetterThanB(
              uppercaseAccount.subscriptionTier,
              lowercaseAccount.subscriptionTier,
            )
              ? uppercaseAccount
              : lowercaseAccount;
          else
            targetAccount = uppercaseAccount.hasSubscription()
              ? uppercaseAccount
              : lowercaseAccount;
        }

        if (lowercaseAccount && uppercaseAccount) {
          const leftoverAccount =
            targetAccount == uppercaseAccount
              ? lowercaseAccount
              : uppercaseAccount;

          // leftoverAccount.keystore = null;
          await this.usersRepository.query(
            `UPDATE "user" SET "keystore" = NULL WHERE "id" = $1;`,
            [leftoverAccount.id],
          );

          for (const type of ["stories", "storycontent"])
            await this.userDataRepository.delete({
              owner: { id: leftoverAccount.id },
              type,
            });

          leftoverAccount.hashedEmail =
            "emailConflict:" + leftoverAccount.hashedEmail;
          await this.usersRepository.save(leftoverAccount);
        }

        targetAccount.hashedEmail = hashedEmail;
        await this.usersRepository.save(targetAccount);
      }

      recoveryStartDto.email = recoveryStartDto.email.toLowerCase();
    }

    const user = await this.findUser({ hashedEmail });
    if (!user) return false;

    const currentUnix = Math.floor(+new Date() / 1000);

    if (
      user.recoverySessionToken &&
      Math.floor(+user.recoverySessionUntil / 1000) > currentUnix
    )
      return false;

    const token = nanoid(64);
    user.recoverySessionToken = hashAccessKey(token);
    user.recoverySessionUntil = new Date(
      (currentUnix + 1 * 24 * 60 * 60) * 1000,
    );

    await this.usersRepository.save(user);

    await this.mailgunService.createEmail(process.env.MAILGUN_DOMAIN, {
      from: `NovelAI <noreply@${process.env.MAILGUN_DOMAIN}>`,
      to: recoveryStartDto.email.trim(),
      subject: "NovelAI - Account Recovery",
      template: "account_recovery",
      "h:X-Mailgun-Variables": JSON.stringify({
        recovery_link: "https://novelai.net/recover?token=" + token,
        recovery_token: token,
      }),
    });

    return true;
  }

  async finishRecovery(recoveryFinishDto: RecoveryFinishRequest) {
    const user = await this.findUser({
      recoverySessionToken: hashAccessKey(recoveryFinishDto.recoveryToken),
    });
    if (!user)
      throw new HttpException("Token not found.", HttpStatus.UNAUTHORIZED);

    const oldSessionUntil = Math.floor(+user.recoverySessionUntil / 1000);

    user.recoverySessionToken = null;
    user.recoverySessionUntil = new Date(0);

    await this.usersRepository.save(user);

    const currentUnix = Math.floor(+new Date() / 1000);
    if (currentUnix >= oldSessionUntil)
      throw new HttpException("Token not found.", HttpStatus.UNAUTHORIZED);

    const newAccessKeyHash = hashAccessKey(recoveryFinishDto.newAccessKey);
    if (user.accessKey === newAccessKeyHash)
      throw new HttpException(
        "Cannot set access key to the same access key.",
        HttpStatus.CONFLICT,
      );

    if (await this.userByAccessKeyHash(newAccessKeyHash))
      throw new HttpException("Incorrect newAccessKey.", HttpStatus.CONFLICT);

    user.accessKey = newAccessKeyHash;
    user.authenticationNonce = nanoid();

    if (recoveryFinishDto.deleteContent) {
      // user.keystore = null;
      await this.usersRepository.query(
        `UPDATE "user" SET "keystore" = NULL WHERE "id" = $1;`,
        [user.id],
      );

      for (const type of ["stories", "storycontent"])
        await this.userDataRepository.delete({
          owner: { id: user.id },
          type,
        });
    }

    await this.usersRepository.save(user);

    return this.authService.generateTokenForUser(user);
  }

  async onUserFetchedActions(user: User) {
    if (!user) return;

    await this.taskPriorityService.synchronizeCurrentPriority(user);
    // await this.subscriptionService.fixSubscriptionDataIfRequired(user);
  }

  async findUser(conditions?: FindOptionsWhere<User>): Promise<User | null> {
    const user = await this.usersRepository.findOne({
      // relations: ["subscriptionData"],
      where: conditions,
      cache:
        conditions.id !== undefined
          ? {
              id: "find_user_" + conditions.id,
              milliseconds: 30 * 1000,
            }
          : false,
    });

    await this.onUserFetchedActions(user);

    return user;
  }

  async userById(id: string): Promise<User | null> {
    return this.findUser({ id });
  }

  async userByAccessKeyHash(accessKeyHash: string): Promise<User | null> {
    return this.findUser({ accessKey: accessKeyHash });
  }

  async getSubscriptionData(user: User): Promise<SubscriptionData | null> {
    return await this.subscriptionService.getSubscriptionData(user.id);
  }

  async loadSubscriptionData(user: User) {
    await this.subscriptionService.loadSubscriptionData(user);
  }

  async getKeystore(user: User): Promise<Buffer> {
    const data = await this.usersRepository
      .createQueryBuilder()
      .select('"User"."keystore"', "keystore")
      .where("id = :id", { id: user.id })
      .getRawOne();
    return data?.keystore;
  }

  async getClientSettings(user: User): Promise<string> {
    const data = await this.usersRepository
      .createQueryBuilder()
      .select('"User"."clientSettings"', "clientSettings")
      .where("id = :id", { id: user.id })
      .getRawOne();
    return data?.clientSettings;
  }
}
