import {
  AfterRemove,
  AfterUpdate,
  BeforeInsert,
  Column,
  DataSource,
  CreateDateColumn,
  Entity,
  JoinColumn,
  OneToOne,
  PrimaryColumn,
  // OneToMany,
} from "typeorm";
import {
  SubscriptionTiers,
  SubscriptionTierData,
} from "../shared/subscription-tiers";
import { nanoid } from "nanoid";
import { SubscriptionData } from "./subscription/subscription.entity";

let databaseConnection: DataSource | null = null;
export function setDatabaseConnection(connection: DataSource) {
  databaseConnection = connection;
}

export const SUBSCRIPTION_GRACE_PERIOD = 1 * 24 * 60 * 60; // 24 hours to renew subscription

export enum UserType {
  RETAIL,
  B2B,
  SERVICE,
}

@Entity("user")
export class User {
  @PrimaryColumn()
  id: string;

  @CreateDateColumn()
  createdAt: Date;

  @Column({ unique: true })
  accessKey: string;

  @Column({ default: null, unique: true })
  hashedEmail: string;

  @Column()
  authenticationNonce: string;

  @Column({ default: 0 })
  maxPriorityActions: number;

  @Column({ default: () => "to_timestamp(0)" })
  nextRefillAt: Date;

  @Column({ type: "float", default: 0 })
  taskPriority: number;

  @Column({ default: 0 })
  subscriptionTier: SubscriptionTiers;

  @Column({ default: () => "to_timestamp(0)" })
  subscriptionUntil: Date;

  /*@Column({
    type: "bytea",
    nullable: true,
    select: false,
    update: false,
  })*/
  keystore: Buffer;

  // @Column({ nullable: true, select: false, update: false })
  clientSettings: string;

  @BeforeInsert()
  private generateID() {
    this.id = nanoid();
  }

  @AfterUpdate()
  @AfterRemove()
  async flushCache() {
    await databaseConnection?.queryResultCache?.remove([
      "find_user_" + this.id,
    ]);
  }

  // @OneToMany(() => UserData, (data) => data.owner)
  // objects: UserData[];

  @OneToOne(() => SubscriptionData, (data) => data.owner, {
    onDelete: "SET NULL",
    cascade: true,
  })
  @JoinColumn()
  subscriptionData: SubscriptionData;

  @Column({ default: null })
  recoverySessionToken: string;

  @Column({ default: () => "to_timestamp(0)" })
  recoverySessionUntil: Date;

  @Column({ default: 0 })
  availableModuleTrainingSteps: number;

  @Column({ default: UserType.RETAIL })
  userType: UserType;

  @Column({ default: null })
  persistentAccessToken: string;

  @Column({ default: 0 })
  classificationCharactersLeft: number;

  @Column({ default: 0 })
  purchasedModuleTrainingSteps: number;

  @Column({ default: 1 })
  keystoreChangeIndex: number;

  @Column({ default: false })
  emailVerified: boolean;

  @Column({ default: null })
  emailVerificationToken: string;

  @Column({ default: () => "to_timestamp(0)" })
  emailVerificationLastSentAt: Date;

  @Column({ default: false })
  trialActivated: boolean;

  @Column({ default: 0, type: "integer" })
  trialActions: number;

  @Column({ default: null })
  deletionToken: string;

  @Column({ default: () => "to_timestamp(0)" })
  deletionSessionUntil: Date;

  // @OneToMany(() => SubscriptionData, (data) => data.referrer, {
  //   cascade: true,
  // })
  // purchasedGiftKeys: SubscriptionData[];

  // eslint-disable-next-line @typescript-eslint/no-unused-vars
  hasSubscription(_allowFree = false) {
    const expired =
      Math.floor(+this.subscriptionUntil / 1000) + SUBSCRIPTION_GRACE_PERIOD <
      Math.floor(+new Date() / 1000);

    /*if (
      allowFree &&
      (this.subscriptionTier == SubscriptionTiers.NONE || expired)
    )
      return true;*/

    return (
      this.subscriptionTier > SubscriptionTiers.NONE &&
      !expired &&
      this.userType == UserType.RETAIL
    );
  }

  getSubscriptionPerks() {
    return SubscriptionTierData[this.subscriptionTier];
  }
}
