import {
  Body,
  Controller,
  Delete,
  Get,
  Header,
  HttpCode,
  HttpException,
  HttpStatus,
  Param,
  Patch,
  Post,
  Put,
  Req,
} from "@nestjs/common";
import {
  ApiAcceptedResponse,
  ApiBadRequestResponse,
  ApiBearerAuth,
  ApiConflictResponse,
  ApiCreatedResponse,
  ApiExcludeEndpoint,
  ApiInternalServerErrorResponse,
  ApiNotFoundResponse,
  ApiOkResponse,
  ApiTags,
  ApiUnauthorizedResponse,
} from "@nestjs/swagger";
import { ApiError } from "../shared/dto/api-error.dto";
import { RequireAccessToken } from "../shared/guards/require-access-token.guard";
import { SuccessfulLoginResponse } from "./dto/successful-login.dto";
import { CreateUserRequest } from "./dto/create-user.dto";
import { GetKeystoreResponse } from "./dto/get-keystore-response.dto";
import { ObjectsResponse } from "./dto/objects-response.dto";
import { PriorityResponse } from "./dto/priority-response.dto";
import { UpdateKeystoreRequest } from "./dto/update-keystore.dto";
import { SUBSCRIPTION_GRACE_PERIOD, User } from "./user.entity";
import { UserService } from "./user.service";
import { UserDataInput } from "./dto/user-data-input.dto";
import { UserData } from "./dto/user-data.dto";
import { SubscriptionResponse } from "./dto/subscription-response.dto";
import { Throttle } from "@nestjs/throttler";
import { hashAccessKey } from "../shared/access-key-utils";
import { LoginRequest } from "./dto/login.dto";
import { ChangeAccessKeyRequest } from "./dto/change-access-key.dto";
import {
  RecoveryFinishRequest,
  RecoveryStartRequest,
} from "./dto/recovery.dto";
import { PaymentProcessors } from "./subscription/payment-processors";
import { UserSubmissionVoteInput } from "./dto/user-submission-vote.dto";
import {
  EmailVerificationRequest,
  EmailVerificationStartRequest,
} from "./dto/email-verification.dto";
import { AccountInformationResponse } from "./dto/account-information.dto";
import { UserAccountDataResponse } from "./dto/user-account-data.dto";
import {
  DeletionFinishRequest,
  DeletionStartRequest,
} from "./dto/deletion.dto";
import { GiftKeysResponse } from "./dto/user-giftkeys.dto";
import { UserSubmissionInput } from "./dto/user-submission-input.dto";

@ApiTags("/user/")
@Controller("user")
@ApiInternalServerErrorResponse({
  description: "An unknown error occured.",
  type: ApiError,
})
export class UserController {
  constructor(private readonly userService: UserService) {}

  @Post("register")
  @Throttle(2, 5)
  @Header("Cache-Control", "no-store")
  @ApiCreatedResponse({
    description: "The user has been successfully created.",
    type: SuccessfulLoginResponse,
  })
  @ApiBadRequestResponse({
    description: "A validation error occured.",
    type: ApiError,
  })
  @ApiConflictResponse({
    description: "A conflict error occured.",
    type: ApiError,
  })
  async createUser(@Body() createUserDto: CreateUserRequest) {
    if (!createUserDto)
      throw new HttpException("Incorrect body.", HttpStatus.BAD_REQUEST);

    if (!(await this.userService.checkCaptchaToken(createUserDto.recaptcha)))
      throw new HttpException(
        "Invalid reCAPTCHA token provided.",
        HttpStatus.BAD_REQUEST,
      );

    const accessToken = await this.userService.createUser(createUserDto);
    return {
      accessToken,
    };
  }

  @Post("login")
  @Throttle(12, 60)
  @Header("Cache-Control", "no-store")
  @ApiCreatedResponse({
    description: "Login successful.",
    type: SuccessfulLoginResponse,
  })
  @ApiBadRequestResponse({
    description: "A validation error occured.",
    type: ApiError,
  })
  @ApiUnauthorizedResponse({
    description: "Access Key is incorrect.",
    type: ApiError,
  })
  async loginUser(@Body() loginDto: LoginRequest) {
    if (!loginDto)
      throw new HttpException("Incorrect body.", HttpStatus.BAD_REQUEST);

    const accessToken = await this.userService.loginUser(loginDto.key);
    return {
      accessToken,
    };
  }

  @Post("change-access-key")
  @Throttle(4, 5)
  @Header("Cache-Control", "no-store")
  @ApiBearerAuth()
  @RequireAccessToken()
  @HttpCode(HttpStatus.OK)
  @ApiUnauthorizedResponse({
    description: "Access Token is incorrect.",
    type: ApiError,
  })
  @ApiOkResponse({
    description: "Access Key change successful.",
    type: SuccessfulLoginResponse,
  })
  @ApiBadRequestResponse({
    description: "A validation error occured.",
    type: ApiError,
  })
  @ApiUnauthorizedResponse({
    description: "Current Access Key is incorrect.",
    type: ApiError,
  })
  @ApiConflictResponse({
    description: "A conflict error occured.",
    type: ApiError,
  })
  async changeAccessKey(
    @Req() req,
    @Body() changeAccessKeyDto: ChangeAccessKeyRequest,
  ) {
    if (!changeAccessKeyDto)
      throw new HttpException("Incorrect body.", HttpStatus.BAD_REQUEST);

    const user: User = req.user;

    const accessToken = await this.userService.changeUserAccessKey(
      user,
      changeAccessKeyDto,
    );

    return {
      accessToken,
    };
  }

  @Post("resend-email-verification")
  // @Throttle(2, 86400)
  @Header("Cache-Control", "no-store")
  @HttpCode(HttpStatus.CREATED)
  // @ApiBearerAuth()
  // @RequireAccessToken()
  @ApiUnauthorizedResponse({
    description: "Access Token is incorrect.",
    type: ApiError,
  })
  @ApiCreatedResponse({
    description: "Email verification letter has been sent.",
  })
  @ApiBadRequestResponse({
    description: "A validation error occured.",
    type: ApiError,
  })
  @ApiConflictResponse({
    description: "A conflict error occured.",
    type: ApiError,
  })
  async requestEmailVerification(
    @Body() emailVerificationDto: EmailVerificationStartRequest,
  ) {
    if (!emailVerificationDto)
      throw new HttpException("Incorrect body.", HttpStatus.BAD_REQUEST);

    await this.userService.startEmailVerification(emailVerificationDto);
  }

  @Post("verify-email")
  @Throttle(2, 5)
  @Header("Cache-Control", "no-store")
  @HttpCode(HttpStatus.OK)
  @ApiOkResponse({
    description: "The email has been successfully verified.",
  })
  @ApiBadRequestResponse({
    description: "A validation error occured.",
    type: ApiError,
  })
  @ApiUnauthorizedResponse({
    description: "Verification token was not found.",
    type: ApiError,
  })
  async verifyEmail(@Body() emailVerificationDto: EmailVerificationRequest) {
    if (!emailVerificationDto)
      throw new HttpException("Incorrect body.", HttpStatus.BAD_REQUEST);

    await this.userService.verifyEmail(emailVerificationDto);
  }

  @Get("information")
  @Header("Cache-Control", "no-store")
  @ApiBearerAuth()
  @RequireAccessToken()
  @ApiUnauthorizedResponse({
    description: "Access Token is incorrect.",
    type: ApiError,
  })
  @ApiOkResponse({
    description: "Account information",
    type: AccountInformationResponse,
  })
  async getAccountInformation(@Req() req) {
    const user: User = req.user;

    return {
      emailVerified: user.emailVerified,
      emailVerificationLetterSent: user.emailVerificationToken !== null,

      trialActivated: user.trialActivated,
      trialActionsLeft: user.trialActions,

      accountCreatedAt: Math.floor(+user.createdAt / 1000),
    };
  }

  @Post("deletion/request")
  @ApiBearerAuth()
  @RequireAccessToken()
  @Throttle(4, 5)
  @Header("Cache-Control", "no-store")
  @HttpCode(HttpStatus.ACCEPTED)
  @ApiAcceptedResponse({
    description:
      "Your request has been processed. If the email address provided is registered, it will receive a letter with further instructions.",
  })
  @ApiBadRequestResponse({
    description: "A validation error occured.",
    type: ApiError,
  })
  @ApiConflictResponse({
    description: "Cannot delete accounts with active recurring subscriptions.",
    type: ApiError,
  })
  async requestAccountDeletion(
    @Body() deletionStartDto: DeletionStartRequest,
    @Req() req,
  ) {
    if (!deletionStartDto)
      throw new HttpException("Incorrect body.", HttpStatus.BAD_REQUEST);

    const user: User = req.user;
    if (!user) {
      throw new HttpException(
        "Cannot delete unauthenticated account.",
        HttpStatus.CONFLICT,
      );
    }

    await this.userService.loadSubscriptionData(user);
    if (
      user.subscriptionData &&
      user.subscriptionData.paymentProcessor == PaymentProcessors.PADDLE
    )
      throw new HttpException(
        "Cannot delete account with an active recurring subscription.",
        HttpStatus.CONFLICT,
      );

    return await this.userService.startDeletion(user, deletionStartDto);
  }

  @Post("deletion/delete")
  @Throttle(2, 5)
  @HttpCode(HttpStatus.OK)
  @ApiOkResponse({
    description: "Account deletion successful.",
  })
  @ApiUnauthorizedResponse({
    description: "Deletion token was not found.",
    type: ApiError,
  })
  @ApiConflictResponse({
    description: "Cannot delete accounts with active recurring subscriptions.",
    type: ApiError,
  })
  async deleteAccount(@Body() deletionStartDto: DeletionFinishRequest) {
    if (!deletionStartDto.deletionToken)
      throw new HttpException(
        "Deletion token required",
        HttpStatus.BAD_REQUEST,
      );

    const user = await this.userService.findUser({
      deletionToken: hashAccessKey(deletionStartDto.deletionToken),
    });

    if (!user)
      throw new HttpException(
        "Deletion token was not found",
        HttpStatus.BAD_REQUEST,
      );

    await this.userService.loadSubscriptionData(user);
    if (
      user.subscriptionData &&
      user.subscriptionData.paymentProcessor == PaymentProcessors.PADDLE
    )
      throw new HttpException(
        "Cannot delete account with an active recurring subscription.",
        HttpStatus.CONFLICT,
      );

    if (+new Date() > +user.deletionSessionUntil)
      throw new HttpException(
        "Deletion token is invalid",
        HttpStatus.UNAUTHORIZED,
      );

    return await this.userService.deleteUser(user);
  }

  @Post("recovery/request")
  @Throttle(4, 5)
  @Header("Cache-Control", "no-store")
  @HttpCode(HttpStatus.ACCEPTED)
  @ApiAcceptedResponse({
    description:
      "Your request has been processed. If the email address provided is registered, it will receive a letter with further instructions.",
  })
  @ApiBadRequestResponse({
    description: "A validation error occured.",
    type: ApiError,
  })
  async requestAccountRecovery(@Body() recoveryStartDto: RecoveryStartRequest) {
    if (!recoveryStartDto)
      throw new HttpException("Incorrect body.", HttpStatus.BAD_REQUEST);

    this.userService.startRecovery(recoveryStartDto);
  }

  @Post("recovery/recover")
  @Throttle(2, 5)
  @Header("Cache-Control", "no-store")
  @ApiCreatedResponse({
    description: "Account recovery successful.",
    type: SuccessfulLoginResponse,
  })
  @ApiBadRequestResponse({
    description: "A validation error occured.",
    type: ApiError,
  })
  @ApiUnauthorizedResponse({
    description: "Recovery token was not found.",
    type: ApiError,
  })
  @ApiConflictResponse({
    description: "A conflict error occured.",
    type: ApiError,
  })
  async recoverAccount(@Body() recoveryFinishDto: RecoveryFinishRequest) {
    if (!recoveryFinishDto)
      throw new HttpException("Incorrect body.", HttpStatus.BAD_REQUEST);

    const accessToken = await this.userService.finishRecovery(
      recoveryFinishDto,
    );

    return {
      accessToken,
    };
  }

  @Post("delete")
  @Throttle(2, 5)
  @ApiBearerAuth()
  @RequireAccessToken()
  @HttpCode(HttpStatus.OK)
  @ApiUnauthorizedResponse({
    description: "Access Token is incorrect.",
    type: ApiError,
  })
  @ApiOkResponse({
    description: "Account deletion successful.",
  })
  @ApiConflictResponse({
    description: "Cannot delete accounts with active recurring subscriptions.",
    type: ApiError,
  })
  async deleteAccountUnchecked(@Req() req) {
    const user: User = req.user;

    await this.userService.loadSubscriptionData(user);
    if (
      user.subscriptionData &&
      user.subscriptionData.paymentProcessor == PaymentProcessors.PADDLE
    )
      throw new HttpException(
        "Cannot delete account with an active recurring subscription.",
        HttpStatus.CONFLICT,
      );

    return await this.userService.deleteUser(user);
  }

  @Get("data")
  @Header("Cache-Control", "no-store")
  @ApiBearerAuth()
  @RequireAccessToken()
  @ApiUnauthorizedResponse({
    description: "Access Token is incorrect.",
    type: ApiError,
  })
  @ApiOkResponse({
    description: "Various user information.",
    type: UserAccountDataResponse,
  })
  async getUserData(@Req() req) {
    return {
      priority: await this.getCurrentPriority(req),
      subscription: await this.getSubscription(req),
      keystore: await this.getKeystore(req),
      settings: await this.getClientSettings(req),
      information: await this.getAccountInformation(req),
    };
  }

  @Get("priority")
  @Header("Cache-Control", "no-store")
  @ApiBearerAuth()
  @RequireAccessToken()
  @ApiUnauthorizedResponse({
    description: "Access Token is incorrect.",
    type: ApiError,
  })
  @ApiOkResponse({
    description:
      "Amount of max priority actions left, next max priority action refill (UNIX timestamp) and current task priority.",
    type: PriorityResponse,
  })
  async getCurrentPriority(@Req() req) {
    const user: User = req.user;

    return {
      maxPriorityActions: user.maxPriorityActions,
      nextRefillAt: Math.floor(+user.nextRefillAt / 1000),
      taskPriority: user.taskPriority,
    };
  }

  @Get("giftkeys")
  @Header("Cache-Control", "no-store")
  @ApiBearerAuth()
  @RequireAccessToken()
  @ApiUnauthorizedResponse({
    description: "Access Token is incorrect.",
    type: ApiError,
  })
  @ApiOkResponse({
    description: "Purchased Gift Keys.",
    type: GiftKeysResponse,
  })
  async getGiftKeys(@Req() req) {
    const user: User = req.user;

    return {
      giftKeys: await this.userService.getUserGiftKeys(user),
    };
  }

  @Get("subscription")
  @Header("Cache-Control", "no-store")
  @ApiBearerAuth()
  @RequireAccessToken()
  @ApiUnauthorizedResponse({
    description: "Access Token is incorrect.",
    type: ApiError,
  })
  @ApiOkResponse({
    description: "Current subscription, date of expiry and perks.",
    type: SubscriptionResponse,
  })
  async getSubscription(@Req() req) {
    const user: User = req.user;

    await this.userService.loadSubscriptionData(user);

    return {
      tier: user.subscriptionTier,
      active: user.hasSubscription(true),
      expiresAt:
        Math.floor(+user.subscriptionUntil / 1000) + SUBSCRIPTION_GRACE_PERIOD,
      perks: user.getSubscriptionPerks(),
      paymentProcessorData: user.subscriptionData?.data || null,
      trainingStepsLeft: {
        fixedTrainingStepsLeft: user.availableModuleTrainingSteps,
        purchasedTrainingSteps: user.purchasedModuleTrainingSteps,
      },
    };
  }

  @Get("keystore")
  @Header("Cache-Control", "no-store")
  @ApiBearerAuth()
  @RequireAccessToken()
  @ApiUnauthorizedResponse({
    description: "Access Token is incorrect.",
    type: ApiError,
  })
  @ApiOkResponse({
    description: "Keystore buffer in Base64 format.",
    type: GetKeystoreResponse,
  })
  async getKeystore(@Req() req) {
    const user: User = req.user;

    const keystore = await this.userService.getKeystore(user);
    return {
      keystore: keystore?.toString("base64"),
      changeIndex: user.keystoreChangeIndex,
    };
  }

  @Put("keystore")
  @Header("Cache-Control", "no-store")
  @ApiBearerAuth()
  @RequireAccessToken()
  @ApiUnauthorizedResponse({
    description: "Access Token is incorrect.",
    type: ApiError,
  })
  @ApiBadRequestResponse({
    description: "A validation error occured.",
    type: ApiError,
  })
  @ApiOkResponse({
    description: "Edit is successful.",
  })
  async updateKeystore(
    @Req() req,
    @Body() updateKeystoreDto: UpdateKeystoreRequest,
  ) {
    const user: User = req.user;

    if (!updateKeystoreDto)
      throw new HttpException("Incorrect body.", HttpStatus.BAD_REQUEST);

    await this.userService.updateKeystore(user, updateKeystoreDto);
  }

  @Get("objects/:type")
  @Header("Cache-Control", "no-store")
  @ApiBearerAuth()
  @RequireAccessToken()
  @ApiUnauthorizedResponse({
    description: "Access Token is incorrect.",
    type: ApiError,
  })
  @ApiOkResponse({
    description: "List of serverside-stored objects of that type.",
    type: ObjectsResponse,
  })
  async getObjects(@Req() req, @Param("type") type: string) {
    /*const BLACKLISTED_OBJECT_TYPES = ["storycontent"];
    if (BLACKLISTED_OBJECT_TYPES.indexOf(type) !== -1)
      throw new HttpException(
        "Can't list objects of this type.",
        HttpStatus.BAD_REQUEST,
      );*/

    const user: User = req.user;
    return {
      objects: (await this.userService.getUserDatasOfType(user, type))
        //.sort((a, b) => a.lastUpdatedAt.getTime() - b.lastUpdatedAt.getTime())
        .map((object) => {
          return this.userService.objectEntityToObjectEntityDto(object);
        }),
    };
  }

  @Get("objects/:type/:id")
  @Header("Cache-Control", "no-store")
  @ApiBearerAuth()
  @RequireAccessToken()
  @ApiUnauthorizedResponse({
    description: "Access Token is incorrect.",
    type: ApiError,
  })
  @ApiOkResponse({
    description: "Found object.",
    type: UserData,
  })
  async getObject(
    @Req() req,
    @Param("type") type: string,
    @Param("id") id: string,
  ) {
    const user: User = req.user;
    const object = await this.userService.getUserData(user, type, id);
    if (!object)
      throw new HttpException("Object not found.", HttpStatus.NOT_FOUND);

    return this.userService.objectEntityToObjectEntityDto(object);
  }

  @Put("objects/:type")
  @Header("Cache-Control", "no-store")
  @ApiBearerAuth()
  @RequireAccessToken()
  @ApiUnauthorizedResponse({
    description: "Access Token is incorrect.",
    type: ApiError,
  })
  @ApiBadRequestResponse({
    description: "A validation error occured.",
    type: ApiError,
  })
  @ApiNotFoundResponse({
    description: "Specified object was not found.",
    type: ApiError,
  })
  @ApiOkResponse({
    description: "Object created successfully.",
    type: Object,
  })
  async createObject(
    @Req() req,
    @Param("type") type: string,
    @Body() userDataInput: UserDataInput,
  ) {
    const user: User = req.user;
    const object = await this.userService.createUserData(
      user,
      type,
      userDataInput,
    );
    return this.userService.objectEntityToObjectEntityDto(object);
  }

  @Patch("objects/:type/:id")
  @Header("Cache-Control", "no-store")
  @ApiBearerAuth()
  @RequireAccessToken()
  @ApiUnauthorizedResponse({
    description: "Access Token is incorrect.",
    type: ApiError,
  })
  @ApiBadRequestResponse({
    description: "A validation error occured.",
    type: ApiError,
  })
  @ApiNotFoundResponse({
    description: "Specified object was not found.",
    type: ApiError,
  })
  @ApiConflictResponse({
    description: "A conflict occured while updating this object.",
    type: ApiError,
  })
  @ApiOkResponse({
    description: "Object edited successfully.",
    type: UserData,
  })
  async editObject(
    @Req() req,
    @Param("id") id: string,
    @Param("type") type: string,
    @Body() userDataInput: UserDataInput,
  ) {
    const user: User = req.user;

    if (!userDataInput)
      throw new HttpException("Incorrect body.", HttpStatus.BAD_REQUEST);

    const object = await this.userService.getUserData(user, type, id);
    if (!object)
      throw new HttpException("Object not found.", HttpStatus.NOT_FOUND);

    await this.userService.updateUserData(object, userDataInput);

    return this.userService.objectEntityToObjectEntityDto(object);
  }

  @Delete("objects/:type/:id")
  @Header("Cache-Control", "no-store")
  @ApiBearerAuth()
  @RequireAccessToken()
  @ApiUnauthorizedResponse({
    description: "Access Token is incorrect.",
    type: ApiError,
  })
  @ApiNotFoundResponse({
    description: "Specified object was not found.",
    type: ApiError,
  })
  @ApiOkResponse({
    description: "Object deleted successfully.",
    type: UserData,
  })
  async deleteObject(
    @Req() req,
    @Param("id") id: string,
    @Param("type") type: string,
  ) {
    const user: User = req.user;

    const object = await this.userService.getUserData(user, type, id);
    if (!object)
      throw new HttpException("Object not found.", HttpStatus.NOT_FOUND);

    await this.userService.deleteUserData(object);

    return this.userService.objectEntityToObjectEntityDto(object);
  }

  @Get("clientsettings")
  @Header("Cache-Control", "no-store")
  @ApiBearerAuth()
  @RequireAccessToken()
  @ApiUnauthorizedResponse({
    description: "Access Token is incorrect.",
    type: ApiError,
  })
  @ApiOkResponse({
    description: "Client settings in an arbitrary format.",
  })
  async getClientSettings(@Req() req) {
    const user: User = req.user;

    return await this.userService.getClientSettings(user);
  }

  @Put("clientsettings")
  @Header("Cache-Control", "no-store")
  @ApiBearerAuth()
  @RequireAccessToken()
  @ApiUnauthorizedResponse({
    description: "Access Token is incorrect.",
    type: ApiError,
  })
  @ApiBadRequestResponse({
    description: "A validation error occured.",
    type: ApiError,
  })
  @ApiOkResponse({
    description: "Edit is successful.",
  })
  async updateClientSettings(@Req() req, @Body() data: string) {
    const user: User = req.user;

    if (!data)
      throw new HttpException("Incorrect body.", HttpStatus.BAD_REQUEST);

    await this.userService.updateClientSettings(user, data);
  }

  @Post("submission")
  @Throttle(4, 5)
  @Header("Cache-Control", "no-store")
  @ApiBearerAuth()
  @RequireAccessToken()
  @HttpCode(HttpStatus.OK)
  @ApiUnauthorizedResponse({
    description: "Access Token is incorrect.",
    type: ApiError,
  })
  @ApiBadRequestResponse({
    description: "A validation error occured.",
    type: ApiError,
  })
  @ApiOkResponse({
    description: "Submission is successful.",
  })
  async postUserSubmission(
    @Req() req,
    @Body() userSubmissionInput: UserSubmissionInput,
  ) {
    const user: User = req.user;
    if (!userSubmissionInput)
      throw new HttpException("Incorrect body.", HttpStatus.BAD_REQUEST);
    return await this.userService.createUserSubmission(
      userSubmissionInput,
      user,
    );
  }

  @Get("submission/:event")
  @Header("Cache-Control", "no-store")
  @ApiBearerAuth()
  @RequireAccessToken()
  @ApiUnauthorizedResponse({
    description: "Access Token is incorrect.",
    type: ApiError,
  })
  @ApiOkResponse({
    description: "User submission",
  })
  async getUserSubmission(@Req() req, @Param("event") event: string) {
    const user: User = req.user;
    return await this.userService.getUserSubmissionByUser(user, event);
  }

  @Get("vote-submission/:event")
  @Header("Cache-Control", "no-store")
  @ApiBearerAuth()
  @RequireAccessToken()
  @ApiUnauthorizedResponse({
    description: "Access Token is incorrect.",
    type: ApiError,
  })
  @ApiOkResponse({
    description: "User submission votes",
  })
  async getUserSubmissionVotes(@Req() req, @Param("event") event: string) {
    const user: User = req.user;

    return await this.userService.getUserSubmissionVotes(user, event);
  }

  @Post("vote-submission/:event")
  @Header("Cache-Control", "no-store")
  @ApiBearerAuth()
  @RequireAccessToken()
  @ApiUnauthorizedResponse({
    description: "Access Token is incorrect.",
    type: ApiError,
  })
  @ApiConflictResponse({
    description: "A conflict error occured.",
    type: ApiError,
  })
  @ApiOkResponse({
    description: "Vote is successful.",
  })
  async voteSubmission(
    @Req() req,
    @Param("event") event: string,
    @Body() userSubmissionVoteInput: UserSubmissionVoteInput,
  ) {
    const user: User = req.user;

    if (!userSubmissionVoteInput)
      throw new HttpException("Incorrect body.", HttpStatus.BAD_REQUEST);

    if (!user.hasSubscription())
      throw new HttpException(
        "An active subscription is required to vote.",
        HttpStatus.CONFLICT,
      );

    await this.userService.voteForSubmission(
      user,
      event,
      userSubmissionVoteInput.id,
      true,
    );
  }

  @Delete("vote-submission/:event")
  @Header("Cache-Control", "no-store")
  @ApiBearerAuth()
  @RequireAccessToken()
  @ApiUnauthorizedResponse({
    description: "Access Token is incorrect.",
    type: ApiError,
  })
  @ApiConflictResponse({
    description: "A conflict error occured.",
    type: ApiError,
  })
  @ApiOkResponse({
    description: "Vote is successful.",
  })
  async retractSubmissionVote(
    @Req() req,
    @Param("event") event: string,
    @Body() userSubmissionVoteInput: UserSubmissionVoteInput,
  ) {
    const user: User = req.user;

    if (!userSubmissionVoteInput)
      throw new HttpException("Incorrect body.", HttpStatus.BAD_REQUEST);

    if (!user.hasSubscription())
      throw new HttpException(
        "An active subscription is required to vote.",
        HttpStatus.CONFLICT,
      );

    await this.userService.voteForSubmission(
      user,
      event,
      userSubmissionVoteInput.id,
      false,
    );
  }

  @Post("admin/terminate")
  @Header("Cache-Control", "no-store")
  @ApiExcludeEndpoint(true)
  async terminateKey(@Req() req, @Body() data: any) {
    if (!process.env.ADMIN_PASSWORD)
      throw new HttpException("Endpoint disabled.", HttpStatus.BAD_REQUEST);

    if (req.headers["x-admin-token"] !== process.env.ADMIN_PASSWORD)
      throw new HttpException("Invalid admin token.", HttpStatus.UNAUTHORIZED);

    const key = data["key"];
    if (!key)
      throw new HttpException("Key not provided.", HttpStatus.BAD_REQUEST);

    const accessKeyHash = hashAccessKey(key);
    const user = await this.userService.userByAccessKeyHash(accessKeyHash);
    if (!user) throw new HttpException("User not found.", HttpStatus.NOT_FOUND);

    await this.userService.deleteUser(user);
    return {
      success: true,
    };
  }

  @Post("admin/flush-cache")
  @Header("Cache-Control", "no-store")
  @ApiExcludeEndpoint(true)
  async flushCache(@Req() req, @Body() data: any) {
    if (!process.env.ADMIN_PASSWORD)
      throw new HttpException("Endpoint disabled.", HttpStatus.BAD_REQUEST);

    if (req.headers["x-admin-token"] !== process.env.ADMIN_PASSWORD)
      throw new HttpException("Invalid admin token.", HttpStatus.UNAUTHORIZED);

    const id = data["id"];
    if (!id)
      throw new HttpException("User ID not provided.", HttpStatus.BAD_REQUEST);

    const user = await this.userService.userById(id);
    if (!user) throw new HttpException("User not found.", HttpStatus.NOT_FOUND);

    await user.flushCache();
    return {
      success: true,
    };
  }
}
