import { Injectable } from "@nestjs/common";
import { InjectRepository } from "@nestjs/typeorm";
import { Repository } from "typeorm";
import { User, UserType } from "./user.entity";

const PRIORITY_REFILL_INTERVAL = 7 * 24 * 60 * 60; // 1 week in seconds
const LEFTOVER_MPA_DIVIDER = 5;
const ACTION_PRIORITY_COST = 0.01;

@Injectable()
export class TaskPriorityService {
  constructor(
    @InjectRepository(User)
    private usersRepository: Repository<User>,
  ) {}

  async synchronizeCurrentPriority(user: User) {
    if (!user.hasSubscription(true)) return;

    const currentUnix = Math.floor(+new Date() / 1000);
    const nextRefillUnix = Math.floor(+user.nextRefillAt / 1000);
    if (nextRefillUnix > currentUnix) return;

    const subscriptionPerks = user.getSubscriptionPerks();

    user.maxPriorityActions =
      subscriptionPerks.maxPriorityActions +
      Math.round(user.maxPriorityActions / LEFTOVER_MPA_DIVIDER);
    user.taskPriority = subscriptionPerks.startPriority;

    user.nextRefillAt = new Date(
      (currentUnix + PRIORITY_REFILL_INTERVAL) * 1000,
    );

    await this.usersRepository.save(user);
  }

  async resetPriority(user: User): Promise<boolean> {
    if (!user.hasSubscription(true)) {
      user.taskPriority = 0;
      user.maxPriorityActions = 0;
    } else {
      const subscriptionPerks = user.getSubscriptionPerks();
      user.maxPriorityActions = subscriptionPerks.maxPriorityActions;
      user.taskPriority = subscriptionPerks.startPriority;

      const currentUnix = Math.floor(+new Date() / 1000);
      user.nextRefillAt = new Date(
        (currentUnix + PRIORITY_REFILL_INTERVAL) * 1000,
      );
    }

    await this.usersRepository.save(user);

    return true;
  }

  async tryConsumeMPA(
    user: User,
    actionsToConsume: number,
  ): Promise<{ success: boolean; priority: number }> {
    if (
      user.getSubscriptionPerks().unlimitedMaxPriority ||
      user.taskPriority == 0
    )
      return {
        success: true,
        priority: user.taskPriority == 0 ? user.taskPriority : 10,
      };

    if (user.maxPriorityActions > 0) {
      const result = await this.usersRepository
        .createQueryBuilder()
        .update(User)
        .set({ maxPriorityActions: user.maxPriorityActions - actionsToConsume })
        .where("maxPriorityActions >= :amount AND id = :id", {
          amount: actionsToConsume,
          id: user.id,
        })
        .execute();

      if (result.affected <= 0) return { success: false, priority: 0 };

      user.maxPriorityActions -= actionsToConsume;
      return { success: true, priority: 10 };
    } else {
      const totalPriorityConsume = Math.min(
        ACTION_PRIORITY_COST * actionsToConsume,
        user.taskPriority,
      );

      const result = await this.usersRepository
        .createQueryBuilder()
        .update(User)
        .set({
          taskPriority: user.taskPriority - totalPriorityConsume,
        })
        .where("taskPriority >= :amount AND id = :id", {
          amount: totalPriorityConsume,
          id: user.id,
        })
        .execute();

      if (result.affected <= 0) return { success: false, priority: 0 };

      user.taskPriority -= totalPriorityConsume;
      return { success: true, priority: user.taskPriority };
    }
  }

  async tryConsumeCharacters(
    user: User,
    charactersToConsume: number,
  ): Promise<boolean> {
    if (user.userType == UserType.SERVICE) return true; // Service accounts have infinite amount of characters

    const result = await this.usersRepository
      .createQueryBuilder()
      .update(User)
      .set({
        classificationCharactersLeft:
          user.classificationCharactersLeft - charactersToConsume,
      })
      .where("classificationCharactersLeft >= :amount AND id = :id", {
        amount: charactersToConsume,
        id: user.id,
      })
      .execute();

    if (result.affected <= 0) return false;

    user.classificationCharactersLeft -= charactersToConsume;
    return true;
  }
}
