import { HttpException, HttpStatus, Injectable } from "@nestjs/common";
import { InjectRepository } from "@nestjs/typeorm";
import { nanoid } from "nanoid";
import {
  isTierABetterThanB,
  SubscriptionTierData,
  SubscriptionTiers,
} from "../../shared/subscription-tiers";
import { Repository } from "typeorm";
import { AuthService } from "../auth.service";
import { TaskPriorityService } from "../task-priority.service";
import { User, UserType } from "../user.entity";
import { BindSubscriptionRequest } from "./dto/bind-subscription.dto";
import {
  isValidPaddleWebhookData,
  PaddleSubscriptionData,
  PaddleSubscriptionPlanIdToSubscriptionTier,
  getTierToPaddleSPIMap,
  paddleUpdateSubscription,
  paddleGetSubscriptionData,
  getTierToPaddlePIMap,
  getStepsToPaddlePIMap,
} from "./paddle.payment-processor";
import { PaymentProcessors } from "./payment-processors";
import { SubscriptionData } from "./subscription.entity";
import { Mutex } from "redis-semaphore";
import Redis from "ioredis";

function sleep(ms) {
  return new Promise((resolve) => setTimeout(resolve, ms));
}

// const TRIAL_LENGTH_SECONDS = 7 * 24 * 60 * 60;
const TEMPORARY_SUBSCRIPTION_LENGTH = 1 * 24 * 60 * 60;

export const TRIAL_ACTIONS_AMOUNT = 50;

@Injectable()
export class SubscriptionService {
  private _webhook_locks: Map<string, Mutex>;
  private _redis_client: Redis | null;

  constructor(
    @InjectRepository(User)
    private usersRepository: Repository<User>,

    @InjectRepository(SubscriptionData)
    private subscriptionDataRepository: Repository<SubscriptionData>,

    private taskPriorityService: TaskPriorityService,
    private authService: AuthService,
  ) {
    this._webhook_locks = new Map();

    if (process.env.REDIS_HOST !== undefined)
      this._redis_client = new Redis(6379, process.env.REDIS_HOST);
  }

  async getSubscriptionData(id: string): Promise<SubscriptionData | null> {
    return this.subscriptionDataRepository.findOne({
      where: {
        owner: { id },
      },
    });
  }

  async loadSubscriptionData(user: User) {
    if (!user.id) return;
    user.subscriptionData = await this.getSubscriptionData(user.id);
  }

  async initializeUserSubscription(
    user: User,
    tier: SubscriptionTiers,
    untilTimestamp: number,
    data: SubscriptionData,
  ) {
    const SECONDS_IN_ONE_MONTH = 1 * 30 * 24 * 60 * 60;

    user.subscriptionTier = tier;
    user.subscriptionUntil = new Date(untilTimestamp * 1000);

    const currentUnix = Math.floor(+new Date() / 1000);
    const amountOfMonths = Math.round(
      (untilTimestamp - currentUnix) / SECONDS_IN_ONE_MONTH,
    );

    const stockTrainingSteps = data
      ? SubscriptionTierData[tier].moduleTrainingSteps *
        (data.paymentProcessor == PaymentProcessors.GIFT_KEY
          ? amountOfMonths
          : 1)
      : 0;
    if (user.availableModuleTrainingSteps < stockTrainingSteps)
      user.availableModuleTrainingSteps = stockTrainingSteps;

    await this.taskPriorityService.resetPriority(user);

    // await sleep(500); // Another bad measure to possibly prevent race conditions.

    if (data) await this.subscriptionDataRepository.save(data);

    user.subscriptionData = data;

    await this.usersRepository.save(user);
  }

  async ensureOneSubscriptionDataInstance(data: SubscriptionData) {
    if (!data.id || !data.owner) return;

    await this.subscriptionDataRepository
      .createQueryBuilder()
      .delete()
      .from(SubscriptionData)
      .where("ownerId = :owner_id AND id != :subscription_id", {
        owner_id: data.owner.id,
        subscription_id: data.id,
      })
      .execute();

    await this.subscriptionDataRepository.save(data);

    if (
      data.owner.subscriptionData &&
      data.owner.subscriptionData.id != data.id
    ) {
      // User had a now-removed subscription data, replace it with the new one
      const user = data.owner;

      user.subscriptionData = data;

      await this.usersRepository.save(user);
    }
  }

  async getUserGiftKeys(referrer: User) {
    const giftKeys = await this.subscriptionDataRepository.findBy({
      referrer: { id: referrer.id },
    });
    return giftKeys;
  }

  async onGiftkeyPaddleWebhook(body: Record<any, any>) {
    if (!isValidPaddleWebhookData(body))
      throw new HttpException("Invalid request.", HttpStatus.BAD_REQUEST);
    if (!body.quantity)
      throw new HttpException("Invalid quantity.", HttpStatus.BAD_REQUEST);

    const tiers = getTierToPaddlePIMap();
    let tier = SubscriptionTiers.NONE;
    switch (Number.parseInt(body.p_product_id)) {
      case tiers[SubscriptionTiers.TABLET]:
        tier = SubscriptionTiers.TABLET;
        break;
      case tiers[SubscriptionTiers.SCROLL]:
        tier = SubscriptionTiers.SCROLL;
        break;
      case tiers[SubscriptionTiers.OPUS]:
        tier = SubscriptionTiers.OPUS;
        break;
    }
    if (tier === SubscriptionTiers.NONE) {
      throw new HttpException("Invalid tier.", HttpStatus.BAD_REQUEST);
    }

    const payload = {
      referrer: undefined as User | undefined,
    };
    try {
      const passthrough = JSON.parse(body.passthrough);
      if (!passthrough.auth_token) throw new Error("No referrer auth_token");
      const authToken = passthrough.auth_token;

      const userData = await this.authService.userIdByJWT(authToken, true);
      if (!userData) throw new Error("Invalid referrer auth_token");

      const user = await this.usersRepository.findOneBy({ id: userData.id });
      payload.referrer = user;
    } catch (e) {
      console.warn(
        "invalid gift key purchase passthrough:",
        body.passthrough,
        e,
      );
    }

    const keys = [];
    for (let i = 0; i < body.quantity; i++) {
      keys.push(
        await this.generateGiftKey(tier, 2592000, "", payload.referrer),
      ); // 30 days
    }

    return keys.join("\n");
  }

  async onStepsPaddleWebhook(body: Record<any, any>) {
    if (!isValidPaddleWebhookData(body))
      throw new HttpException("Invalid request.", HttpStatus.BAD_REQUEST);
    if (!body.quantity)
      throw new HttpException("Invalid quantity.", HttpStatus.BAD_REQUEST);

    const productId = Number.parseInt(body.p_product_id);
    const steps = getStepsToPaddlePIMap();

    const amount = steps[productId];
    if (amount === undefined)
      throw new HttpException("Invalid product ID.", HttpStatus.BAD_REQUEST);

    const payload = {
      referrer: undefined as User | undefined,
    };
    try {
      const passthrough = JSON.parse(body.passthrough);
      if (!passthrough.auth_token) throw new Error("No referrer auth_token");
      const authToken = passthrough.auth_token;

      const userData = await this.authService.userIdByJWT(authToken, true);
      if (!userData) throw new Error("Invalid referrer auth_token");

      const user = await this.usersRepository.findOneBy({ id: userData.id });
      payload.referrer = user;
    } catch (e) {
      console.warn(
        "invalid gift key purchase passthrough:",
        body.passthrough,
        e,
      );
    }

    if (!payload.referrer)
      throw new HttpException("Invalid passthrough.", HttpStatus.BAD_REQUEST);

    payload.referrer.purchasedModuleTrainingSteps += amount;
    await this.usersRepository.save(payload.referrer);

    return `${amount} Anlas have been deposited into your account.`;
  }

  async onPaddleWebhook(body: Record<any, any>) {
    if (!isValidPaddleWebhookData(body)) return;
    if (!body.alert_name) return;

    // only for subscriptions
    if (
      body.alert_name == "high_risk_transaction_created" &&
      getTierToPaddleSPIMap()[+body.product_id]
    ) {
      const currentUnix = Math.floor(+new Date() / 1000);
      const temporarySubscriptionUntil =
        currentUnix + TEMPORARY_SUBSCRIPTION_LENGTH;

      const newTemporarySubscription = this.subscriptionDataRepository.create({
        id: body.case_id,
        paymentProcessor: PaymentProcessors.PADDLE,
        data: {
          t: PaymentProcessors.PADDLE,
          c: "",
          u: "",
          p: +body.product_id,
          s: "high_risk",
          o: body.checkout_id,
          r: body.order_id,
          n: temporarySubscriptionUntil,
        },
      } as SubscriptionData);

      await this.bindSubscriptionByPassthrough(
        newTemporarySubscription,
        body.passthrough,
      );

      await this.subscriptionDataRepository.save(newTemporarySubscription);

      return;
    }

    // only for subscriptions
    if (
      body.alert_name == "high_risk_transaction_updated" &&
      getTierToPaddleSPIMap()[+body.product_id]
    ) {
      const temporarySubscription =
        await this.subscriptionDataRepository.findOne({
          where: {
            id: body.case_id,
          },
          relations: ["owner"],
        });

      if (!temporarySubscription) return; // Already handled

      if (temporarySubscription.data.t != PaymentProcessors.PADDLE)
        throw new HttpException(
          "Incorrect subscriptionData type.",
          HttpStatus.INTERNAL_SERVER_ERROR,
        );

      // await this.subscriptionDataRepository.delete(temporarySubscription.id);

      if (body.status == "rejected") {
        // doom
        temporarySubscription.owner.subscriptionTier = SubscriptionTiers.NONE;
        temporarySubscription.owner.subscriptionUntil = new Date(0);

        await this.usersRepository.save(temporarySubscription.owner);
      }

      return;
    }

    if (!body.subscription_id) return;

    let mutex: Mutex | null = null;

    if (this._redis_client) {
      if (!this._webhook_locks.has(body.subscription_id))
        this._webhook_locks.set(
          body.subscription_id,
          new Mutex(this._redis_client, body.subscription_id),
        );

      mutex = this._webhook_locks.get(body.subscription_id);
      await mutex.acquire();
    }

    try {
      // console.log(body);

      const subscriptionData =
        (await this.subscriptionDataRepository.findOne({
          where: {
            id: body.subscription_id,
          },
          relations: ["owner"],
        })) ||
        this.subscriptionDataRepository.create({
          id: body.subscription_id,
          paymentProcessor: PaymentProcessors.PADDLE,
          data: {
            t: PaymentProcessors.PADDLE,
          },
        } as SubscriptionData);

      if (subscriptionData.data.t != PaymentProcessors.PADDLE)
        throw new HttpException(
          "Incorrect subscriptionData type.",
          HttpStatus.INTERNAL_SERVER_ERROR,
        );

      switch (body.alert_name) {
        case "subscription_created":
          subscriptionData.data = {
            ...subscriptionData.data,

            c: body.cancel_url,
            u: body.update_url,

            s: body.status,

            p: +body.subscription_plan_id,
            o: body.checkout_id,

            n: Math.floor(+new Date(body.next_bill_date) / 1000),
          } as PaddleSubscriptionData;

          break;
        case "subscription_updated":
          subscriptionData.data = {
            ...subscriptionData.data,

            p: +body.subscription_plan_id,

            c: body.cancel_url,
            u: body.update_url,

            s: body.status,

            n: Math.floor(+new Date(body.next_bill_date) / 1000),
          } as PaddleSubscriptionData;

          if (subscriptionData.owner) {
            if (body.paused_from)
              subscriptionData.owner.subscriptionUntil = new Date(
                body.paused_from,
              );

            // if (body.status == "active")
            //   subscriptionData.owner.subscriptionTier =
            //     PaddleSubscriptionPlanIdToSubscriptionTier[
            //       subscriptionData.data.p
            //     ];

            if (body.status == "paused")
              subscriptionData.owner.subscriptionTier = SubscriptionTiers.NONE;

            await this.usersRepository.save(subscriptionData.owner);
          }

          break;
        case "subscription_cancelled":
          await this.subscriptionDataRepository.delete({
            id: subscriptionData.id,
          });

          if (subscriptionData.owner) {
            subscriptionData.owner.subscriptionUntil = new Date(
              body.cancellation_effective_date,
            );

            await this.usersRepository.save(subscriptionData.owner);
          }

          return;
        case "subscription_payment_succeeded":
          // Check if this is a one-off charge, those charges have unit_price set to "0". If so, discard.
          if (body.unit_price == "0") break;

          await sleep(200); // A bad measure to possibly prevent race conditions.

          subscriptionData.data = {
            ...subscriptionData.data,

            s: body.status,
            r: body.order_id,

            p: +body.subscription_plan_id,

            n: Math.floor(+new Date(body.next_bill_date) / 1000),
          };

          const userWithSubscriptionDataId =
            await this.usersRepository.findOneBy({
              subscriptionData: { id: body.subscription_id },
            });

          if (userWithSubscriptionDataId)
            subscriptionData.owner = userWithSubscriptionDataId;

          /*const existingSubscriptionDataObject = await this.subscriptionDataRepository.findOne(
              body.subscription_id,
            );

            if (existingSubscriptionDataObject) {
              subscriptionData.id = existingSubscriptionDataObject.id;
              subscriptionData.createdAt =
                existingSubscriptionDataObject.createdAt;
            }*/

          if (subscriptionData.owner) {
            // Only have one subscription data per user
            await this.ensureOneSubscriptionDataInstance(subscriptionData);

            await this.initializeUserSubscription(
              subscriptionData.owner,
              PaddleSubscriptionPlanIdToSubscriptionTier[
                subscriptionData.data.p
              ],
              subscriptionData.data.n,
              subscriptionData,
            );
          } else if (body.passthrough)
            await this.bindSubscriptionByPassthrough(
              subscriptionData,
              body.passthrough,
            );

          break;
      }

      await this.subscriptionDataRepository.save(subscriptionData);
    } finally {
      if (mutex) await mutex.release();
    }
  }

  async bindSubscriptionByPassthrough(
    subscriptionData: SubscriptionData,
    passthrough: string,
  ) {
    if (subscriptionData.data.t != PaymentProcessors.PADDLE) return;

    const data = JSON.parse(passthrough);

    if (!data.auth_token || typeof data.auth_token != "string") return;

    const authToken = data.auth_token;

    const userData = await this.authService.userIdByJWT(authToken, true); // ignore token expiration for correct subscription rebind
    if (!userData) return;

    const user = await this.usersRepository.findOne({
      where: {
        id: userData.id,
      },
      relations: ["subscriptionData"],
    });

    if (!user) return;
    if (user.hasSubscription()) {
      if (
        user.subscriptionData &&
        user.subscriptionData.data &&
        user.subscriptionData.data.t == PaymentProcessors.PADDLE &&
        user.subscriptionData.data.s != "high_risk"
      )
        return;
    }

    subscriptionData.owner = user;

    // Only have one subscription data per user
    await this.ensureOneSubscriptionDataInstance(subscriptionData);

    await this.initializeUserSubscription(
      user,
      PaddleSubscriptionPlanIdToSubscriptionTier[subscriptionData.data.p],
      subscriptionData.data.n,
      subscriptionData,
    );
  }

  async bindSubscription(user: User, body: BindSubscriptionRequest) {
    if (user.userType != UserType.RETAIL)
      throw new HttpException(
        "Cannot bind a new subscription on non-retail user types.",
        HttpStatus.CONFLICT,
      );

    await this.loadSubscriptionData(user);
    if (
      user.hasSubscription() &&
      user.subscriptionData &&
      user.subscriptionData.paymentProcessor == PaymentProcessors.PADDLE &&
      body.paymentProcessor != "giftkey"
    )
      throw new HttpException(
        "Cannot bind a new subscription while there's an active recurring subscription.",
        HttpStatus.CONFLICT,
      );

    switch (body.paymentProcessor) {
      case "paddle":
        const result = await this.subscriptionDataRepository
          .createQueryBuilder("subscription")
          .select()
          .where(
            "subscription.data ::jsonb @> :search_object AND subscription.ownerId IS NULL",
            {
              search_object: {
                o: body.subscriptionId,
              },
            },
          )
          .execute();

        if (result.length <= 0)
          throw new HttpException(
            "Subscription ID not found.",
            HttpStatus.NOT_FOUND,
          );

        const subDataId = result[0].subscription_id;
        const subscriptionData =
          await this.subscriptionDataRepository.findOneBy({ id: subDataId });

        if (subscriptionData.data.t != PaymentProcessors.PADDLE)
          throw new HttpException(
            "Incorrect subscriptionData type.",
            HttpStatus.INTERNAL_SERVER_ERROR,
          );

        if (subscriptionData.data.s != "active")
          throw new HttpException(
            "Cannot bind an inactive subscription.",
            HttpStatus.CONFLICT,
          );

        subscriptionData.owner = user;

        // Only have one subscription data per user
        await this.ensureOneSubscriptionDataInstance(subscriptionData);

        await this.initializeUserSubscription(
          user,
          PaddleSubscriptionPlanIdToSubscriptionTier[subscriptionData.data.p],
          subscriptionData.data.n,
          subscriptionData,
        );

        break;
      case "giftkey":
        const keyData = await this.subscriptionDataRepository.findOneBy({
          id: body.subscriptionId,
        });

        if (!keyData)
          throw new HttpException("Key not found.", HttpStatus.NOT_FOUND);

        if (keyData.data.t != PaymentProcessors.GIFT_KEY)
          throw new HttpException(
            "Incorrect subscriptionData type.",
            HttpStatus.INTERNAL_SERVER_ERROR,
          );

        if (keyData.owner || keyData.data.u)
          throw new HttpException(
            "Cannot bind a used key.",
            HttpStatus.CONFLICT,
          );

        // jank
        if (keyData.data.k.startsWith("steps-")) {
          if (typeof user.purchasedModuleTrainingSteps === "undefined")
            user.purchasedModuleTrainingSteps = 0;

          user.purchasedModuleTrainingSteps += keyData.data.l;

          keyData.data.u = true;

          await this.subscriptionDataRepository.save(keyData);
          await this.usersRepository.save(user);
        } else {
          if (
            user.hasSubscription() &&
            user.subscriptionData &&
            user.subscriptionData.paymentProcessor == PaymentProcessors.PADDLE
          )
            throw new HttpException(
              "Cannot bind a new subscription while there's an active recurring subscription.",
              HttpStatus.CONFLICT,
            );

          await this.usersRepository.save(user); // Ensure user ID exists before setting it in keyData (giftkey registration)

          keyData.data.u = true;
          keyData.owner = user;

          // Only have one subscription data per user
          await this.ensureOneSubscriptionDataInstance(keyData);

          await this.initializeUserSubscription(
            user,
            keyData.data.i,
            Math.floor(+new Date() / 1000) + keyData.data.l,
            keyData,
          );
        }

        break;
      case "trial":
        throw new HttpException("Disabled", HttpStatus.GONE);

        if (user.hasSubscription())
          throw new HttpException(
            "A subscription is already present.",
            HttpStatus.CONFLICT,
          );

        if (user.trialActivated)
          throw new HttpException(
            "The trial has already been activated on this account.",
            HttpStatus.CONFLICT,
          );

        if (!user.emailVerified)
          throw new HttpException(
            "This account does not have a verified email address.",
            HttpStatus.CONFLICT,
          );

        user.trialActivated = true;
        user.trialActions = TRIAL_ACTIONS_AMOUNT;

        await this.usersRepository.save(user);

        /*await this.initializeUserSubscription(
          user,
          SubscriptionTiers.TABLET,
          Math.floor(+new Date() / 1000) + TRIAL_LENGTH_SECONDS,
          null,
        );*/

        break;
    }
  }

  async checkGiftKey(key: string) {
    const keyData = await this.subscriptionDataRepository.findOneBy({
      id: key,
    });
    if (!keyData) return false;
    if (keyData.owner || keyData.data.u) return false;
    if (keyData.data.t != PaymentProcessors.GIFT_KEY) return false;
    if (keyData.data.k.startsWith("steps-")) return false;
    return true;
  }

  /*async fixSubscriptionDataIfRequired(user: User) {
    if (!user.hasSubscription() || !user.subscriptionData) return;

    if (user.subscriptionData.paymentProcessor == PaymentProcessors.PADDLE) {
      if (user.subscriptionData.data.t != PaymentProcessors.PADDLE) return;

      if (!user.subscriptionData.data.c || !user.subscriptionData.data.u) {
        const data = await paddleGetSubscriptionData(user.subscriptionData.id);
        if (!data.success) {
          console.error(
            "fixSubscriptionDataIfRequired: paddleGetSubscriptionData error",
            data.error.message,
          );
          return;
        }

        if (
          !data.response ||
          !Array.isArray(data.response) ||
          !data.response.length
        ) {
          console.error(
            "fixSubscriptionDataIfRequired: invalid data structure",
            data.error.message,
          );
          return;
        }

        const userObj = data.response[0];

        user.subscriptionData.data.c = userObj.cancel_url;
        user.subscriptionData.data.u = userObj.update_url;

        await this.subscriptionDataRepository.save(user.subscriptionData);
      }
    }
  }*/

  async tryChangeSubscriptionPlan(user: User, tierSKU: SubscriptionTiers) {
    if (user.userType != UserType.RETAIL)
      throw new HttpException(
        "Cannot change subscription on non-retail user types.",
        HttpStatus.CONFLICT,
      );

    if (!user.hasSubscription())
      throw new HttpException(
        "Cannot change subscription plan while not subscribed.",
        HttpStatus.CONFLICT,
      );

    if (user.subscriptionTier == tierSKU)
      throw new HttpException(
        "Cannot change subscription plan to the same plan.",
        HttpStatus.CONFLICT,
      );

    if (isTierABetterThanB(user.subscriptionTier, tierSKU))
      throw new HttpException(
        "Cannot downgrade subscription plan.",
        HttpStatus.CONFLICT,
      );

    await this.loadSubscriptionData(user);
    if (
      !user.subscriptionData ||
      user.subscriptionData.paymentProcessor == PaymentProcessors.NONE
    )
      throw new HttpException(
        "Cannot change subscription plan if it was manually assigned.",
        HttpStatus.CONFLICT,
      );

    switch (user.subscriptionData.paymentProcessor) {
      case PaymentProcessors.PADDLE:
        if (user.subscriptionData.data.t != PaymentProcessors.PADDLE)
          throw new HttpException(
            "Incorrect payment processor data for this record.",
            HttpStatus.INTERNAL_SERVER_ERROR,
          );

        try {
          const data = await paddleUpdateSubscription(
            +user.subscriptionData.id,
            getTierToPaddleSPIMap()[tierSKU],
          );

          if (!data.success)
            throw new HttpException(
              data.error
                ? data.error.message
                : "The request was not successful yet no error was provided.",
              HttpStatus.CONFLICT,
            );
        } catch (ex) {
          console.error(ex); // for troubleshooting

          throw new HttpException(
            "Payment processor error.",
            HttpStatus.CONFLICT,
          );
        }

        break;
      case PaymentProcessors.GIFT_KEY:
        throw new HttpException(
          "Cannot change subscription plan if it was assigned through a Gift Key.",
          HttpStatus.CONFLICT,
        );
    }
  }

  async generateGiftKey(
    tier: SubscriptionTiers,
    length: number,
    prefix = "",
    referrer?: User,
  ): Promise<string | null> {
    // Should be secure *enough*
    let key = nanoid() + nanoid();
    key = key.replace(/\_/g, "=");

    const keySubscriptionData = this.subscriptionDataRepository.create({
      id: key,
      paymentProcessor: PaymentProcessors.GIFT_KEY,
      data: {
        t: PaymentProcessors.GIFT_KEY,
        k: prefix + key,
        i: +tier,
        l: length,
        u: false,
      },
      referrer: referrer,
    });

    await this.subscriptionDataRepository.save(keySubscriptionData);

    return key;
  }
}
