import {
  Body,
  Controller,
  Header,
  HttpCode,
  HttpException,
  HttpStatus,
  Post,
  Req,
} from "@nestjs/common";
import {
  ApiBadRequestResponse,
  ApiBearerAuth,
  ApiConflictResponse,
  ApiCreatedResponse,
  ApiExcludeEndpoint,
  ApiInternalServerErrorResponse,
  ApiNotFoundResponse,
  ApiOkResponse,
  ApiTags,
  ApiUnauthorizedResponse,
} from "@nestjs/swagger";
import { SkipThrottle, Throttle } from "@nestjs/throttler";
import { RequireAccessToken } from "../../shared/guards/require-access-token.guard";
import { ApiError } from "../../shared/dto/api-error.dto";
import { User } from "../user.entity";
import { BindSubscriptionRequest } from "./dto/bind-subscription.dto";
import { SubscriptionService } from "./subscription.service";
import { ChangeSubscriptionPlanRequest } from "./dto/change-subscription-plan.dto";
import { MAX_TIER, SubscriptionTiers } from "../../shared/subscription-tiers";

@ApiTags("/user/subscription/")
@Controller("user/subscription")
@ApiInternalServerErrorResponse({
  description: "An unknown error occured.",
  type: ApiError,
})
export class SubscriptionController {
  constructor(private readonly subscriptionService: SubscriptionService) {}

  @Post("bind")
  @ApiBearerAuth()
  @RequireAccessToken()
  @Throttle(2, 10)
  @Header("Cache-Control", "no-store")
  @ApiCreatedResponse({
    description: "Subscription has been bound properly.",
  })
  @ApiUnauthorizedResponse({
    description: "Access Token is incorrect.",
    type: ApiError,
  })
  @ApiBadRequestResponse({
    description: "A validation error occured.",
    type: ApiError,
  })
  @ApiConflictResponse({
    description: "A conflict occured while binding subscription.",
    type: ApiError,
  })
  @ApiNotFoundResponse({
    description: "Subscription ID was not found",
    type: ApiError,
  })
  async bindSubscription(@Req() req, @Body() body: BindSubscriptionRequest) {
    if (!body) throw new HttpException("Invalid body.", HttpStatus.BAD_REQUEST);

    const user: User = req.user;
    return await this.subscriptionService.bindSubscription(user, body);
  }

  @Post("change")
  @ApiBearerAuth()
  @RequireAccessToken()
  @Throttle(2, 10)
  @HttpCode(HttpStatus.OK)
  @ApiOkResponse({
    description: "Subscription plan has been changed properly.",
  })
  @ApiUnauthorizedResponse({
    description: "Access Token is incorrect.",
    type: ApiError,
  })
  @ApiBadRequestResponse({
    description: "A validation error occured.",
    type: ApiError,
  })
  @ApiConflictResponse({
    description: "A conflict occured while changing subscription plan.",
    type: ApiError,
  })
  @ApiNotFoundResponse({
    description: "Subscription SKU was not found",
    type: ApiError,
  })
  async changeSubscriptionPlan(
    @Req() req,
    @Body() body: ChangeSubscriptionPlanRequest,
  ) {
    if (!body) throw new HttpException("Invalid body.", HttpStatus.BAD_REQUEST);

    const user: User = req.user;
    const tierSKU: SubscriptionTiers =
      SubscriptionTiers[body.newSubscriptionPlan as unknown as string];

    return await this.subscriptionService.tryChangeSubscriptionPlan(
      user,
      tierSKU,
    );
  }

  @Post("paddle/webhook")
  @SkipThrottle()
  @Header("Cache-Control", "no-store")
  @HttpCode(HttpStatus.OK)
  @ApiExcludeEndpoint(true)
  async paddleWebhook(@Body() body: Record<any, any>) {
    if (!body) throw new HttpException("Invalid body.", HttpStatus.BAD_REQUEST);
    await this.subscriptionService.onPaddleWebhook(body);
  }

  @Post("giftkey/paddle/webhook")
  @SkipThrottle()
  @Header("Cache-Control", "no-store")
  @HttpCode(HttpStatus.OK)
  @ApiExcludeEndpoint(true)
  async giftkeyPaddleWebhook(@Body() body: Record<any, any>) {
    if (!body) throw new HttpException("Invalid body.", HttpStatus.BAD_REQUEST);
    return await this.subscriptionService.onGiftkeyPaddleWebhook(body);
  }

  @Post("steps/paddle/webhook")
  @SkipThrottle()
  @Header("Cache-Control", "no-store")
  @HttpCode(HttpStatus.OK)
  @ApiExcludeEndpoint(true)
  async stepsPaddleWebhook(@Body() body: Record<any, any>) {
    if (!body) throw new HttpException("Invalid body.", HttpStatus.BAD_REQUEST);
    return await this.subscriptionService.onStepsPaddleWebhook(body);
  }

  @Post("giftkey/generate")
  @Header("Cache-Control", "no-store")
  @ApiExcludeEndpoint(true)
  async generateGiftKeys(@Req() req, @Body() data: any) {
    if (!process.env.ADMIN_PASSWORD)
      throw new HttpException("Endpoint disabled.", HttpStatus.BAD_REQUEST);

    if (req.headers["x-admin-token"] !== process.env.ADMIN_PASSWORD)
      throw new HttpException("Invalid admin token.", HttpStatus.UNAUTHORIZED);

    const tier = data["tier"];
    if (
      !tier ||
      isNaN(+tier) ||
      tier < SubscriptionTiers.TABLET ||
      tier > MAX_TIER
    )
      throw new HttpException("Invalid tier.", HttpStatus.BAD_REQUEST);

    let length = data["length"];
    if (!length || isNaN(+length))
      throw new HttpException(
        "Invalid length (in seconds).",
        HttpStatus.BAD_REQUEST,
      );

    length = +length;

    let amount = data["amount"];
    if (!amount || isNaN(+amount)) amount = 1;

    const prefix = data["prefix"] || "";

    const keys = [];

    for (let i = 0; i < amount; i++)
      keys.push(
        await this.subscriptionService.generateGiftKey(tier, length, prefix),
      );

    return keys;
  }
}
