import { PaymentProcessors } from "./payment-processors";
import { SubscriptionTiers } from "../../shared/subscription-tiers";
import fetch from "node-fetch";
import { verifyPaddleWebhook } from "verify-paddle-webhook";

const PRODUCTION_PUBLIC_KEY = `-----BEGIN PUBLIC KEY-----
MIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIICCgKCAgEAuS5WlmbG92zr5WUmWinv
+1SVW55rhPbU02kEYJmu2rJgwsWe19oB64/Zx3jHs6bZSj9mxVgxzw8n8EzGtr5T
hg41fa2YrFAKV0yfd9GN+SGtCAFZqWmwDNE4vQkWm57jOJB8IBo32Uhcx55wMOT5
b9MVMkBul1pZ97Gsmkn9CpZpXnpQRS5+9oJWqssMQXXMBD9Uw5+j922LzAjdx3Ie
5ZD5OAmhSWkv+dMIJw4SFHuMMfm7pqc+gODVXltaBQPCv9zNfBL5E6Hwh/sZJjrb
z/ypllyFt15VXUCkoZ82szkPU6qYvQPoJVBs/t2Av+4h+jIfioC+9HKGk5x30Kqb
joE/F3g4rm56Ge5gm1WA+t8oYIJwInN2uvqtwUq2D2yEuxruys3yMN3CK/xoCmKb
/cn7Pc5ukiE3R9oWDxozqBgR2cVUYPDBPL0WQDOYa4Mg/abLxd34rkyWWohUTF0G
tYGT+2lT6myBGrzPX6clMiiQgmuUMSo1W8vNwU4mtzTmGZmEi9dK2soKhNrkjFTx
xyCn7wCo+3BVHJAeCGc06xK8lb78btI3Tvj6QYfmT+aoBPcSOd0E1oIkAWKZVLgM
uPoSWXMUlpDeTxOpgmc3qBrDMf45mf8jYomrYOZPHWJV7gGW3OI2dV4QTezK3+Rq
CZNSYAuQYcKw+ZQRiwrKLYkCAwEAAQ==
-----END PUBLIC KEY-----`;

const SANDBOX_PUBLIC_KEY = `-----BEGIN PUBLIC KEY-----
MIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIICCgKCAgEAw2ykOhznx4035X2h086M
FKuVvq0zxQ4OJGtQwtF5QwSPfI2x22PJ96ZI514klguQhdvztmMISJTfUAjG6HKO
u1H+p8dO0ecx1p48GQ4mD3f048Xylwhg1cJv3/rqx1kwOA76r9zj4ZTgKEjpQgkb
j9PCzE2XQwuuLQWv04uxJHrlwyBSvW2d3Paudnm98I3JSiJxtiaw4LLMWLfDI8vN
CWfb772IImJzZjBlLFimnbLpDn2OCOL62Qg5wRBTLBnoGmQzdtHNRsY0ai4bUnvS
D65CTzdWvBCb8PaZVNpIrmjI4gcljINgoN4RcmBeLW6kKWnwkcXl8ZQJGkFw7QS8
kWZMKDW0d4znntikVGkxHEZrMcfpBkq40eg2qn3NT7ivEFUa0z/hcpaNneuHKKiB
LQu9RkREba13NKRlG8hEDWWFFbOAeqt/zcJYyaGxK+OSU+s+e9A4w+zESUJfPIF/
WUIu1bEvrqckrV4Wf1NkC9HWFr+EK91OPsKsmp102aeWttayDApIlUi8sFcsDgO3
Ap//RvekfqrGPdrYjw/xMGq9nUcCEHCrVTROnr7KAm51sRHtM2DARw0GNmLtSlc8
CWvFOuuPeuNKnA9XgmglQMI7ret79NXvQINTN7MPjYP4TJPShLoTOXpk2XolSXJv
+awrxOZ396Gxv1mVPFjvxN8CAwEAAQ==
-----END PUBLIC KEY-----`;

async function paddleWebAPIRequest(
  path,
  method = "GET",
  data = null,
): Promise<any> {
  const BASE_URL =
    process.env.NODE_ENV === "production"
      ? "https://vendors.paddle.com/api/2.0"
      : "https://sandbox-vendors.paddle.com/api/2.0";

  if (data && typeof data == "object") {
    data.vendor_id = process.env.PADDLE_VENDOR_ID;
    data.vendor_auth_code = process.env.PADDLE_VENDOR_TOKEN;
  }

  const stringified_data = JSON.stringify(data);

  const res = await fetch(BASE_URL + path, {
    method,
    body: data ? stringified_data : undefined,
    headers: {
      Accept: "application/json",
      "Content-Type": "application/json",
    },
  });

  if (res.status >= 400)
    throw new Error(`Failed to send request to Paddle! ${stringified_data}`);

  const json = await res.json();
  return json;
}

export async function paddleGetSubscriptionData(subscription_id) {
  return paddleWebAPIRequest("/subscription/users", "POST", {
    subscription_id,
  });
}

export async function paddleUpdateSubscription(subscription_id, plan_id) {
  return paddleWebAPIRequest("/subscription/users/update", "POST", {
    subscription_id,
    plan_id,
    quantity: 1,
    prorate: true,
    bill_immediately: true,
  });
}

export async function paddleCreateSubscriptionCharge(
  subscription_id,
  amount,
  charge_name,
) {
  return paddleWebAPIRequest(
    `/subscription/${subscription_id}/charge`,
    "POST",
    {
      amount,
      charge_name,
    },
  );
}

export function isValidPaddleWebhookData(data: any): boolean {
  return verifyPaddleWebhook(
    process.env.NODE_ENV === "production"
      ? PRODUCTION_PUBLIC_KEY
      : SANDBOX_PUBLIC_KEY,
    data,
  );
}

export interface PaddleSubscriptionData {
  t: PaymentProcessors.PADDLE; // Type
  c: string; // Cancel URL
  u: string; // Update Payment Information URL
  p: number; // Subscription Plan ID
  s: string; // Status
  o: string; // Checkout ID
  r: string | null; // Order ID, null until subscription_payment_succeeded
  n: number; // Next bill date, UNIX timestamp
}

export const PaddleSubscriptionPlanIdToSubscriptionTier: {
  [key: number]: SubscriptionTiers;
} = {
  // Production
  657630: SubscriptionTiers.TABLET,
  657632: SubscriptionTiers.SCROLL,
  657633: SubscriptionTiers.OPUS,
  // 0: SubscriptionTiers.CODEX, TODO

  // Sandbox
  12384: SubscriptionTiers.TABLET,
  12385: SubscriptionTiers.SCROLL,
  12386: SubscriptionTiers.OPUS,
  // 34070: SubscriptionTiers.CODEX,
};

type SubscriptionTierToPaddleSubscriptionPlanId = {
  [key in SubscriptionTiers]: number;
};

const ProductionSubscriptionTierToPaddleSubscriptionPlanId: SubscriptionTierToPaddleSubscriptionPlanId =
  {
    [SubscriptionTiers.NONE]: 0,
    [SubscriptionTiers.TABLET]: 657630,
    [SubscriptionTiers.SCROLL]: 657632,
    [SubscriptionTiers.OPUS]: 657633,
    // [SubscriptionTiers.CODEX]: 0, // TODO
  };

const SandboxSubscriptionTierToPaddleSubscriptionPlanId: SubscriptionTierToPaddleSubscriptionPlanId =
  {
    [SubscriptionTiers.NONE]: 0,
    [SubscriptionTiers.TABLET]: 12384,
    [SubscriptionTiers.SCROLL]: 12385,
    [SubscriptionTiers.OPUS]: 12386,
    // [SubscriptionTiers.CODEX]: 34070,
  };

const ProductionGiftKeyTierToPaddleProductId: SubscriptionTierToPaddleSubscriptionPlanId =
  {
    [SubscriptionTiers.NONE]: 0,
    [SubscriptionTiers.TABLET]: 767856,
    [SubscriptionTiers.SCROLL]: 767855,
    [SubscriptionTiers.OPUS]: 767584,
    // [SubscriptionTiers.CODEX]: 0, // TODO
  };

const SandboxGiftKeyTierToPaddleProductId: SubscriptionTierToPaddleSubscriptionPlanId =
  {
    [SubscriptionTiers.NONE]: 0,
    [SubscriptionTiers.TABLET]: 26963,
    [SubscriptionTiers.SCROLL]: 27468,
    [SubscriptionTiers.OPUS]: 27469,
    // [SubscriptionTiers.CODEX]: 34071,
  };

const ProductionStepsToPaddleProductId: Record<number, number> = {
  [793218]: 2000,
  [793222]: 5000,
  [793224]: 10000,
};

const SandboxStepsToPaddleProductId: Record<number, number> = {
  [36187]: 2000,
  [36186]: 5000,
  [36185]: 10000,
};

export function getTierToPaddleSPIMap(): SubscriptionTierToPaddleSubscriptionPlanId {
  return process.env.NODE_ENV === "production"
    ? ProductionSubscriptionTierToPaddleSubscriptionPlanId
    : SandboxSubscriptionTierToPaddleSubscriptionPlanId;
}

export function getTierToPaddlePIMap(): SubscriptionTierToPaddleSubscriptionPlanId {
  return process.env.NODE_ENV === "production"
    ? ProductionGiftKeyTierToPaddleProductId
    : SandboxGiftKeyTierToPaddleProductId;
}

export function getStepsToPaddlePIMap(): Record<number, number> {
  return process.env.NODE_ENV === "production"
    ? ProductionStepsToPaddleProductId
    : SandboxStepsToPaddleProductId;
}
