import {
  applyDecorators,
  CanActivate,
  ExecutionContext,
  HttpException,
  HttpStatus,
  Injectable,
  SetMetadata,
  UseGuards,
} from "@nestjs/common";
import { User } from "../../user/user.entity";
import { AuthService } from "../../user/auth.service";
import { UserService } from "../../user/user.service";
import { hashAccessKey } from "../access-key-utils";
import { Reflector } from "@nestjs/core";

type TokenType = "bearer" | "persistent";

interface AccessTokenData {
  type: TokenType;
  token: string;
}

function getAccessTokenDataFromRequestHeaders(
  request: any,
): AccessTokenData | null {
  const authorization = request.headers["authorization"];
  if (!authorization) return null;

  const headerSplit: string[] = authorization.split(" ");
  if (headerSplit.length != 2)
    throw new HttpException(
      "Invalid Authorization header content.",
      HttpStatus.BAD_REQUEST,
    );

  const tokenType = headerSplit[0].toLowerCase();
  if (tokenType !== "bearer" && tokenType !== "persistent")
    throw new HttpException(
      "Invalid accessToken type.",
      HttpStatus.BAD_REQUEST,
    );

  const accessToken = headerSplit[1];
  if (!accessToken)
    throw new HttpException(
      "accessToken not found in Authorization header content.",
      HttpStatus.BAD_REQUEST,
    );

  return {
    type: tokenType as TokenType,
    token: accessToken,
  };
}

// eslint-disable-next-line @typescript-eslint/no-unused-vars
function getAccessTokenDataFromRequestCookies(
  request: any,
): AccessTokenData | null {
  const token = request.cookies["at"];
  if (!token) return null;

  return {
    type: "bearer",
    token,
  };
}

export function getAccessTokenDataFromRequest(
  request: any,
): AccessTokenData | null {
  // eslint-disable-next-line prefer-const
  let data = getAccessTokenDataFromRequestHeaders(request);
  // if (!data) data = getAccessTokenDataFromRequestCookies(request);

  return data;
}

@Injectable()
class AccessTokenGuard implements CanActivate {
  constructor(
    private userService: UserService,
    private authService: AuthService,

    private reflector: Reflector,
  ) {}

  async canActivate(context: ExecutionContext): Promise<boolean> {
    const request = context.switchToHttp().getRequest();

    const credentialsData = getAccessTokenDataFromRequest(request);
    if (!credentialsData) {
      request.user = null;

      if (
        this.reflector.get<boolean>(
          "accessTokenIsOptional",
          context.getHandler(),
        )
      )
        return true;
      else
        throw new HttpException(
          "Missing accessToken for an authenticated endpoint.",
          HttpStatus.UNAUTHORIZED,
        );
    }

    let user!: User;

    const tokenType = credentialsData.type;
    const accessToken = credentialsData.token;

    switch (tokenType) {
      case "bearer":
        const tokenData = await this.authService.userIdByJWT(accessToken);
        if (!tokenData)
          throw new HttpException(
            "Invalid accessToken.",
            HttpStatus.UNAUTHORIZED,
          );

        user = await this.userService.userById(tokenData.id);
        if (!user)
          throw new HttpException("User not found.", HttpStatus.UNAUTHORIZED);

        if (user.authenticationNonce !== tokenData.nonce)
          throw new HttpException(
            "Invalid accessToken.",
            HttpStatus.UNAUTHORIZED,
          );
        break;
      case "persistent":
        user = await this.userService.findUser({
          persistentAccessToken: hashAccessKey(accessToken),
        });
        if (!user)
          throw new HttpException("User not found.", HttpStatus.UNAUTHORIZED);
        break;
      default:
        throw new HttpException(
          "accessToken type not supported.",
          HttpStatus.UNAUTHORIZED,
        );
    }

    context.switchToHttp().getRequest().user = user;
    return true;
  }
}

export function RequireAccessToken() {
  return applyDecorators(UseGuards(AccessTokenGuard));
}

export function OptionalAccessToken() {
  return applyDecorators(
    UseGuards(AccessTokenGuard),
    SetMetadata("accessTokenIsOptional", true),
  );
}
