import { User, UserType } from "../user/user.entity";
import { SubscriptionTiers } from "./subscription-tiers";

// Allowed text generation models list
export enum GenerationModels {
  "2.7B" = "2.7B",
  // "6B-v3" = "6B-v3",
  "6B-v4" = "6B-v4",
  // "euterpe-v0" = "euterpe-v0",
  "euterpe-v2" = "euterpe-v2",
  "genji-python-6b" = "genji-python-6b",
  "genji-jp-6b" = "genji-jp-6b",
  "genji-jp-6b-v2" = "genji-jp-6b-v2",
  // "krake-v1" = "krake-v1",
  "krake-v2" = "krake-v2",
  "hypebot" = "hypebot",
  "infillmodel" = "infillmodel",
  // "cassandra-v4" = "cassandra-v4",
}

// Allowed classification models list
export enum ClassificationModels {
  "debertaxxl",
  "bartlarge",
}

// Allowed voice models list
export enum VoiceModels {
  "v1" = "v1",
  "v2" = "v2",
}

// Default generation model used, if none are present or in API example
export const DEFAULT_GENERATION_MODEL = "euterpe-v2";

// Text generation model ACL settings
// null - No restrictions
export type GenerationModelAccessRights = {
  allowedSubscriptionTiers: SubscriptionTiers[] | null;
  allowedUserAccountTypes: UserType[] | null;
  accountRequired: boolean;
  trialCanAccess: boolean;
  trialDecreaseActions: boolean;
  canTrainModules: boolean;
};

export const GenerationModelAccessRightsData: {
  [key in GenerationModels]: GenerationModelAccessRights;
} = {
  ["2.7B"]: {
    allowedSubscriptionTiers: [
      SubscriptionTiers.TABLET,
      SubscriptionTiers.SCROLL,
      SubscriptionTiers.OPUS,
      // SubscriptionTiers.CODEX,
    ],
    allowedUserAccountTypes: null,
    accountRequired: false,
    trialCanAccess: true,
    trialDecreaseActions: false,
    canTrainModules: false,
  },
  /*["6B-v3"]: {
    allowedSubscriptionTiers: [
      SubscriptionTiers.TABLET,
      SubscriptionTiers.SCROLL,
      SubscriptionTiers.OPUS,
    ],
    allowedUserAccountTypes: null,
    accountRequired: true,
  },*/
  ["6B-v4"]: {
    allowedSubscriptionTiers: [
      SubscriptionTiers.TABLET,
      SubscriptionTiers.SCROLL,
      SubscriptionTiers.OPUS,
      // SubscriptionTiers.CODEX,
    ],
    allowedUserAccountTypes: null,
    accountRequired: false,
    trialCanAccess: true,
    trialDecreaseActions: true,
    canTrainModules: true,
  },
  /* ["euterpe-v0"]: {
    allowedSubscriptionTiers: [SubscriptionTiers.OPUS],
    allowedUserAccountTypes: null,
    accountRequired: true,
    trialCanAccess: false,
    trialDecreaseActions: true,
    canTrainModules: false,
  }, */
  ["euterpe-v2"]: {
    allowedSubscriptionTiers: [
      SubscriptionTiers.TABLET,
      SubscriptionTiers.SCROLL,
      SubscriptionTiers.OPUS,
      // SubscriptionTiers.CODEX,
    ],
    allowedUserAccountTypes: null,
    accountRequired: false,
    trialCanAccess: true,
    trialDecreaseActions: true,
    canTrainModules: true,
  },
  ["genji-python-6b"]: {
    allowedSubscriptionTiers: [SubscriptionTiers.OPUS], // , SubscriptionTiers.CODEX
    allowedUserAccountTypes: null,
    accountRequired: true,
    trialCanAccess: false,
    trialDecreaseActions: true,
    canTrainModules: false,
  },
  ["genji-jp-6b"]: {
    allowedSubscriptionTiers: [
      SubscriptionTiers.TABLET,
      SubscriptionTiers.SCROLL,
      SubscriptionTiers.OPUS,
      // SubscriptionTiers.CODEX,
    ],
    allowedUserAccountTypes: null,
    accountRequired: false,
    trialCanAccess: true,
    trialDecreaseActions: true,
    canTrainModules: false,
  },
  ["genji-jp-6b-v2"]: {
    allowedSubscriptionTiers: [
      SubscriptionTiers.TABLET,
      SubscriptionTiers.SCROLL,
      SubscriptionTiers.OPUS,
      // SubscriptionTiers.CODEX,
    ],
    allowedUserAccountTypes: null,
    accountRequired: false,
    trialCanAccess: true,
    trialDecreaseActions: true,
    canTrainModules: false,
  },
  /* ["krake-v1"]: {
    allowedSubscriptionTiers: [SubscriptionTiers.OPUS],
    allowedUserAccountTypes: null,
    accountRequired: true,
    trialCanAccess: false,
    trialDecreaseActions: true,
    canTrainModules: false,
  }, */
  ["krake-v2"]: {
    allowedSubscriptionTiers: [SubscriptionTiers.OPUS], // , SubscriptionTiers.CODEX
    allowedUserAccountTypes: null,
    accountRequired: true,
    trialCanAccess: false,
    trialDecreaseActions: true,
    canTrainModules: false,
  },
  ["hypebot"]: {
    allowedSubscriptionTiers: [
      SubscriptionTiers.TABLET,
      SubscriptionTiers.SCROLL,
      SubscriptionTiers.OPUS,
      // SubscriptionTiers.CODEX,
    ],
    allowedUserAccountTypes: null,
    accountRequired: false,
    trialCanAccess: true,
    trialDecreaseActions: true,
    canTrainModules: true,
  },
  ["infillmodel"]: {
    allowedSubscriptionTiers: [
      SubscriptionTiers.TABLET,
      SubscriptionTiers.SCROLL,
      SubscriptionTiers.OPUS,
      // SubscriptionTiers.CODEX,
    ],
    allowedUserAccountTypes: null,
    accountRequired: false,
    trialCanAccess: true,
    trialDecreaseActions: true,
    canTrainModules: true,
  },
  /*["cassandra-v4"]: {
    allowedSubscriptionTiers: [
      SubscriptionTiers.TABLET,
      SubscriptionTiers.SCROLL,
      SubscriptionTiers.OPUS,
      SubscriptionTiers.CODEX,
    ],
    allowedUserAccountTypes: null,
    accountRequired: false,
    trialCanAccess: true,
    trialDecreaseActions: true,
    canTrainModules: true,
  },*/
};

export type VoiceModelConfig = {
  baseUri: string;
};

export const VoiceModelData: {
  [key in VoiceModels]: VoiceModelConfig;
} = {
  ["v1"]: {
    baseUri: "https://novelai-tts.tenant-novelai.knative.chi.coreweave.com",
  },
  ["v2"]: {
    baseUri: "https://novelai-ttsv2.tenant-novelai.knative.chi.coreweave.com",
  },
};

export enum ImageGenerationModels {
  // "dalle-mega" = "dalle-mega",
  // "vqgan-diffusion" = "vqgan-diffusion",
  "stable-diffusion" = "stable-diffusion",
  "nai-diffusion" = "nai-diffusion",
  "safe-diffusion" = "safe-diffusion",
  // "waifu-diffusion" = "waifu-diffusion",
  "nai-diffusion-furry" = "nai-diffusion-furry",
}

type ImageGenerationSourceAmountCallback = (request: any) => number;

export type ImageGenerationModelSettings = {
  url: string;
  customResolutionSupported: boolean;
  defaultWidth: number;
  defaultHeight: number;
  getAmountOfSourcesToCreate: ImageGenerationSourceAmountCallback | null;
  tagsUrl: string | null;
  tagsModel: string | null;
};

// eslint-disable-next-line @typescript-eslint/no-unused-vars
const advancedSourcesToCreate = (request) => {
  /*let sources = 1;
  if (request.parameters.advanced) {
    sources = (+request.parameters.n_samples as number) || 1;
    request.parameters.n_samples = 1;
  }
  return sources;*/
  return 1;
};

export const ImageGenerationModelSettingsData: {
  [key in ImageGenerationModels]: ImageGenerationModelSettings;
} = {
  /// TODO: fill urls
  /*["dalle-mega"]: {
    url: "",
  },
  ["vqgan-diffusion"]: {
    url: "",
  },*/
  ["stable-diffusion"]: {
    url: "https://stable-diffusion.default.knative.anlatan.ai/generate-stream",
    customResolutionSupported: true,
    defaultWidth: 512,
    defaultHeight: 512,
    getAmountOfSourcesToCreate: advancedSourcesToCreate,
    tagsUrl: null,
    tagsModel: null,
  },
  ["nai-diffusion"]: {
    // url: "https://anime-diffusion-clip.default.knative.anlatan.ai/generate-stream",
    url:
      process.env.NODE_ENV === "production"
        ? "https://full-diffusion-predictor-default.tenant-novelai.knative.chi.coreweave.com/generate-stream"
        : "https://anime-diffusion-clip.default.knative.anlatan.ai/generate-stream",
    customResolutionSupported: true,
    defaultWidth: 512,
    defaultHeight: 512,
    getAmountOfSourcesToCreate: advancedSourcesToCreate,
    tagsUrl:
      "https://tagsearch.tenant-novelai.knative.chi.coreweave.com/predict-tags",
    tagsModel: "anime",
  },
  ["safe-diffusion"]: {
    // url: "https://anime-diffusion-clip-sfw.default.knative.anlatan.ai/generate-stream",
    url:
      process.env.NODE_ENV === "production"
        ? "https://curated-diffusion-predictor-default.tenant-novelai.knative.chi.coreweave.com/generate-stream"
        : "https://anime-diffusion-clip-sfw.default.knative.anlatan.ai/generate-stream",
    customResolutionSupported: true,
    defaultWidth: 512,
    defaultHeight: 512,
    getAmountOfSourcesToCreate: advancedSourcesToCreate,
    tagsUrl:
      "https://tagsearch.tenant-novelai.knative.chi.coreweave.com/predict-tags",
    tagsModel: "anime",
  },
  /*["waifu-diffusion"]: {
    url: "https://waifu-diffusion.default.knative.anlatan.ai/generate-stream",
    customResolutionSupported: true,
    defaultWidth: 512,
    defaultHeight: 512,
    getAmountOfSourcesToCreate: advancedSourcesToCreate,
    tagsUrl: null,
    tagsModel: null,
  },*/
  ["nai-diffusion-furry"]: {
    // url: "https://furry-diffusion.default.knative.anlatan.ai/generate-stream",
    url:
      process.env.NODE_ENV === "production"
        ? "https://furry-diffusion-predictor-default.tenant-novelai.knative.chi.coreweave.com/generate-stream"
        : "https://furry-diffusion.default.knative.anlatan.ai/generate-stream",
    customResolutionSupported: true,
    defaultWidth: 512,
    defaultHeight: 512,
    getAmountOfSourcesToCreate: advancedSourcesToCreate,
    tagsUrl:
      "https://tagsearch.tenant-novelai.knative.chi.coreweave.com/predict-tags",
    tagsModel: "furry",
  },
};

export function isUserEligibleForGenerationModel(
  user: User | null,
  model: GenerationModels,
): boolean {
  const accessRightsData = GenerationModelAccessRightsData[model];
  if (!accessRightsData) return false;

  if (accessRightsData.accountRequired && !user) return false;

  if (user) {
    if (
      accessRightsData.trialCanAccess &&
      user.trialActivated &&
      user.trialActions > 0
    )
      return true;

    if (Array.isArray(accessRightsData.allowedSubscriptionTiers)) {
      if (user.userType == UserType.RETAIL)
        if (!user.hasSubscription(true)) return false;

      if (
        !accessRightsData.allowedSubscriptionTiers.includes(
          user.subscriptionTier,
        )
      )
        return false;
    }

    if (Array.isArray(accessRightsData.allowedUserAccountTypes)) {
      if (!accessRightsData.allowedUserAccountTypes.includes(user.userType))
        return false;
    }
  }

  return true;
}
