import { Module } from "@nestjs/common";
import { ThrottlerModule } from "@nestjs/throttler";
import { TypeOrmModule } from "@nestjs/typeorm";
import { AIModule } from "./ai/ai.module";
import { AppController } from "./app.controller";
import { AppService } from "./app.service";
import { UserModule } from "./user/user.module";
import { WinstonModule } from "nest-winston";
import { format, transports } from "winston";
import { GoogleRecaptchaModule } from "@nestlab/google-recaptcha";
import { PromModule } from "@digikare/nestjs-prom";
import { ThrottlerStorageRedisService } from "./shared/throttler-storage-redis.service";
import { ScheduleModule } from "@nestjs/schedule";

const PRODUCTION_DO_CA = `-----BEGIN CERTIFICATE-----
MIIEQTCCAqmgAwIBAgIUL14KvcRNAv7qZc7+bYw37v/0r7YwDQYJKoZIhvcNAQEM
BQAwOjE4MDYGA1UEAwwvYTBlMGMyOWEtMTZmZC00OWUwLWIyNjEtODIwYWViMmE2
YjFkIFByb2plY3QgQ0EwHhcNMjEwNjE2MDcxMDE4WhcNMzEwNjE0MDcxMDE4WjA6
MTgwNgYDVQQDDC9hMGUwYzI5YS0xNmZkLTQ5ZTAtYjI2MS04MjBhZWIyYTZiMWQg
UHJvamVjdCBDQTCCAaIwDQYJKoZIhvcNAQEBBQADggGPADCCAYoCggGBAKi0wvJy
H/QcW8gQ7QF/YErBrxdL/dNJFREOYK1OVbmK2v4Wi22UmKqNEC8Kzm3bZ84xiV74
MvrN4CHrOiHL/mm4zV1T2vV7ifJqPdNszvypJAWSPEnnKLMLkYeL8FTENbzN3ci0
vUMQHAbDNGs0wHKT3vaZC136NUwBXCMGa8U45GCpdIoKFpqHoddwCPUTOX4Bjh3c
KghIxTlvyltJWc9awheK1vu3lugQxO0A7XlsT9NCPCu0ACsJ3XXVyKVBM/hRU+kz
X8ydcXtshOrL9Q5ZXXwptzXaxgCU7wJWBGlBrzx743oouz1s9owylROTMu8oWNqI
FoA/AaycQ4VXsWD14bqUL5l1VS9SeR879LIP5o8yknasXcgftYs6BRQfTTrvSQ4T
yB0ALrWToGjqIFkERgeD9FMPryM+rwcaBmgLLMNzo0lXbap1HWSK0XCChe20b7Ea
4v3rBy6UY4i9ERBG6SIakQyckNtTgxq/UrnlM0rakGSSZxD8+8PbSf8w4QIDAQAB
oz8wPTAdBgNVHQ4EFgQU7ZaNm3Yvr1vqYjIa6TtXhqtBGEswDwYDVR0TBAgwBgEB
/wIBADALBgNVHQ8EBAMCAQYwDQYJKoZIhvcNAQEMBQADggGBAA0A7+iEzNpvUlfP
LoZ6xmNYUCgVsjkml6bQvQtsWTLD7ALSAB9vSkt+KGyWWyd9Qk5KNgebM0sl/DrD
PYGJ7W9fzbXM9E8BrEGwAePd253Vk98N1ORCAikC9Xop7i1Mu/3A/QIvmR5cWYM8
Y/NU6T+0nkBg78LAmIKuoQSph9omvl0w8OULsPwJBcUS5Qvuip8sEltr4gRYa7o5
vSU7QIiYW1rmrcVHghVMuiH/ekHSyBjtT6xLkXi/rHxvTgb3uaP6NVCm7UhVvqKF
K2ptOdi9oHh0Mav2Rnri6d5vbO/laiS0P9Z4jOimXGMBCyW6arub9BXoiYWEo66h
1akw7r16MVx26O1AYFWIETZ57ZPLrfazgV0jLau1+x0J6AqMP0VzjoaTm5MUUi6t
Ufz+TceWwoJLBbtT73nx6WukiUPcCI+R0chHS+v8CexVPNIncbpmmNmmqr6FA5V7
ugQIIIjPsvfXT5XItbkBzw6Bd8ihQaZYrm1vZP7f3bm0OkF0Dw==
-----END CERTIFICATE-----
`;

@Module({
  imports: [
    TypeOrmModule.forRootAsync({
      useFactory: () => {
        return {
          type: "postgres",
          url: process.env.DB_URL,
          host: process.env.DB_HOST,
          port: process.env.DB_PORT ? parseInt(process.env.DB_PORT) : undefined,
          ssl:
            process.env.NODE_ENV === "production"
              ? { rejectUnauthorized: false, ca: PRODUCTION_DO_CA }
              : undefined,
          username: process.env.DB_USER,
          password: process.env.DB_PASSWORD,
          database: process.env.DB_NAME,
          entities: ["dist/**/*.entity.js"],
          migrations: ["dist/src/migrations/**/*.js"],
          synchronize: false,
          migrationsRun: true,
          logging: process.env.NODE_ENV === "development",
          cache:
            process.env.REDIS_HOST !== undefined
              ? {
                  type: "ioredis",
                  options: {
                    host: process.env.REDIS_HOST,
                    port: 6379,
                  },
                }
              : false,
          extra: {
            max: 20,
            connectionTimeoutMillis: 10000,
            statement_timeout: 30000,
            query_timeout: 15000,
          },
        };
      },
    }),
    ThrottlerModule.forRoot({
      ttl: 60,
      limit: 100,
      storage:
        process.env.REDIS_HOST !== undefined
          ? new ThrottlerStorageRedisService(
              {
                host: process.env.REDIS_HOST,
              },
              1000,
            )
          : undefined,
      ignoreUserAgents: [/Prometheus\/[0-9\.]*/, /SigurdBot\/f9835dcc/],
    }),
    WinstonModule.forRoot({
      format: format.combine(
        format.timestamp({ format: "isoDateTime" }),
        format.json(),
      ),
      transports: [new transports.Console()],
    }),
    GoogleRecaptchaModule.forRootAsync({
      useFactory: () => {
        return {
          secretKey: process.env.RECAPTCHA_SECRET_KEY,
          response: (req) => req.body && req.body.recaptcha,
        };
      },
    }),
    PromModule.forRoot({
      defaultLabels: {
        app: "novelai_backend",
        env: process.env.NODE_ENV || "development",
      },

      metricPath: "/Q2oloyzi9LZvqXAQAcT4pzDuORktuZYA",

      withHttpMiddleware: {
        enable: true,
        pathNormalizationExtraMasks: [/^[a-zA-Z0-9\-_]{21}$/],
      },
    }),
    ScheduleModule.forRoot(),
    UserModule,
    AIModule,
  ],
  controllers: [AppController],
  providers: [AppService],
})
export class AppModule {}
