import { ApiProperty } from "@nestjs/swagger";
import {
  IsBase64,
  IsEnum,
  IsInt,
  IsNotEmpty,
  IsNumber,
  IsString,
  Max,
  MaxLength,
  Min,
} from "class-validator";
import {
  DEFAULT_GENERATION_MODEL,
  GenerationModels,
} from "../../shared/generation-models";

export class AiModuleTrainRequest {
  @IsString()
  @IsNotEmpty()
  @IsBase64()
  @ApiProperty({
    description:
      "Base64-encoded data if ready or it's a training request, error text if error",
  })
  data: string;

  @IsNumber()
  @ApiProperty({
    description: "Learning rate",
  })
  lr: number;

  @IsInt()
  @ApiProperty({
    description: "Training steps",
  })
  steps: number;

  @IsString()
  @MaxLength(64)
  name: string;

  @IsString()
  @MaxLength(256)
  description: string;

  @IsString()
  @IsNotEmpty()
  @IsEnum(GenerationModels)
  @ApiProperty({
    description: "Used text generation model for module training",
    default: DEFAULT_GENERATION_MODEL,
    required: true,
  })
  model: string;
}

export class AiModuleDto extends AiModuleTrainRequest {
  id: string;

  @ApiProperty({
    description: "UNIX timestamp",
  })
  lastUpdatedAt: number;

  @ApiProperty({
    enum: ["pending", "training", "ready", "error"],
  })
  status: string;

  @ApiProperty({
    description: "Recorded loss values",
  })
  lossHistory: number[];
}

export class AiTrainingUpdate {
  id: string;
  status: "pending" | "training" | "ready" | "error";
  data: string;
}

export class BuyTrainingStepsRequest {
  @IsInt()
  @Min(2000)
  @Max(10000)
  @ApiProperty({
    type: "integer",
    minimum: 2000,
    maximum: 10000,
    description: "Amount of module training steps to purchase.",
  })
  amount: number;
}
