import {
  Body,
  Controller,
  Get,
  Header,
  HttpCode,
  HttpException,
  HttpStatus,
  ParseBoolPipe,
  ParseIntPipe,
  Post,
  Query,
  Req,
  Res,
} from "@nestjs/common";
import { HttpService } from "@nestjs/axios";
import {
  ApiBadRequestResponse,
  ApiBearerAuth,
  ApiConflictResponse,
  ApiCreatedResponse,
  ApiExcludeEndpoint,
  ApiInternalServerErrorResponse,
  ApiOkResponse,
  ApiResponse,
  ApiTags,
  ApiUnauthorizedResponse,
} from "@nestjs/swagger";
import { ApiError } from "../shared/dto/api-error.dto";
import {
  AIService,
  // ANONYMOUS_ALLOWED_IMAGE_GENERATIONS_AMOUNT,
  // ANONYMOUS_ALLOWED_IMAGE_GENERATIONS_RECORD_TTL,
  ANONYMOUS_ALLOWED_VOICE_GENERATIONS_AMOUNT,
  ANONYMOUS_ALLOWED_VOICE_GENERATIONS_RECORD_TTL,
  StepCostData,
} from "./ai.service";
import {
  AiGenerateRequest,
  AiGenerateResponse,
  AiGenerateStreamableResponse,
  AiSequenceClassificationDebertaRequest,
  AiSequenceClassificationBartRequest,
  AiSequenceClassificationResponse,
  AiGenerateImageRequest,
  AiGenerateImageResponse,
  AiRequestImageGenerationPriceRequest,
  AiRequestImageGenerationTagsResponse,
} from "./dto/ai-generate.dto";
import {
  RequireAccessToken,
  OptionalAccessToken,
} from "../shared/guards/require-access-token.guard";
import { User, UserType } from "../user/user.entity";
import { Throttle } from "@nestjs/throttler";
import {
  CounterMetric,
  HistogramMetric,
  PromService,
} from "@digikare/nestjs-prom";
import { validate, ValidationError } from "class-validator";
import {
  ImageGenerationModelSettingsData,
  isUserEligibleForGenerationModel,
  VoiceModelData,
  VoiceModels,
} from "../shared/generation-models";
import { AIModuleService } from "./ai-module.service";
import { EventSource } from "../lib/eventsource";
import {
  SubscriptionTierData,
  SubscriptionTiers,
} from "src/shared/subscription-tiers";
import { firstValueFrom } from "rxjs";

function errorsToString(errors: ValidationError[]): string {
  return errors
    .map((error) => {
      if (error.children.length > 0)
        return Object.values(error.children)
          .map((error) => Object.values(error.constraints).join(", "))
          .join(", ");

      return Object.values(error.constraints).join(", ");
    })
    .join("; ");
}

@ApiTags("/ai/")
@Controller("ai")
@ApiInternalServerErrorResponse({
  description: "An unknown error occured.",
  type: ApiError,
})
@Throttle(6, 5)
export class AIController {
  private readonly _generation_requests_counter: CounterMetric;
  private readonly _generation_counter: CounterMetric;
  private readonly _image_generation_requests_counter: CounterMetric;
  private readonly _image_generation_counter: CounterMetric;
  private readonly _generation_request_completion_duration_histogram: HistogramMetric;
  private readonly _generation_request_completion_duration_histogram_raw: HistogramMetric;
  private readonly _image_generation_request_completion_duration_histogram: HistogramMetric;

  constructor(
    private readonly aiService: AIService,
    private readonly httpService: HttpService,
    private readonly promService: PromService,
    private readonly aiModuleService: AIModuleService,
  ) {
    this._generation_requests_counter = this.promService.getCounter({
      name: "app_text_generation_request_count",
      help: "Amount of AI text generation requests",
      labelNames: ["model"],
    });

    this._generation_counter = this.promService.getCounter({
      name: "app_text_generation_count",
      help: "Amount of successful AI text generations",
      labelNames: ["model"],
    });

    this._image_generation_requests_counter = this.promService.getCounter({
      name: "app_image_generation_request_count",
      help: "Amount of AI image generation requests",
      labelNames: ["model"],
    });

    this._image_generation_counter = this.promService.getCounter({
      name: "app_image_generation_count",
      help: "Amount of successful AI image generations",
      labelNames: ["model"],
    });

    this._generation_request_completion_duration_histogram =
      this.promService.getHistogram({
        name: "app_text_generation_request_completion_duration",
        help: "A histogram of generation request durations, each multiplied by (priority / 10)",
        labelNames: ["model", "priority"],
        buckets: [0.005, 0.01, 0.025, 0.05, 0.1, 0.25, 0.5, 1, 2.5, 5, 10],
      });

    this._generation_request_completion_duration_histogram_raw =
      this.promService.getHistogram({
        name: "app_text_generation_request_completion_duration_raw",
        help: "A histogram of generation request durations",
        labelNames: ["model", "priority"],
        buckets: [0.005, 0.01, 0.025, 0.05, 0.1, 0.25, 0.5, 1, 2.5, 5, 10],
      });

    this._image_generation_request_completion_duration_histogram =
      this.promService.getHistogram({
        name: "app_text_generation_request_completion_duration",
        help: "A histogram of image generation request durations, each multiplied by",
        labelNames: ["model"],
        buckets: [0.005, 0.01, 0.025, 0.05, 0.1, 0.25, 0.5, 1, 2.5, 5, 10],
      });
  }

  @Post("generate")
  @Header("Cache-Control", "no-store")
  @ApiBearerAuth()
  @OptionalAccessToken()
  @ApiUnauthorizedResponse({
    description: "Access Token is incorrect.",
    type: ApiError,
  })
  @ApiResponse({
    status: HttpStatus.PAYMENT_REQUIRED,
    description: "An active subscription is required to access this endpoint.",
    type: ApiError,
  })
  @ApiConflictResponse({
    description: "A conflict error occured.",
    type: ApiError,
  })
  @ApiCreatedResponse({
    description: "The output has been successfully generated.",
    type: AiGenerateResponse,
  })
  @ApiBadRequestResponse({
    description: "A validation error occured.",
    type: ApiError,
  })
  async aiGenerate(@Req() req, @Body() aiGenerateRequest: AiGenerateRequest) {
    const user: User = req.user;

    if (!aiGenerateRequest)
      throw new HttpException("Incorrect body.", HttpStatus.BAD_REQUEST);

    if (user && user.userType == UserType.B2B)
      throw new HttpException(
        "Text generation is currently not available for B2B accounts.",
        HttpStatus.CONFLICT,
      );

    if (!isUserEligibleForGenerationModel(user, aiGenerateRequest.model))
      throw new HttpException(
        "Not eligible for this model.",
        HttpStatus.CONFLICT,
      );

    const subscriptionData = await this.aiService.checkSubscription(
      user,
      aiGenerateRequest.model,
      req.headers["cf-connecting-ip"] || req.ip,
    );

    (aiGenerateRequest.parameters as any).max_context_tokens =
      subscriptionData.contextTokens;
    (aiGenerateRequest.parameters as any).subscription_tier =
      subscriptionData.tier;

    const priority = subscriptionData.priority;

    this._generation_requests_counter.inc(
      {
        model: aiGenerateRequest.model,
      },
      isNaN(Math.ceil(priority / 10)) ? 1 : Math.ceil(priority / 10),
    );

    (aiGenerateRequest.parameters as any).priority = priority;

    if (aiGenerateRequest.parameters.use_string === undefined)
      aiGenerateRequest.parameters.use_string = false;

    const startTime = +new Date();
    const generationResult = await this.aiService.predictText(
      aiGenerateRequest,
    );
    const endTime = +new Date();

    const tookTime = (endTime - startTime) / 1000;

    this._generation_request_completion_duration_histogram.observe(
      { model: aiGenerateRequest.model, priority: priority },
      tookTime * (priority / 10),
    );
    this._generation_request_completion_duration_histogram_raw.observe(
      { model: aiGenerateRequest.model, priority: priority },
      tookTime,
    );

    this._generation_counter.inc({
      model: aiGenerateRequest.model,
    });

    return generationResult;
  }

  @Post("generate-stream")
  @Header("Cache-Control", "no-store")
  @ApiBearerAuth()
  @OptionalAccessToken()
  @ApiUnauthorizedResponse({
    description: "Access Token is incorrect.",
    type: ApiError,
  })
  @ApiResponse({
    status: HttpStatus.PAYMENT_REQUIRED,
    description: "An active subscription is required to access this endpoint.",
    type: ApiError,
  })
  @ApiConflictResponse({
    description: "A conflict error occured.",
    type: ApiError,
  })
  @ApiCreatedResponse({
    description:
      "The request has been accepted and the output is generating (SSE).",
    type: AiGenerateStreamableResponse,
  })
  @ApiBadRequestResponse({
    description: "A validation error occured.",
    type: ApiError,
  })
  async aiGenerateStreamable(
    @Req() req,
    @Res() res,
    @Body() aiGenerateRequest: AiGenerateRequest,
  ) {
    const user: User = req.user;

    if (!aiGenerateRequest)
      throw new HttpException("Incorrect body.", HttpStatus.BAD_REQUEST);

    if (user && user.userType == UserType.B2B)
      throw new HttpException(
        "Text generation is currently not available for B2B accounts.",
        HttpStatus.CONFLICT,
      );

    if (!isUserEligibleForGenerationModel(user, aiGenerateRequest.model))
      throw new HttpException(
        "Not eligible for this model.",
        HttpStatus.CONFLICT,
      );

    const subscriptionData = await this.aiService.checkSubscription(
      user,
      aiGenerateRequest.model,
      req.headers["cf-connecting-ip"] || req.ip,
    );

    (aiGenerateRequest.parameters as any).max_context_tokens =
      subscriptionData.contextTokens;
    (aiGenerateRequest.parameters as any).subscription_tier =
      subscriptionData.tier;

    const priority = subscriptionData.priority;

    this._generation_requests_counter.inc(
      {
        model: aiGenerateRequest.model,
      },
      isNaN(Math.ceil(priority / 10)) ? 1 : Math.ceil(priority / 10),
    );

    (aiGenerateRequest.parameters as any).priority = priority;

    if (aiGenerateRequest.parameters.use_string === undefined)
      aiGenerateRequest.parameters.use_string = false;

    const startTime = +new Date();

    res.raw.writeHead(HttpStatus.CREATED, {
      Connection: "keep-alive",
      "Content-Type": "text/event-stream",
      "Cache-Control": "no-cache",
      "Access-Control-Allow-Origin": req.headers["origin"] || "*",
      "Access-Control-Allow-Credentials": true,
    });

    await this.aiService.predictTextStream(
      aiGenerateRequest,
      (data: AiGenerateStreamableResponse) => {
        delete (data as any).uuid;

        if (!res.raw.finished) {
          res.raw.write(
            `event: newToken\nid: ${
              data.ptr >= 0 ? data.ptr : -1
            }\ndata:${JSON.stringify(data)}`,
          );

          res.raw.write("\n\n");
        }

        if (data.final) {
          if (!res.raw.finished) res.raw.end();

          const endTime = +new Date();

          const tookTime = (endTime - startTime) / 1000;

          this._generation_request_completion_duration_histogram.observe(
            { model: aiGenerateRequest.model },
            tookTime * (priority / 10),
          );
          this._generation_request_completion_duration_histogram_raw.observe(
            { model: aiGenerateRequest.model, priority: priority },
            tookTime,
          );

          this._generation_counter.inc({
            model: aiGenerateRequest.model,
          });
        }
      },
    );
  }

  @Post("generate-image/request-price")
  @Header("Cache-Control", "no-store")
  @ApiExcludeEndpoint(true)
  @HttpCode(HttpStatus.OK)
  @ApiOkResponse({
    type: StepCostData,
  })
  @Throttle(16, 2)
  async aiRequestImageGenerationPrice(
    @Body() aiPriceRequest: AiRequestImageGenerationPriceRequest,
  ) {
    const tierNumber = SubscriptionTiers[
      aiPriceRequest.tier
    ] as unknown as SubscriptionTiers;
    if (tierNumber <= SubscriptionTiers.NONE)
      throw new HttpException("Invalid tier.", HttpStatus.BAD_REQUEST);

    const perks = SubscriptionTierData[tierNumber];
    if (!perks)
      throw new HttpException(
        "Tier data not found.",
        HttpStatus.INTERNAL_SERVER_ERROR,
      );

    return this.aiService.calculateImageStepCost(aiPriceRequest.request, perks);
  }

  @Post("generate-image")
  @Header("Cache-Control", "no-store")
  @ApiBearerAuth()
  @RequireAccessToken()
  @ApiExcludeEndpoint(true)
  @ApiUnauthorizedResponse({
    description: "Access Token is incorrect.",
    type: ApiError,
  })
  @ApiResponse({
    status: HttpStatus.PAYMENT_REQUIRED,
    description: "An active subscription is required to access this endpoint.",
    type: ApiError,
  })
  @ApiConflictResponse({
    description: "A conflict error occured.",
    type: ApiError,
  })
  @ApiCreatedResponse({
    description:
      "The request has been accepted and the output is generating (SSE).",
    type: AiGenerateImageResponse,
  })
  @ApiBadRequestResponse({
    description: "A validation error occured.",
    type: ApiError,
  })
  async aiGenerateImage(
    @Req() req,
    @Res() res,
    @Body() aiGenerateRequest: AiGenerateImageRequest,
  ) {
    const user: User = req.user;

    if (!aiGenerateRequest)
      throw new HttpException("Incorrect body.", HttpStatus.BAD_REQUEST);

    if (user && user.userType == UserType.B2B)
      throw new HttpException(
        "Image generation is currently not available for B2B accounts.",
        HttpStatus.CONFLICT,
      );

    let perks = user.getSubscriptionPerks();
    if (!perks || !user.hasSubscription()) {
      perks = SubscriptionTierData[SubscriptionTiers.TABLET];
    }
    if (!perks.imageGeneration)
      throw new HttpException(
        "Subscription perk requirement not met.",
        HttpStatus.UNAUTHORIZED,
      );

    const costData = this.aiService.calculateImageStepCost(
      aiGenerateRequest,
      perks,
    );

    const numSamples = +(
      aiGenerateRequest.parameters["n_samples"] ?? 1
    ) as number;

    if (
      costData.requestEligibleForUnlimitedGeneration ||
      user.userType == UserType.SERVICE
    )
      costData.costPerPrompt = 0;

    const requiredStepAmount = costData.costPerPrompt * costData.numPrompts;

    // if (user && user.hasSubscription()) {
    const userSteps =
      user.availableModuleTrainingSteps + user.purchasedModuleTrainingSteps;

    if (userSteps < requiredStepAmount)
      throw new HttpException(
        `Not enough Anlas (need ${requiredStepAmount}, have ${userSteps}).`,
        HttpStatus.PAYMENT_REQUIRED,
      );
    /*} else {
      throw new HttpException("Account required.", HttpStatus.UNAUTHORIZED);

      if (
        !(await this.aiService.checkIpBasedLimit(
          "img:" + (req.headers["cf-connecting-ip"] || req.ip),
          ANONYMOUS_ALLOWED_IMAGE_GENERATIONS_AMOUNT,
          ANONYMOUS_ALLOWED_IMAGE_GENERATIONS_RECORD_TTL,
        ))
      )
        throw new HttpException(
          "Anonymous quota reached.",
          HttpStatus.PAYMENT_REQUIRED,
        );

      /// TODO if (requiredStepAmount > someTrialNumber) throw new HttpException("Action is too expensive.", HttpStatus.PAYMENT_REQUIRED);
    }*/

    this._image_generation_requests_counter.inc(
      {
        model: aiGenerateRequest.model,
      },
      1,
    );

    const startTime = +new Date();

    const modelConfig =
      ImageGenerationModelSettingsData[aiGenerateRequest.model];

    /*const sourcesToCreate = modelConfig.getAmountOfSourcesToCreate
      ? modelConfig.getAmountOfSourcesToCreate(aiGenerateRequest)
      : 1;*/

    // capped at 1 since advanced parameter is not used anymore
    const sourcesToCreate = 1;

    const sources = [];
    const sourcePromises: Promise<void>[] = [];

    let writtenHeaders = false;
    let amountSourcesEnded = 0;
    let amountImagesSent = 0;

    const end = ((success) => {
      for (const source of sources) {
        source.removeAllListeners("newImage");
        source.removeAllListeners("error");
        source.removeAllListeners("end");

        source.close();
      }

      res.raw.end();

      const endTime = +new Date();
      const tookTime = (endTime - startTime) / 1000;

      this._image_generation_request_completion_duration_histogram.observe(
        { model: aiGenerateRequest.model },
        tookTime,
      );

      this._image_generation_counter.inc({
        model: aiGenerateRequest.model,
      });

      if (user && (success || amountImagesSent > 0))
        this.aiModuleService.tryChargeTrainingSteps(
          user,
          success
            ? requiredStepAmount
            : costData.costPerPrompt *
                (amountImagesSent - costData.freePrompts),
          "image_generation",
          JSON.stringify({
            rs: requiredStepAmount,
            p: costData.costPerPrompt,
            se: amountImagesSent,
          }),
        );
    }).bind(this);

    const imageIds = [];
    function getImageId(proxyId) {
      const index = imageIds.indexOf(proxyId);
      if (index !== -1) return index + 1;
      return imageIds.push(proxyId);
    }

    function createSource(i) {
      const seed =
        aiGenerateRequest.parameters.seed !== undefined
          ? (+aiGenerateRequest.parameters.seed ||
              Math.floor(Math.random() * 1000)) + i
          : undefined;

      const stream = new EventSource(modelConfig.url, {
        headers: {
          "Content-Type": "application/json",
          Authorization: "Bearer " + process.env.VOICE_TOKEN,
        },
        body: JSON.stringify({
          ...aiGenerateRequest.parameters,
          prompt: aiGenerateRequest.input,
          tier: user.subscriptionTier,
          seed,
          width: modelConfig.customResolutionSupported
            ? aiGenerateRequest.parameters["width"] ?? modelConfig.defaultWidth
            : modelConfig.defaultWidth,
          height: modelConfig.customResolutionSupported
            ? aiGenerateRequest.parameters["height"] ??
              modelConfig.defaultHeight
            : modelConfig.defaultHeight,
        }),
        method: "POST",
      });

      stream.addEventListener("newImage", (e) => {
        if (!writtenHeaders) {
          res.raw.writeHead(HttpStatus.CREATED, {
            Connection: "keep-alive",
            "Cache-Control": "no-cache",
            "Access-Control-Allow-Origin": req.headers["origin"] || "*",
            "Access-Control-Allow-Credentials": true,
            "Content-Type": "text/event-stream",
          });
          res.raw.flushHeaders();

          writtenHeaders = true;
        }

        const sessionId = e.lastEventId;
        const nodeId = e.streamId;
        const str = nodeId + "_image_" + sessionId;
        const id = getImageId(str);

        res.raw.write(`event: newImage\nid: ${id}\ndata:${e.data}\n\n`);
        amountImagesSent++;
      });

      stream.addEventListener("error", (err) => {
        console.log("error", err);
        let errData = "unknown error";
        if (err.data) {
          errData = err.data;
        } else if (err.statusCode) {
          errData = "failed to contact the generation node";
        } else if (err.body) {
          errData = `connection error (${err.status || "unknown"})`;
        }

        if (writtenHeaders) {
          res.raw.write(`event: error\nid: 1\ndata:${errData}\n\n`);
        } else {
          res.raw.writeHead(HttpStatus.BAD_REQUEST, {
            Connection: "keep-alive",
            "Cache-Control": "no-cache",
            "Access-Control-Allow-Origin": req.headers["origin"] || "*",
            "Access-Control-Allow-Credentials": true,
            "Content-Type": "application/json",
          });
          res.raw.write(
            JSON.stringify({
              error: errData,
            }),
          );

          writtenHeaders = true;
        }

        setTimeout(() => end(false), 500);
      });

      stream.addEventListener("end", () => {
        amountSourcesEnded++;
        if (amountSourcesEnded == sourcesToCreate)
          setTimeout(() => end(amountImagesSent == numSamples), 500);
      });

      sources.push(stream);
      sourcePromises.push(
        new Promise((resolve) => {
          stream.addEventListener("end", resolve);
          stream.addEventListener("closed", resolve);
        }),
      );
    }

    for (let i = 0; i < sourcesToCreate; i++) createSource(i);
    await Promise.all(sourcePromises);
  }

  @Post("classify")
  @Header("Cache-Control", "no-store")
  @ApiBearerAuth()
  @RequireAccessToken()
  @ApiUnauthorizedResponse({
    description: "Access Token is incorrect.",
    type: ApiError,
  })
  @ApiResponse({
    status: HttpStatus.PAYMENT_REQUIRED,
    description: "An active subscription is required to access this endpoint.",
    type: ApiError,
  })
  @ApiConflictResponse({
    description: "A conflict error occured.",
    type: ApiError,
  })
  @ApiCreatedResponse({
    description: "The output has been successfully generated.",
    type: AiSequenceClassificationResponse,
  })
  @ApiBadRequestResponse({
    description: "A validation error occured.",
    type: ApiError,
  })
  async classify(@Req() req, @Body() aiClassificationRequest: any) {
    const user: User = req.user;

    if (!aiClassificationRequest)
      throw new HttpException("Incorrect body.", HttpStatus.BAD_REQUEST);

    if (user.userType == UserType.RETAIL)
      throw new HttpException(
        "Classification is currently not available for RETAIL accounts.",
        HttpStatus.CONFLICT,
      );

    let allSentencesLength = 0;

    console.log(aiClassificationRequest);

    if (aiClassificationRequest.model == "debertaxxl") {
      const testCls = new AiSequenceClassificationDebertaRequest();
      for (const key in aiClassificationRequest)
        testCls[key] = aiClassificationRequest[key];

      const errors = await validate(testCls, {
        forbidUnknownValues: true,
        whitelist: true,
      });

      if (errors.length > 0)
        throw new HttpException(errorsToString(errors), HttpStatus.BAD_REQUEST);

      for (const sentences of aiClassificationRequest.sentences)
        for (const sentence of sentences) allSentencesLength += sentence.length;
    } else if (aiClassificationRequest.model == "bartlarge") {
      const testCls = new AiSequenceClassificationBartRequest();
      for (const key in aiClassificationRequest)
        testCls[key] = aiClassificationRequest[key];

      const errors = await validate(testCls, {
        forbidUnknownValues: true,
        whitelist: true,
      });

      if (errors.length > 0)
        throw new HttpException(errorsToString(errors), HttpStatus.BAD_REQUEST);

      for (const sentence of aiClassificationRequest.sequence)
        allSentencesLength += sentence.length;
    } else
      throw new HttpException(
        "Incorrect model type provided.",
        HttpStatus.CONFLICT,
      );

    const subscriptionData = await this.aiService.checkSubscription(
      user,
      null,
      allSentencesLength,
    );

    return await this.aiService.classifyText(
      aiClassificationRequest,
      subscriptionData.priority,
    );
  }

  @Get("generate-image/suggest-tags")
  @ApiBearerAuth()
  // @ApiCookieAuth()
  @RequireAccessToken()
  @ApiUnauthorizedResponse({
    description: "Access Token is incorrect.",
    type: ApiError,
  })
  @ApiOkResponse({
    type: AiRequestImageGenerationTagsResponse,
  })
  async generateImageTags(
    @Req() req,
    @Res() res,
    // @Body() generateImageTagsRequest: AiRequestImageGenerationTagsRequest,
    @Query("model") model: string,
    @Query("prompt") prompt: string,
  ) {
    const user: User = req.user;

    if (!model)
      throw new HttpException(
        "Missing model parameter.",
        HttpStatus.BAD_REQUEST,
      );

    if (!ImageGenerationModelSettingsData[model])
      throw new HttpException(
        "Invalid model parameter.",
        HttpStatus.BAD_REQUEST,
      );

    if (!prompt)
      throw new HttpException(
        "Missing prompt parameter.",
        HttpStatus.BAD_REQUEST,
      );

    let perks = user.getSubscriptionPerks();
    if (!perks || !user.hasSubscription()) {
      perks = SubscriptionTierData[SubscriptionTiers.TABLET];
    }
    if (!perks.imageGeneration)
      throw new HttpException(
        "Subscription perk requirement not met.",
        HttpStatus.UNAUTHORIZED,
      );

    const modelData = ImageGenerationModelSettingsData[model];

    if (!modelData.tagsUrl)
      throw new HttpException(
        "Unable to contact generation node.",
        HttpStatus.INTERNAL_SERVER_ERROR,
      );

    const response = await firstValueFrom(
      this.httpService.post(
        modelData.tagsUrl,
        JSON.stringify({ model: modelData.tagsModel, prompt }),
        {
          responseType: "stream",
          headers: {
            Authorization: "Bearer " + process.env.VOICE_TOKEN,
            "Content-Type": "application/json",
          },
        },
      ),
    );

    res.raw.writeHead(HttpStatus.OK, {
      Connection: "keep-alive",
      "Cache-Control":
        response.status == HttpStatus.OK
          ? "public, max-age=3600"
          : "no-cache, no-store",
      "Access-Control-Allow-Origin": req.headers["origin"] || "*",
      "Access-Control-Allow-Credentials": true,
      "Content-Type": response.headers["content-type"] || "application/json",
    });
    res.raw.flushHeaders();

    response.data.pipe(res.raw);

    await new Promise((resolve, reject) => {
      response.data.on("end", resolve);
      response.data.on("error", reject);
    });
  }

  @Get("generate-voice")
  @Header("Cache-Control", "no-store")
  @ApiBearerAuth()
  // @ApiCookieAuth()
  // @RequireAccessToken()
  @OptionalAccessToken()
  @ApiUnauthorizedResponse({
    description: "Access Token is incorrect.",
    type: ApiError,
  })
  async generateVoice(
    @Req() req,
    @Res() res,
    @Query("text") text: string,
    @Query("seed") seed: string,
    @Query("voice", ParseIntPipe) voice: number,
    @Query("opus", ParseBoolPipe) opus: boolean,
    @Query("version") version: VoiceModels,
  ) {
    const user: User = req.user;

    if (!text)
      throw new HttpException(
        "Missing text parameter.",
        HttpStatus.BAD_REQUEST,
      );

    if (user && user.hasSubscription()) {
      if (!user.getSubscriptionPerks().voiceGeneration)
        throw new HttpException(
          "Subscription perk requirement not met.",
          HttpStatus.UNAUTHORIZED,
        );
    } else {
      if (
        !(await this.aiService.checkIpBasedLimit(
          "vce:" + (req.headers["cf-connecting-ip"] || req.ip),
          ANONYMOUS_ALLOWED_VOICE_GENERATIONS_AMOUNT,
          ANONYMOUS_ALLOWED_VOICE_GENERATIONS_RECORD_TTL,
        ))
      )
        throw new HttpException(
          "Anonymous quota reached.",
          HttpStatus.PAYMENT_REQUIRED,
        );
    }

    if (voice < -1)
      throw new HttpException(
        "Voice parameter must not be lower than -1.",
        HttpStatus.BAD_REQUEST,
      );

    if (voice == -1 && !seed)
      throw new HttpException(
        "Voice -1 requires valid seed parameter.",
        HttpStatus.BAD_REQUEST,
      );

    if (!version) version = VoiceModels["v1"];
    else {
      if (!VoiceModels[version])
        throw new HttpException("Invalid TTS version.", HttpStatus.BAD_REQUEST);
    }

    const ttsData = VoiceModelData[version];

    if (!ttsData.baseUri)
      throw new HttpException(
        "Unable to contact generation node.",
        HttpStatus.INTERNAL_SERVER_ERROR,
      );

    const response = await firstValueFrom(
      this.httpService.get(
        ttsData.baseUri +
          `/audio?text=${encodeURIComponent(
            text,
          )}&sid=${voice}&seed=${encodeURIComponent(seed)}&opus=${opus}`,
        {
          responseType: "stream",
          headers: {
            Authorization: "Bearer " + process.env.VOICE_TOKEN,
          },
        },
      ),
    );

    res.raw.writeHead(HttpStatus.OK, {
      Connection: "keep-alive",
      "Cache-Control": "no-cache",
      "Access-Control-Allow-Origin": req.headers["origin"] || "*",
      "Access-Control-Allow-Credentials": true,
      "Content-Type": response.headers["content-type"] || "audio/mpeg",
    });
    res.raw.flushHeaders();

    response.data.pipe(res.raw);

    await new Promise((resolve, reject) => {
      response.data.on("end", resolve);
      response.data.on("error", reject);
    });
  }
}
