package gobot

import (
	"fmt"
	"github.com/bwmarrin/discordgo"
	"gopkg.in/yaml.v3"
	"log"
	"nai-gobot/includes"
	"os"
)

type Command struct {
	Name     string
	AllowDMs bool
	Command  *discordgo.ApplicationCommand
}

type Config struct {
	ImageModels     map[string]includes.ImageModel        `yaml:"ImageModels"`
	DefaultModel    string                                `yaml:"DefaultModel"`
	DefaultSampler  string                                `yaml:"DefaultSampler"`
	DefaultSteps    int                                   `yaml:"DefaultSteps"`
	DefaultHeight   int                                   `yaml:"DefaultHeight"`
	DefaultWidth    int                                   `yaml:"DefaultWidth"`
	DefaultNumGen   int                                   `yaml:"DefaultNumGen"`
	DefaultScale    int                                   `yaml:"DefaultScale"`
	DefaultModule   string                                `yaml:"DefaultModule"`
	DefaultStrength float64                               `yaml:"DefaultStrength"`
	DefaultNoise    float64                               `yaml:"DefaultNoise"`
	LogRequests     bool                                  `yaml:"LogRequests"`
	AllowConfig     bool                                  `yaml:"AllowConfig"`
	Buttons         map[string]discordgo.MessageComponent `yaml:"-"`
	ValidSamplers   []string                              `yaml:"ValidSamplers"`
	ImageModules    []string                              `yaml:"ImageModules"`
	Commands        []*Command                            `yaml:"-"`
	AllowDM         bool                                  `yaml:"AllowDM"`
}

func MakeConfig(confPath string) *Config {
	conf := new(Config)
	reader, err := os.Open(confPath)
	if err != nil {
		log.Fatalf("Error opening config file: %v", err)
	}
	yamlDecoder := yaml.NewDecoder(reader)
	err = yamlDecoder.Decode(conf)

	if err != nil {
		log.Fatalf("Error decoding config file: %v", err)
	}
	conf.Buttons = map[string]discordgo.MessageComponent{
		"enhance": discordgo.Button{
			Label:    "Enhance ✨",
			Style:    discordgo.PrimaryButton,
			CustomID: "enhance",
		},
		"strength": discordgo.SelectMenu{
			Placeholder: "Select strength",
			CustomID:    "strength",
			Options:     *GenerateDropdownOptions(0.0, 1.0, 0.1, conf.DefaultStrength),
		},
		"noise": discordgo.SelectMenu{
			Placeholder: "Select noise",
			CustomID:    "noise",
			Options:     *GenerateDropdownOptions(0.0, 1.0, 0.1, conf.DefaultNoise),
		},
		"none": nil,
	}
	initializeCommands(conf)
	return conf
}

func (c Config) Update() error {
	initializeCommands(&c)
	f, err := os.Create("./config/devel.yaml")
	if err != nil {
		return fmt.Errorf("error creating config file: %v", err)
	}
	defer func(f *os.File) {
		err := f.Close()
		if err != nil {
			log.Printf("Error closing config file: %v", err)
		}
	}(f)
	yamlEncoder := yaml.NewEncoder(f)
	err = yamlEncoder.Encode(c)
	if err != nil {
		return fmt.Errorf("error encoding config file: %v", err)
	}
	return nil
}

func initializeCommands(conf *Config) {
	sampleChoices := make([]*discordgo.ApplicationCommandOptionChoice, len(conf.ValidSamplers))
	for i, sampler := range conf.ValidSamplers {
		sampleChoices[i] = &discordgo.ApplicationCommandOptionChoice{
			Name:  sampler,
			Value: sampler,
		}
	}
	moduleChoices := make([]*discordgo.ApplicationCommandOptionChoice, len(conf.ImageModules))
	for i, module := range conf.ImageModules {
		moduleChoices[i] = &discordgo.ApplicationCommandOptionChoice{
			Name:  module,
			Value: module,
		}
	}
	modelChoices := make([]*discordgo.ApplicationCommandOptionChoice, 0)
	for modelKey, model := range conf.ImageModels {
		modelChoices = append(modelChoices, &discordgo.ApplicationCommandOptionChoice{
			Name:  model.Name,
			Value: modelKey,
		})
	}
	var drawCommand = Command{
		Name:     "Draw",
		AllowDMs: true,
		Command: &discordgo.ApplicationCommand{
			Name:        "draw",
			Description: "Draws an image from a prompt",
			Options: []*discordgo.ApplicationCommandOption{
				{
					Type:        discordgo.ApplicationCommandOptionString,
					Name:        "prompt",
					Description: "The prompt to draw from",
					Required:    true,
				},
				{
					Type:        discordgo.ApplicationCommandOptionString,
					Name:        "model",
					Description: "The model to use",
					Required:    false,
					Choices:     modelChoices,
				},
				{
					Type:        discordgo.ApplicationCommandOptionString,
					Name:        "sampler",
					Description: "The sampling method to use",
					Required:    false,
					Choices:     sampleChoices,
				},
				{
					Type:        discordgo.ApplicationCommandOptionInteger,
					Name:        "steps",
					Description: "The number of steps to take",
					Required:    false,
					MinValue:    F64Ptr(float64(1)),
					MaxValue:    float64(50),
				},
				{
					Type:        discordgo.ApplicationCommandOptionInteger,
					Name:        "height",
					Description: "The height of the image",
					Required:    false,
					MinValue:    F64Ptr(float64(128)),
					MaxValue:    float64(1024),
				},
				{
					Type:        discordgo.ApplicationCommandOptionInteger,
					Name:        "width",
					Description: "The width of the image",
					Required:    false,
					MinValue:    F64Ptr(float64(128)),
					MaxValue:    float64(1024),
				},
				{
					Type:        discordgo.ApplicationCommandOptionInteger,
					Name:        "scale",
					Description: "The scale of the image",
					Required:    false,
					MinValue:    F64Ptr(float64(1)),
					MaxValue:    float64(36),
				},
				{
					Type:        discordgo.ApplicationCommandOptionInteger,
					Name:        "num_gen",
					Description: "The number of generations to draw",
					Required:    false,
					MinValue:    F64Ptr(float64(1)),
					MaxValue:    float64(10),
				},
				{
					Type:        discordgo.ApplicationCommandOptionInteger,
					Name:        "seed",
					Description: "The seed to use for the random number generator",
					Required:    false,
					MinValue:    F64Ptr(float64(0)),
					MaxValue:    float64(4294967295),
				},
				{
					Type:        discordgo.ApplicationCommandOptionBoolean,
					Name:        "advanced",
					Description: "Whether to use advanced mode",
					Required:    false,
				},
				{
					Type:        discordgo.ApplicationCommandOptionNumber,
					Name:        "strength",
					Description: "The strength of the advanced mode",
					Required:    false,
					MinValue:    F64Ptr(float64(0)),
					MaxValue:    float64(1),
				},
				{
					Type:        discordgo.ApplicationCommandOptionNumber,
					Name:        "noise",
					Description: "The noise of the advanced mode",
					Required:    false,
					MinValue:    F64Ptr(float64(0)),
					MaxValue:    float64(1),
				},
				{
					Type:        discordgo.ApplicationCommandOptionString,
					Name:        "module",
					Description: "The module to use",
					Required:    false,
					Choices:     moduleChoices,
				},
				{
					Type:        discordgo.ApplicationCommandOptionString,
					Name:        "uc",
					Description: "The negative prompt",
					Required:    false,
				},
			},
		},
	}

	var confCommand = Command{
		Name:     "Configure",
		AllowDMs: false,
		Command: &discordgo.ApplicationCommand{
			Name:        "set_gobot_conf",
			Description: "Configures the bot",
			Options: []*discordgo.ApplicationCommandOption{
				{
					Type:        discordgo.ApplicationCommandOptionString,
					Name:        "api_url",
					Description: "The API endpoint to set",
					Required:    false,
				},
				{
					Type:        discordgo.ApplicationCommandOptionString,
					Name:        "default_sampler",
					Description: "The default sampler to set",
					Required:    false,
					Choices:     sampleChoices,
				},
				{
					Type:        discordgo.ApplicationCommandOptionInteger,
					Name:        "default_steps",
					Description: "The default number of steps to set",
					Required:    false,
					MinValue:    F64Ptr(float64(1)),
					MaxValue:    float64(50),
				},
				{
					Type:        discordgo.ApplicationCommandOptionInteger,
					Name:        "default_height",
					Description: "The default height to set",
					Required:    false,
					MinValue:    F64Ptr(float64(128)),
					MaxValue:    float64(1024),
				},
				{
					Type:        discordgo.ApplicationCommandOptionInteger,
					Name:        "default_width",
					Description: "The default width to set",
					Required:    false,
					MinValue:    F64Ptr(float64(128)),
					MaxValue:    float64(1024),
				},
				{
					Type:        discordgo.ApplicationCommandOptionString,
					Name:        "default_module",
					Description: "The default module to set",
					Required:    false,
				},
				{
					Type:        discordgo.ApplicationCommandOptionBoolean,
					Name:        "enable_logging",
					Description: "Whether to enable logging",
					Required:    false,
				},
			},
		},
	}
	var showConfCommand = Command{
		Name:     "Show Configuration",
		AllowDMs: false,
		Command: &discordgo.ApplicationCommand{
			Name:        "show_gobot_conf",
			Description: "Shows the current configuration",
		},
	}
	var resetConfCommand = Command{
		Name:     "Reset Configuration",
		AllowDMs: false,
		Command: &discordgo.ApplicationCommand{
			Name:        "reset_gobot_conf",
			Description: "Resets the configuration to the default",
		},
	}
	commands := []*Command{
		&drawCommand,
	}
	if conf.AllowConfig {
		commands = append(commands, &confCommand, &showConfCommand, &resetConfCommand)
	}
	conf.Commands = commands
}
