package gobot

import (
	"encoding/json"
	"fmt"
	"github.com/bwmarrin/discordgo"
	"log"
	"nai-gobot/includes"
)

func RespondCommand(s *discordgo.Session, i *discordgo.InteractionCreate) {
	err := s.InteractionRespond(i.Interaction, &discordgo.InteractionResponse{
		Type: discordgo.InteractionResponseDeferredChannelMessageWithSource,
	})
	if err != nil {
		log.Printf("Error responding to interaction: %v", err)
	}
}

func CommandHandler(conf *Config, s *discordgo.Session, i *discordgo.InteractionCreate,
	commandData *discordgo.ApplicationCommandInteractionData, guildMembers *[]*discordgo.User) error {
	isAllowed := false
	if i.User != nil {
		for _, member := range *guildMembers {
			if member.ID == i.User.ID {
				isAllowed = true
				break
			}
		}
	}
	if i.Member != nil {
		isAllowed = true
	}
	if !isAllowed {
		return fmt.Errorf("not allowed to use this bot")
	}
	switch commandData.Name {
	case "draw":
		RespondCommand(s, i)
		genReq := MakeImageGenerationRequest(conf, commandData)
		apiObject := includes.APIEndpoint{
			URL:  genReq.Model.APIEndpointUrl,
			Type: "imagegen",
		}
		return SendImage(conf, s, i, genReq, &apiObject, genReq.Model.AdvancedSupport)
	case "set_gobot_conf":
		RespondCommand(s, i)
		return ConfigureBot(conf, s, i, commandData)
	case "show_gobot_conf":
		RespondCommand(s, i)
		retStr, err := json.MarshalIndent(conf, "", "  ")
		if err != nil {
			return err
		}
		_, err = s.FollowupMessageCreate(i.Interaction, true, &discordgo.WebhookParams{
			Content: "```json\n" + string(retStr) + "\n```",
		})
		return err
	case "reset_gobot_conf":
		RespondCommand(s, i)
		conf = new(Config)
		conf = MakeConfig("./config/default.yaml")
		err := conf.Update()
		if err == nil {
			_, err = s.FollowupMessageCreate(i.Interaction, true, &discordgo.WebhookParams{
				Content: "Configuration reset!",
			})
		}
		return err
	}
	return nil
}

func ConfigureBot(conf *Config, s *discordgo.Session, i *discordgo.InteractionCreate, commandData *discordgo.ApplicationCommandInteractionData) error {
	for _, command := range commandData.Options {
		switch command.Name {
		case "default_sampler":
			conf.DefaultSampler = command.StringValue()
		case "default_steps":
			conf.DefaultSteps = int(command.IntValue())
		case "default_height":
			conf.DefaultHeight = int(command.IntValue())
		case "default_width":
			conf.DefaultWidth = int(command.IntValue())
		case "default_module":
			conf.DefaultModule = command.StringValue()
		case "enable_logging":
			conf.LogRequests = command.BoolValue()
		}
	}
	err := conf.Update()
	if err == nil {
		_, err = s.FollowupMessageCreate(i.Interaction, true, &discordgo.WebhookParams{
			Content: "Configuration updated!",
		})
	}
	return err
}
