import { NextPage } from 'next'
import { Block, ClampText, Column, Container, Fill, Grid, MaskImage, MOBILE_SMALL, PrimaryContent, Row, RowFull, SecondaryContent, Space, TABLET_LARGE, TABLET_SMALL as TABLET_SMALL } from '../components/common';
import styled from 'styled-components';
import NaiLogo from '../assets/icons/nai.svg';
import ArrowRightSmall from '../assets/icons/arrow-right-small.svg';
import { useEffect, useState } from 'react';
import { AnimatePresence, motion, useAnimation, Variants } from "framer-motion"
import ArrowRight from '../assets/icons/arrow-right.svg';
import ArrowLeft from '../assets/icons/arrow-left.svg';
import { Button } from '../components/button';
import Link from 'next/link';




const ListContent = styled(Block)`
  background-color: #E4E4F5;
  border-bottom: 1px solid #1A1A1A;
  border-top: 1px solid #1A1A1A;
  height: 453px;
  @media (max-width: ${TABLET_SMALL}px) {
      height: 833px;
  }
`


const Project = (props: { logo: string, url: string, desc: string }) => {
  return <div style={{ maxWidth: 440 }}>
    <a href={props.url} target='_blank' rel="noreferrer">
      <RowFull>
        <MaskImage style={{ height: 28, width: 144 }} src={props.logo} color='#5200FF' />
        <Row>
          <span className='bold altColor' style={{ fontSize: '18px' }}>Link</span>
          <Space width={10} />
          <MaskImage style={{ height: 8, width: 5 }} src={ArrowRightSmall.src} color='#5200FF' />
        </Row>
      </RowFull>
    </a>
    <Space height={20} />
    <p>{props.desc}</p>
  </div>
}

interface BlogPost {
  title: string,
  url: string,
  content: string
  date: number,
}

const BlogDisplay = (props: { blog: BlogPost | undefined }) => {
  return <Column style={{ height: 190 }}>
    {props.blog && <>
      <a href={props.blog.url} target='_blank' rel="noreferrer" >
        <ClampText lines={2}>
          <h3 className={'bold'}>{props.blog.title}</h3>
        </ClampText>
      </a>
      <Space height={10} />
      <ClampText lines={4}>{props.blog.content.split('\n').map((text, i)=>{
        return <span key={i}>{text}<br/></span>
      })}</ClampText>
      <Fill />
      {props.blog.date > 0 ? <p style={{ color: 'rgba(26, 26, 26, 0.4)' }}>{MONTHS_SHORT[new Date(props.blog.date).getMonth()]} {new Date(props.blog.date).getDate()}</p> : ''}

    </>

    }
  </Column>
}


const MONTHS_SHORT = ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec'];

const variants: Variants = {
  enter: (direction: number) => {
    return {
      x: direction > 0 ? 1000 : -1000,
      opacity: 0
    };
  },
  center: {
    zIndex: 1,
    x: 0,
    opacity: 1
  },
  exit: (direction: number) => {
    return {
      zIndex: 0,
      x: direction < 0 ? 1000 : -1000,
      opacity: 0
    };
  }
};

const swipeConfidenceThreshold = 10000;
const swipePower = (offset: number, velocity: number) => {
  return Math.abs(offset) * velocity;
};

function htmlEncode(s:string )
{
  var el = document.createElement("div");
  el.innerHTML = s;
  return el.textContent;
}



const Home: NextPage = () => {
  const controls = useAnimation()
  let [blogPosts, setBlogPosts] = useState([] as (BlogPost | undefined)[]);

  useEffect(() => {
    // Simulate fetching data from a server
    fetch('https://api.rss2json.com/v1/api.json?rss_url=https://medium.com/feed/@novelai').then(async (res) => {
      let json = await res.json()
      let blogPosts = json.items.map((item:any) => {
        // convert html description to text
        let elem = document.createElement('div');
        elem.innerHTML = item.description;
        let text = '';
        for (const e of elem.childNodes) {
          if (e.nodeName === 'P' && e.textContent) {
            text += e.textContent?.replace(/\s+/g, ' ').trim() + '\n';
          }
          if (e.nodeName === 'BR') {
            text += '\n';
          }
        }

        return {
          title: htmlEncode(item.title),
          url: item.link,
          content: text.slice(0, 300),
          date: item.pubDate ? new Date(item.pubDate).getTime() : 0
        }
      });
      console.log({ blogPosts })
      setBlogPosts(blogPosts);
    })

  }, []);


  const [[page, direction], setPage] = useState([0, 0]);
  const paginate = (newDirection: number, drag?: boolean) => {
    let newPage = page + newDirection;
    if (newPage < 0 || newPage >= blogPosts.length) {
      newPage = page;
      if (!drag)
        controls.start({
          x: [-5, 5, -3, 3, 0],
          transition: { type: "spring", bounce: 0.8, duration: 0.4 },
        })
    }
    setPage([newPage, newDirection]);
  };



  return (
    <>
      <Grid columns={['5fr', '4fr']} breakpoints={[{ width: TABLET_LARGE, columns: ['1fr'] }]}>
        <PrimaryContent >
          <Column reverse>
            <h1 className={'altFont altColor'} style={{ maxWidth: 560 }}>We’re here to make ripples in the AI world.</h1>
            <Space height={50} />
            <h2 className={'altFont'}>Welcome.</h2>
          </Column>
          <Space height={20} />
          <p style={{ fontSize: '18px', lineHeight: '24px' }}>Advancing the fields of text-based<br />and image AI the best we can.</p>
          <Space height={288} />
        </PrimaryContent>
        <SecondaryContent>
          <h2 className={'altFont'}>Our Active Projects</h2>
          <Space height={50} />
          <Grid columns={1} gap={40} breakpoints={[{ width: TABLET_LARGE, columns: 3, gap: 50 }, { width: TABLET_SMALL, columns: 1 }]}>
            <Project logo={NaiLogo.src} url='https://novelai.net' desc={`NovelAI is a monthly subscription service for AI-assisted authorship, storytelling, virtual companionship, or simply a GPT-J powered sandbox for your imagination.`} />
            <div>
              <h3 className='altColor bold' >More Coming Soon</h3>
              <Space height={21} />
              <div>More exciting stuff coming soon!</div>
            </div>
          </Grid>
        </SecondaryContent>
      </Grid>
      <ListContent>
        <Space height={60} />
        <RowFull>
          <h2>
            Latest Blog Posts
          </h2>
          <Grid columns={2} gap={20}>
            <Button onClick={() => paginate(-3)} style={{ padding: '9px 17px 9px 14px' }}>
              <MaskImage style={{ height: 18, width: 9 }} src={ArrowLeft.src} color='#1A1A1A' />
            </Button>
            <Button onClick={() => paginate(3)} style={{ padding: '9px 14px 9px 17px' }}>
              <MaskImage style={{ height: 18, width: 9 }} src={ArrowRight.src} color='#1A1A1A' />
            </Button>
          </Grid>
        </RowFull>
        <Space height={50} />
        <motion.div animate={controls} >
          <div style={{ position: 'relative' }}>
            <AnimatePresence initial={false} custom={direction} >
              <motion.div
                key={page}
                custom={direction}
                variants={variants}
                initial="enter"
                animate="center"
                exit="exit"
                transition={{
                  x: { type: "spring", stiffness: 300, damping: 30 },
                  opacity: { duration: 0.3 }
                }}
                drag="x"
                dragConstraints={{ left: 0, right: 0 }}
                dragElastic={1}
                onDragEnd={(e, { offset, velocity }) => {
                  const swipe = swipePower(offset.x, velocity.x);

                  if (swipe < -swipeConfidenceThreshold) {
                    paginate(3, true);
                  } else if (swipe > swipeConfidenceThreshold) {
                    paginate(-3, true);
                  }
                }}

              >
                <Grid columns={3} breakpoints={[{ width: TABLET_SMALL, columns: 1, gap: 30 }]} style={{ position: 'absolute', width: '100%' }} gap={50}>
                  {blogPosts.slice(page, page + 3).map((post, i) => {
                    return <BlogDisplay
                      key={i}
                      blog={post}
                    />
                  })}
                  {new Array(3 - blogPosts.slice(page, page + 3).length).fill(0).map((_, i) => {
                    return <BlogDisplay
                      key={i}
                      blog={{
                        title: 'Loading...',
                        url: '',
                        date: 0,
                        content: 'Loading...',
                      }}
                    />
                  })}
                </Grid>
              </motion.div>
            </AnimatePresence>
          </div>
        </motion.div>
      </ListContent>
    </>
  )
}

export default Home
