import { CSSProperties } from 'react';
import styled from 'styled-components';
import CircleBackground from '../assets/backgrounds/circles.svg';


export const DESKTOP_LARGE = 1920
export const DESKTOP_SMALL = 1800
export const TABLET_LARGE = 1440
export const TABLET_SMALL = 1050
export const MOBILE_LARGE = 700
export const MOBILE_SMALL = 425

export const Container = styled.div`
    min-height: 100vh;
    //background-color: #E4E4F5;
    //background-image: url(${CircleBackground.src});
    background-repeat: no-repeat;
    background-position: left -1361px top -1605px;
    display: flex;
    flex-direction: column;
    overflow-x: hidden;
    
    @media (max-width: ${DESKTOP_SMALL}px) {
      background-position: left -1461px top -1605px;
    }
    @media (max-width: ${TABLET_LARGE}px) {
      background-position: left -1511px top -1605px;
    }
    @media (max-width: ${MOBILE_LARGE}px) {
      background-position: left -1561px top -1605px;
    }
    @media (max-width: ${MOBILE_SMALL}px) {
      background-size: 1591px;
      background-position: left -726px top -734px;
    }
`

type ColumnDefinition = string[] | number
type RowDefinition = string[] | number

export const Grid = styled.div<{ columns?: ColumnDefinition, rows?: RowDefinition, gap?: string | number, breakpoints?: { width: number, columns?: ColumnDefinition, rows?: RowDefinition, gap?: string | number, style?: CSSProperties }[] }>`
    display: grid;
    // If columns is a number, use that many columns evenly spaced. Otherwise it's an array of column widths. Auto if undefined
    grid-template-columns: ${props => props.columns ? (typeof props.columns === 'number' ? `repeat(${props.columns}, 1fr)` : props.columns.join(' ')) : 'auto'};
    // If rows is a number, use that many rows evenly spaced. Otherwise it's an array of row heights. Auto if undefined
    grid-template-rows: ${props => props.rows ? (typeof props.rows === 'number' ? `repeat(${props.rows}, 1fr)` : props.rows.join(' ')) : 'auto'};
    // gap can be either a number of pixels or a string with a unit.
    grid-gap: ${props => props.gap ? typeof props.gap === 'number' ? `${props.gap}px` : props.gap : '0'};
    
    ${props => (props.breakpoints ?? []).map(bp => `
        @media (max-width: ${bp.width}px) {
            ${bp.columns ? `grid-template-columns: ${typeof bp.columns === 'number' ? `repeat(${bp.columns}, 1fr)` : bp.columns.join(' ')};` : ''}
            ${bp.rows ? `grid-template-rows: ${typeof bp.rows === 'number' ? `repeat(${bp.rows}, 1fr)` : bp.rows.join(' ')};` : ''}
            ${bp.gap ? `grid-gap: ${typeof bp.gap === 'number' ? `${bp.gap}px` : bp.gap};` : ''}
            ${bp.style ? `${bp.style}` : ''}
        }
    `)}
`

export const Row = styled.div`
  display: flex;
  align-items: center;
`

export const Column = styled.div<{ reverse?: boolean }>`
    display: flex;
    flex-direction: ${props => props.reverse ? 'column-reverse' : 'column'};
`

export const RowFull = styled.div`
  display: flex;
  width: 100%;
  justify-content: space-between;
  align-items: center;
`

export const MaskImage = styled.div<{ src: string, color: string }>`
    mask-image: url(${props => props.src});
    mask-size: contain;
    mask-repeat: no-repeat;
    background-color: ${props => props.color};
`

export const Fill = styled.div`
    flex: 1;
`

export const Space = styled.div<{ height?: number, width?: number, breakpoints?: { breakpointWidth: number, height?: number, width?: number }[] }>`
    height: ${props => props.height ?? 0}px;
    width: ${props => props.width ?? 0}px;
    ${props => (props.breakpoints ?? []).map(bp => `
        @media (max-width: ${bp.breakpointWidth}px) {
            ${bp.height ? `height: ${bp.height}px;` : ''}
            ${bp.width ? `width: ${bp.width}px;` : ''}
        }
    `)}
  flex: 0 0 auto;
`

export const ClampText = styled.div<{ lines: number }>`
    overflow: hidden;
    text-overflow: ellipsis;
    // See https://developer.mozilla.org/en-US/docs/Web/CSS/-webkit-line-clamp
    display: -webkit-box;
    -webkit-line-clamp: ${props => props.lines};
    -webkit-box-orient: vertical;
`

export const Block = styled.div`
    width: 100%;
    padding: 0 250px;

    @media (max-width: ${DESKTOP_SMALL}px) {
        padding: 0 150px;
    }
    @media (max-width: ${TABLET_LARGE}px) {
        padding: 0 100px;
    }
    @media (max-width: ${MOBILE_LARGE}px) {
        padding: 0 50px;
    }
    @media (max-width: ${MOBILE_SMALL}px) {
        padding: 0 20px;
    }
`

const Content = styled.div`
  padding-top: 190px;
`

export const PrimaryContent = styled(Content)`
  padding-top: 190px;
  padding-left: 250px;
  @media (max-width: ${DESKTOP_SMALL}px) {
    padding-left: 150px;
    padding-right: 150px;
  }
  @media (max-width: ${TABLET_LARGE}px) {
    padding-left: 100px;
    padding-right: 100px;
  }
  @media (max-width: ${MOBILE_LARGE}px) {
    padding-left: 50px;
    padding-right: 50px;
  }
  @media (max-width: ${MOBILE_SMALL}px) {
    padding-left: 20px;
    padding-right: 20px;
  }
`

export const SecondaryContent = styled(Content)`
  padding-top: 190px;
  padding-right: 250px;
  padding-left: 70px;
  border-left: 1px solid #1A1A1A;
  background-color: #E4E4F5;
  @media (max-width: ${TABLET_LARGE}px) {
    padding: 50px 100px 100px;
    border-left: none;
    border-top: 1px solid #1A1A1A;
  }
  @media (max-width: ${MOBILE_LARGE}px) {
    padding: 50px 50px 60px;
  }
  @media (max-width: ${MOBILE_SMALL}px) {
    padding: 50px 20px 60px;
  }
`