package db

import (
	"gorm.io/driver/postgres"
	"gorm.io/driver/sqlite"
	"gorm.io/gorm"
	"gorm.io/gorm/logger"
	"log"
	"stability-image-api-middleware/config"
	"stability-image-api-middleware/db/types"
	"stability-image-api-middleware/util"
	"time"
)

var DB *gorm.DB

func Connect() {
	if config.Data.Database.Dsn == "" {
		log.Fatalln("DB DSN is empty")
	}
	if config.Data.Database.Backend == "" {
		log.Fatalln("DB backend is empty")
	}

	level := logger.Error
	if config.Data.Debug {
		level = logger.Info
	}

	var dbHandle gorm.Dialector

	if config.Data.Database.Backend == "sqlite" {
		dbHandle = sqlite.Open(config.Data.Database.Dsn)
	} else if config.Data.Database.Backend == "postgres" {
		dbHandle = postgres.Open(config.Data.Database.Dsn + " connect_timeout=1")
	} else {
		log.Fatalf("'%s' is not a supported backend type\n", dbHandle)
	}

	dbLogger := logger.New(log.Default(), logger.Config{
		SlowThreshold:             200 * time.Millisecond,
		LogLevel:                  level,
		IgnoreRecordNotFoundError: true,
		Colorful:                  true,
	})
	db, err := gorm.Open(dbHandle, &gorm.Config{
		PrepareStmt:                              true,
		Logger:                                   dbLogger,
		DisableForeignKeyConstraintWhenMigrating: true,
	})
	if err != nil {
		log.Fatalf("failed opening database: %v\n", err.Error())
	}

	sqlDb, err := db.DB()
	if err != nil {
		log.Fatalf("failed getting sql database object: %v\n", err.Error())
	}

	sqlDb.SetMaxIdleConns(50)
	sqlDb.SetMaxOpenConns(100)
	sqlDb.SetConnMaxLifetime(time.Hour)

	DB = db
}

func Migrate() {
	err := DB.AutoMigrate(&types.User{})
	if err != nil {
		log.Fatalf("failed auto migrating DB: %v\n", err.Error())
	}

	var userCount int64
	DB.Model(&types.User{}).Count(&userCount)
	if userCount == 0 {
		token, err := util.GenerateRandomString(64)
		if err != nil {
			log.Fatalf("failed generating random string for default user token: %v\n", err)
		}

		user := &types.User{
			Name:  "Default User",
			Token: token,
		}
		db := DB.Create(user)
		if db.Error != nil {
			log.Fatalf("failed creating default user: %v\n", db.Error)
		} else {
			log.Printf("default user created, token is '%s'\n", token)
		}
	}
}

func init() {
	Connect()
	Migrate()
}
