import { Buffer } from 'buffer'
import { FormEvent, useState } from 'react'
import { default as sodium } from 'libsodium-wrappers'
import Link from 'next/link'
import Head from 'next/head'

import { useRouter } from 'next/router'
import {
    LoginError,
    Field,
    ImportantInfo,
    Policy,
    Submit,
    LoginContainer,
    LoginTop,
    RegisterBackground,
    Spacer,
    SignupBox,
    LoginHeader,
} from '../styles/components/login'
import { RecaptchaKey } from '../globals/constants'
import { getRegisterRequest } from '../data/request/request'
import { ArrowDownIcon, ArrowLeftIcon, ArrowUpIcon } from '../styles/ui/icons'
import { getStorage } from '../data/storage/storage'
import { User } from '../data/user/user'
import { setLocalTrialState } from '../components/trialactions'
import { SubtleButton } from '../styles/ui/button'
import { getLocalStorage } from '../util/storage'
import { loadScript, unloadScript } from '../util/util'

export default function Register(): JSX.Element {
    return (
        <LoginContainer>
            <RegisterBackground />
            <RegisterContent />
        </LoginContainer>
    )
}

function RegisterContent(): JSX.Element {
    const router = useRouter()
    const [rawUsername, setUsername] = useState('')
    const [password, setPassword] = useState('')
    const [password2, setPassword2] = useState('')
    const [giftKey, setGiftKey] = useState('')
    const [giftKeyShown, setGiftKeyShown] = useState(false)
    const [error, setError] = useState('')
    const [inputsEnabled, setInputsEnabled] = useState(true)

    const register = async (event?: FormEvent) => {
        event?.preventDefault()

        const username = rawUsername.toLowerCase().trim()

        if (!username || !password) {
            setError('Email and Password cannot be empty')
            return
        }

        if (password !== password2) {
            setError('Passwords do not match')
            return
        }

        if (!username.includes('@') || username.length <= 3) {
            setError('Email must be in email format')
            return
        }
        if (password.length < 8) {
            setError('Password must be 8 characters or more')
            return
        }

        setInputsEnabled(false)
        setError('')

        await Promise.all([
            sodium.ready,
            loadScript(`https://www.google.com/recaptcha/api.js?render=${RecaptchaKey}`),
        ])

        localStorage.removeItem('session')

        const email = username

        try {
            const register = function (captcha_token: string) {
                const access_key = sodium
                    .crypto_pwhash(
                        64,
                        new Uint8Array(Buffer.from(password)),
                        sodium.crypto_generichash(
                            sodium.crypto_pwhash_SALTBYTES,
                            password.slice(0, 6) + username + 'novelai_data_access_key'
                        ),
                        2,
                        2000000,
                        sodium.crypto_pwhash_ALG_ARGON2ID13,
                        'base64'
                    )
                    .slice(0, 64)
                const encryption_key = sodium.crypto_pwhash(
                    128,
                    new Uint8Array(Buffer.from(password)),
                    sodium.crypto_generichash(
                        sodium.crypto_pwhash_SALTBYTES,
                        password.slice(0, 6) + username + 'novelai_data_encryption_key'
                    ),
                    2,
                    2000000,
                    sodium.crypto_pwhash_ALG_ARGON2ID13,
                    'base64'
                )
                getRegisterRequest(access_key, encryption_key, captcha_token, email, giftKey)
                    .register()
                    .then(async ({ auth_token }) => {
                        const user = new User(auth_token, encryption_key)
                        const localSettingsString = getLocalStorage('noAccountSettings')
                        if (localSettingsString) {
                            user.settings = JSON.parse(localSettingsString)
                            user.settings.remoteDefault = true
                            getStorage(user).saveSettings(user.settings)
                        }
                        setLocalTrialState(-1)
                        unloadScript(`https://www.google.com/recaptcha/api.js?render=${RecaptchaKey}`)
                        delete (global.window as any).grecaptcha
                        delete (global.window as any).rwt

                        router.push(
                            '/confirm?email=' + encodeURIComponent(Buffer.from(email).toString('base64'))
                        )
                    })
                    .catch((error: any) => {
                        if (`${error}` === 'Error: Incorrect access key.') {
                            setError('Account already exists.')
                        } else {
                            setError(error)
                        }
                        setInputsEnabled(true)
                    })
            }
            if (!global.window) return
            if (!RecaptchaKey) {
                await register('placeholder')
            } else {
                const grecaptcha = (global.window as any).grecaptcha
                grecaptcha.ready(function () {
                    grecaptcha
                        .execute(RecaptchaKey, { action: 'submit' })
                        .then(register)
                        .catch((error: any) => {
                            if (`${error}` === 'Error: Incorrect access key.') {
                                setError('Account already exists.')
                            } else {
                                setError(error)
                            }
                            setInputsEnabled(true)
                        })
                })
            }
        } catch (error: any) {
            if (`${error}` === 'Error: Incorrect access key.') {
                setError('Account already exists.')
            } else {
                setError(error)
            }
            setInputsEnabled(true)
        }
    }

    function handleKeyDown(e: any) {
        if (e.key === 'Enter' && rawUsername !== '' && password !== '' && password2 !== '') {
            register()
        }
    }

    return (
        <>
            <Head>
                <title>Register - NovelAI</title>
            </Head>
            <SignupBox>
                <div>
                    <LoginTop>
                        <Link href="/" passHref scroll={false}>
                            <a>
                                <ArrowLeftIcon />
                                Back
                            </a>
                        </Link>
                        <Link href="/login">Log In</Link>
                    </LoginTop>
                    {giftKeyShown ? <Spacer style={{ marginBottom: '52px' }}></Spacer> : <></>}
                    <LoginHeader>
                        <div>Sign Up</div>
                        <div>Get Started.</div>
                    </LoginHeader>

                    <div>
                        <label htmlFor="email">Email</label>
                    </div>
                    <Field
                        id="email"
                        value={rawUsername}
                        disabled={!inputsEnabled}
                        onChange={(e) => setUsername(e.target.value)}
                        onKeyDown={handleKeyDown}
                        type="email"
                        autoComplete="email"
                        required
                    />
                    <div>
                        <label htmlFor="password">Password</label>
                    </div>
                    <Field
                        id="password"
                        value={password}
                        disabled={!inputsEnabled}
                        onChange={(e) => setPassword(e.target.value)}
                        onKeyDown={handleKeyDown}
                        type="password"
                        autoComplete="new-password"
                        required
                    />
                    <div>
                        <label htmlFor="password2">Repeat Password</label>
                    </div>
                    <Field
                        id="password2"
                        value={password2}
                        disabled={!inputsEnabled}
                        onChange={(e) => setPassword2(e.target.value)}
                        onKeyDown={handleKeyDown}
                        type="password"
                        autoComplete="new-password"
                        required
                    />
                    <div>
                        <label htmlFor="giftKey">
                            <SubtleButton
                                onClick={(e) => {
                                    setGiftKeyShown((v) => !v)
                                    e.preventDefault()
                                    e.stopPropagation()
                                }}
                                style={{
                                    cursor: 'pointer',
                                    display: 'flex',
                                    flexDirection: 'row',
                                    justifyContent: 'space-between',
                                    alignItems: 'center',
                                    width: '100%',
                                }}
                                aria-label="Hide or Expand Gift Key Input"
                            >
                                Gift Key (optional) {giftKeyShown ? <ArrowUpIcon /> : <ArrowDownIcon />}
                            </SubtleButton>
                        </label>
                    </div>
                    {giftKeyShown ? (
                        <Field
                            id="giftKey"
                            value={giftKey}
                            disabled={!inputsEnabled}
                            onChange={(e) => setGiftKey(e.target.value)}
                            onKeyDown={handleKeyDown}
                            type="text"
                        />
                    ) : (
                        <div style={{ marginBottom: '20px' }}></div>
                    )}
                    <div style={{ marginBottom: '20px' }}></div>
                    <Submit
                        type="submit"
                        value={inputsEnabled ? 'Start Writing!' : 'Loading...'}
                        disabled={!inputsEnabled}
                        onClick={register}
                    />
                    {error ? <LoginError>{`${error}`}</LoginError> : <></>}
                    <Policy>
                        <ImportantInfo>
                            <span>NOTE: </span>
                            Please take good care of your login credentials. Due to local encryption, losing
                            your email or password results in permanently losing access to your remotely
                            stored stories.
                        </ImportantInfo>
                        <p>
                            By registering, you agree to consent to the NovelAI{' '}
                            <a href="./terms/" target="_blank">
                                Privacy Policy and Terms of Service
                            </a>
                            .
                        </p>
                        <p>
                            This site is protected by reCAPTCHA and the Google
                            <a href="https://policies.google.com/privacy" target="_blank" rel="noreferrer">
                                {' '}
                                Privacy Policy
                            </a>{' '}
                            and
                            <a href="https://policies.google.com/terms" target="_blank" rel="noreferrer">
                                {' '}
                                Terms of Service
                            </a>{' '}
                            apply.
                        </p>
                    </Policy>
                </div>
            </SignupBox>
        </>
    )
}
