/**
 * @jest-environment node
 */
import { test, expect } from '@jest/globals'

import { Story, DataOrigin } from '../story/story'
import { StoryContainer, StoryContent, StoryMetadata } from '../story/storycontainer'
import { SubtleKeyStore } from './keystore/subtle'
import { SodiumKeyStore } from './keystore/sodium'
import { KeyStore } from './keystore/keystore'

test('encrypt and decrypt subtle keystore', async () => {
    const password = 'password'
    const userkeystore = new SubtleKeyStore()

    await userkeystore.create(password)
    const keystore1 = Object.assign({}, userkeystore)
    const encrypted1 = await userkeystore.store()

    await userkeystore.load(encrypted1, password)
    const keystore2 = Object.assign({}, userkeystore)
    const encrypted2 = await userkeystore.store()

    expect(keystore2).toStrictEqual(keystore1)
    expect(encrypted2).toStrictEqual(encrypted1)
})

test('encrypt and decrypt subtle story', async () => {
    const password = 'password'
    const userkeystore = new SubtleKeyStore()

    await userkeystore.create(password)

    const story1: StoryContainer = new StoryContainer()
    story1.content.story = new Story()
    story1.metadata.description = 'Hello World'
    story1.metadata.title = 'Title'
    story1.metadata.favorite = false
    story1.metadata.id = '100200300'
    story1.metadata.tags = ['tag1', 'tag2']

    story1.content.story?.append('prompt' as DataOrigin, 'This is a story!')
    story1.content.story?.edit('This is an edited story!')
    story1.content.story?.undo()

    const encrypted = await userkeystore.encryptStoryContent(story1.content, story1.metadata)
    const story2 = await userkeystore.decryptStoryContent(encrypted)
    story2?.getStoryText()
    expect(story2).toEqual(story1.content)
})

test('encrypt and decrypt subtle story metadata', async () => {
    const password = 'password'
    const userkeystore = new SubtleKeyStore()

    await userkeystore.create(password)

    const metadata1: StoryMetadata = new StoryMetadata()
    metadata1.description = 'Hello World'
    metadata1.title = 'Title'
    metadata1.favorite = false
    metadata1.id = '100200300'
    metadata1.tags = ['tag1', 'tag2']

    const encrypted = await userkeystore.encryptStoryMetadata(metadata1)
    const metadata2 = await userkeystore.decryptStoryMetadata(encrypted)

    expect(metadata2).toEqual(metadata1)
})

test('encrypt and decrypt subtle story content', async () => {
    const password = 'password'
    const userkeystore = new SubtleKeyStore()

    await userkeystore.create(password)

    const metadata1: StoryMetadata = new StoryMetadata()
    const content1: StoryContent = new StoryContent()
    content1.story = new Story()

    content1.story?.append('prompt' as DataOrigin, 'This is a story!')
    content1.story?.edit('This is an edited story!')
    content1.story?.undo()

    const encrypted = await userkeystore.encryptStoryContent(content1, metadata1)
    const content2 = await userkeystore.decryptStoryContent(encrypted)
    if (content2 !== null) content2.getStoryText()

    expect(content2).toEqual(content1)
})

test('encrypt and decrypt sodium keystore', async () => {
    const password = 'password'
    const userkeystore = new SodiumKeyStore()

    await userkeystore.create(password)
    const keystore1 = Object.assign({}, userkeystore)
    const encrypted1 = await userkeystore.store()

    await userkeystore.load(encrypted1, password)
    const keystore2 = Object.assign({}, userkeystore)

    expect(keystore2).toStrictEqual(keystore1)
})

test('encrypt and decrypt sodium story', async () => {
    const password = 'password'
    const userkeystore = new SodiumKeyStore()

    await userkeystore.create(password)

    const story1: StoryContainer = new StoryContainer()
    story1.content.story = new Story()
    story1.metadata.description = 'Hello World'
    story1.metadata.title = 'Title'
    story1.metadata.favorite = false
    story1.metadata.id = '100200300'
    story1.metadata.tags = ['tag1', 'tag2']

    story1.content.story?.append('prompt' as DataOrigin, 'This is a story!')
    story1.content.story?.edit('This is an edited story!')
    story1.content.story?.undo()

    const encrypted = await userkeystore.encryptStoryContent(story1.content, story1.metadata)
    const story2 = await userkeystore.decryptStoryContent(encrypted)
    story2?.getStoryText()

    expect(story2).toEqual(story1.content)
})

test('encrypt and decrypt sodium story metadata', async () => {
    const password = 'password'
    const userkeystore = new SodiumKeyStore()

    await userkeystore.create(password)

    const metadata1: StoryMetadata = new StoryMetadata()
    metadata1.description = 'Hello World'
    metadata1.title = 'Title'
    metadata1.favorite = false
    metadata1.id = '100200300'
    metadata1.tags = ['tag1', 'tag2']

    const encrypted = await userkeystore.encryptStoryMetadata(metadata1)
    const metadata2 = await userkeystore.decryptStoryMetadata(encrypted)

    expect(metadata2).toEqual(metadata1)
})

test('encrypt and decrypt sodium story content', async () => {
    const password = 'password'
    const userkeystore = new SodiumKeyStore()

    await userkeystore.create(password)

    const metadata1: StoryMetadata = new StoryMetadata()
    const content1: StoryContent = new StoryContent()
    content1.story = new Story()

    content1.story?.append('prompt' as DataOrigin, 'This is a story!')
    content1.story?.edit('This is an edited story!')
    content1.story?.undo()

    const encrypted = await userkeystore.encryptStoryContent(content1, metadata1)
    const content2 = await userkeystore.decryptStoryContent(encrypted)
    if (content2 !== null) content2.getStoryText()
    expect(content2).toEqual(content1)
})

test('encrypt and decrypt sodium keystore with stories', async () => {
    const password = 'password'
    const userkeystore = new SodiumKeyStore()

    await userkeystore.create(password)

    const story1: StoryContainer = new StoryContainer()
    story1.content.story = new Story()
    story1.metadata.description = 'Hello World'
    story1.metadata.title = 'Title'
    story1.metadata.favorite = false
    story1.metadata.id = '100200300'
    story1.metadata.tags = ['tag1', 'tag2']

    story1.content.story?.append('prompt' as DataOrigin, 'This is a story!')
    story1.content.story?.edit('This is an edited story!')
    story1.content.story?.undo()

    const encryptedstory = await userkeystore.encryptStoryContent(story1.content, story1.metadata)
    const stored = await userkeystore.store()
    const encryptedcontainer = Buffer.from(JSON.stringify(stored)).toString('base64')

    const newkeystore = new KeyStore()
    await newkeystore.load(encryptedcontainer, password, 1)

    const story2 = await newkeystore.decryptStoryContent(encryptedstory)
    story2?.getStoryText()
    expect(story2).toEqual(story1.content)
})

test('encrypt and decrypt sodium keystore with subtle compatibility', async () => {
    const password = 'password'
    const userkeystore = new SubtleKeyStore()

    await userkeystore.create(password)

    const story1: StoryContainer = new StoryContainer()
    story1.content.story = new Story()
    story1.metadata.description = 'Hello World'
    story1.metadata.title = 'Title'
    story1.metadata.favorite = false
    story1.metadata.id = '100200300'
    story1.metadata.tags = ['tag1', 'tag2']

    story1.content.story?.append('prompt' as DataOrigin, 'This is a story!')
    story1.content.story?.edit('This is an edited story!')
    story1.content.story?.undo()

    const encryptedstory = await userkeystore.encryptStoryContent(story1.content, story1.metadata)
    const stored = await userkeystore.store()
    const encryptedcontainer = Buffer.from(JSON.stringify(stored)).toString('base64')

    const newkeystore = new KeyStore()
    await newkeystore.load(encryptedcontainer, password, 1)

    const story2 = await newkeystore.decryptStoryContent(encryptedstory)
    story2?.getStoryText()
    expect(story2).toEqual(story1.content)
})

test('encrypt and decrypt sodium keystore with multiple iterations', async () => {
    const password = 'password'
    const userkeystore = new SubtleKeyStore()

    await userkeystore.create(password)

    const story1: StoryContainer = new StoryContainer()
    story1.content.story = new Story()
    story1.metadata.description = 'Hello World'
    story1.metadata.title = 'Title'
    story1.metadata.favorite = false
    story1.metadata.id = '100200300'
    story1.metadata.tags = ['tag1', 'tag2']

    story1.content.story?.append('prompt' as DataOrigin, 'This is a story!')
    story1.content.story?.edit('This is an edited story!')
    story1.content.story?.undo()

    const encryptedstory = await userkeystore.encryptStoryContent(story1.content, story1.metadata)
    const stored = await userkeystore.store()
    const encryptedcontainer = Buffer.from(JSON.stringify(stored)).toString('base64')

    const newkeystore = new KeyStore()
    await newkeystore.load(encryptedcontainer, password, 1)

    const story2 = await newkeystore.decryptStoryContent(encryptedstory)

    const story3: StoryContainer = new StoryContainer()
    story3.content.story = new Story()
    story3.metadata.description = 'Hello World'
    story3.metadata.title = 'Title'
    story3.metadata.favorite = false
    story3.metadata.id = '200200300'
    story3.metadata.tags = ['tag1', 'tag2']

    story3.content.story?.append('prompt' as DataOrigin, 'This is a story!')
    story3.content.story?.edit('This is an edited story!')
    story3.content.story?.undo()

    const encryptedstory3 = await newkeystore.encryptStoryContent(story3.content, story3.metadata)

    const stored2 = await newkeystore.store()

    const newkeystore2 = new KeyStore()
    await newkeystore2.load(stored2, password, 1)

    const story4 = await newkeystore2.decryptStoryContent(encryptedstory3)

    story2?.getStoryText()
    story4?.getStoryText()

    expect(story2).toEqual(story1.content)
    expect(story4).toEqual(story3.content)
})
