import React, { useState } from 'react'
import { Text } from 'react-native'
import { useRecoilState } from 'recoil'
import { Session } from '../../../shared/globals/state'
import { WarningText } from '../../styles/fonts'
import { isSpellCheckSupported } from '../../util/compat'
import Checkbox from '../common/checkbox'
import { SettingsSectionHeader, SettingsView } from './settings.style'

export default function InterfaceSettings(): JSX.Element {
    const [session, setSession] = useRecoilState(Session)

    const [showInputBox, setShowInputBox] = useState(session.settings.showInputBox)
    const changeShowInputBox = (state: boolean) => {
        setSession({ ...session, settings: { ...session.settings, showInputBox: state } })
        setShowInputBox(state)
    }

    //Will be an editor feature, definitely
    const [editorHighlighting, setEditorHighlighting] = useState(session.settings.editorHighlighting)
    const changeEditorHighlighting = (state: boolean) => {
        setSession({ ...session, settings: { ...session.settings, editorHighlighting: state } })
        setEditorHighlighting(state)
    }

    //Check how spellcheck works on text components
    const [outputSpellcheck, setOutputSpellcheck] = useState(session.settings.editorSpellcheck)
    const changeOutputSpellcheck = (state: boolean) => {
        setSession({ ...session, settings: { ...session.settings, editorSpellcheck: state } })
        setOutputSpellcheck(state)
    }

    //Finish context viewer, and test once the editor is done
    const [contextViewerColors, setContextViewerColors] = useState(session.settings.contextViewerColors)
    const changeContextViewerColors = (state: boolean) => {
        setSession({ ...session, settings: { ...session.settings, contextViewerColors: state } })
        setContextViewerColors(state)
    }

    //The editor will have to cover this as well
    const [editorLoreKeys, setEditorLoreKeys] = useState(session.settings.editorLoreKeys)
    const changeEditorLoreKeys = (state: boolean) => {
        setSession({ ...session, settings: { ...session.settings, editorLoreKeys: state } })
        setEditorLoreKeys(state)
    }

    //A story row is needed
    const [showStoryTitle, setShowStoryTitle] = useState(session.settings.showStoryTitle)
    const changeShowStoryTitle = (state: boolean) => {
        setSession({ ...session, settings: { ...session.settings, showStoryTitle: state } })
        setShowStoryTitle(state)
    }

    return (
        <SettingsView>
            <SettingsSectionHeader>Other Settings</SettingsSectionHeader>
            <Checkbox
                label="Input Box"
                value={showInputBox ?? true}
                setValue={changeShowInputBox}
                checkedText={'The input box is visible'}
                uncheckedText={'The input box is hidden'}
            />
            <Checkbox
                label="Output Spellcheck"
                disabled={!isSpellCheckSupported}
                value={outputSpellcheck ?? true}
                setValue={changeOutputSpellcheck}
                checkedText={
                    <Text>
                        Spellcheck is enabled in the output box (on supported browsers)
                        {!isSpellCheckSupported ? (
                            <WarningText>
                                {'\n'}Your browser version has known performance issues with spellcheck.
                                Spellcheck has been automatically disabled.
                            </WarningText>
                        ) : (
                            <></>
                        )}
                    </Text>
                }
                uncheckedText={
                    <Text>
                        Spellcheck is disabled in the output box
                        {!isSpellCheckSupported ? (
                            <WarningText>
                                {'\n'}Your browser version has known performance issues with spellcheck.
                                Spellcheck has been automatically disabled.
                            </WarningText>
                        ) : (
                            <></>
                        )}
                    </Text>
                }
            />
            <Checkbox
                label="Show lorebook keys in editor"
                value={editorLoreKeys ?? false}
                setValue={changeEditorLoreKeys}
                checkedText={`Keys of currently active Lorebook entries will be bolded\
                            in the output box. Disabling this setting may improve performance\
                            on very large stories or stories with large Lorebooks.`}
                uncheckedText={'No special styling will be applied to Lorebook keys in the output box.'}
            />
            <Checkbox
                label="Output highlighting"
                value={editorHighlighting ?? true}
                setValue={changeEditorHighlighting}
                checkedText={'Text in the output box will be highlighted based on origin'}
                uncheckedText={'Text in the output box will not be highlighted'}
            />
            <Checkbox
                label="Context viewer colors"
                value={contextViewerColors ?? true}
                setValue={changeContextViewerColors}
                checkedText={'Text in the context viewer will be color coded based on origin'}
                uncheckedText={'Text in the context viewer will use the default color'}
            />
            <Checkbox
                label="Show Story Title"
                value={showStoryTitle ?? true}
                setValue={changeShowStoryTitle}
                checkedText={'The story title will be shown above the editor'}
                uncheckedText={'The story title will not be shown above the editor'}
            />
        </SettingsView>
    )
}
