import React, { useEffect, useState } from 'react'
import { SetterOrUpdater, useRecoilState, useRecoilValue, useSetRecoilState } from 'recoil'
import { View } from 'react-native'
import styled from 'styled-components/native'
import Icon from 'react-native-vector-icons/MaterialIcons'
import { PresetInfo } from '../../shared/hooks/usePresetInfo'
import { StoryPreset } from '../../shared/data/story/storysettings'
import { Session, StoryUpdate, UserPresets } from '../../shared/globals/state'
import { useSelectedStoryUpdate } from '../../shared/hooks/useSelectedStory'
import { getAvailiableModels } from '../../shared/util/models'
import { getUserSetting } from '../../shared/data/user/settings'
import { DefaultModel, normalizeModel } from '../../shared/data/request/model'
import { getModelPresets } from '../../shared/data/story/defaultpresets'
import { compareStoryAndPreset, copyPresetToStory } from '../../shared/util/presets'
import { SetterPackage, updateStory } from '../../shared/component-logic/optionslogic'
import { usePresetOptions } from '../hooks/usePresetOptions'
import SelectorModal, { SelectorSection } from './modals/selectormodal'
import { BodyLarge400 } from '../styles/fonts'

let lastStoryID = ''

export function useStoryPresetSelect(
    onUpdate?: () => void,
    onPresetChange?: () => void
): {
    presetSelect: JSX.Element
    currentPreset: PresetInfo
    setUserPresets: SetterOrUpdater<StoryPreset[]>
    setPreset: (id: string) => void
    userPresets: StoryPreset[]
    defaultPresets: StoryPreset[]
} {
    const { id: storyId, update: storyUpdate, story, meta } = useSelectedStoryUpdate()
    const setStoryUpdate = useSetRecoilState(StoryUpdate(storyId))
    const session = useRecoilValue(Session)

    const modelOptions = getAvailiableModels(session.subscription.tier >= 3)
    const selectedModel =
        modelOptions.find((m) => story && m.str === normalizeModel(story.settings.model)) ??
        modelOptions.find((m) => m.str === getUserSetting(session.settings, 'defaultModel')) ??
        modelOptions.find((m) => m.str === DefaultModel) ??
        modelOptions[0]

    const settings = story?.settings

    const [userPresets, setUserPresets] = useRecoilState(UserPresets)
    const options = usePresetOptions(selectedModel.str, true)

    let defaultPresets: StoryPreset[] = []
    if (story) {
        defaultPresets = [...getModelPresets(selectedModel.str)]
        if (story.scenarioPreset) {
            defaultPresets.push(story.scenarioPreset)
        }
    }
    // eslint-disable-next-line prefer-const
    let combinedPresets = [...userPresets, ...defaultPresets]
    const modelDefaultPresets = getModelPresets(selectedModel.str)

    const [currentPreset, setCurrentPreset] = useState<PresetInfo>({
        id: settings?.preset ?? modelDefaultPresets[0].id,
        name:
            combinedPresets.find((preset) => settings?.preset === preset.id)?.name ??
            modelDefaultPresets[0].name,
        description:
            combinedPresets.find((preset) => settings?.preset === preset.id)?.description ??
            modelDefaultPresets[0].description,
        changed: !story
            ? false
            : compareStoryAndPreset(
                  combinedPresets.find((preset) => settings?.preset === preset.id) ?? modelDefaultPresets[0],
                  story
              ),
        model:
            combinedPresets.find((preset) => settings?.preset === preset.id)?.model ??
            modelDefaultPresets[0].model,
    })

    const setterPackage: SetterPackage = {
        currentStory: meta,
        currentStoryContent: story,
        genSettings: settings?.parameters,
        updateState: setStoryUpdate,
    }

    const setPreset = (id: string): void => {
        if (!settings || !story) {
            return
        }
        const preset = combinedPresets.find((preset) => preset.id === id) ?? defaultPresets[0]
        updateStory(() => {
            copyPresetToStory(preset, story, id)
        }, setterPackage)
    }

    // eslint-disable-next-line react-hooks/rules-of-hooks
    useEffect(() => {
        if (onUpdate) {
            onUpdate()
        }

        if (!story) {
            return
        }
        const modelDefaultPresets = getModelPresets(selectedModel.str)
        const currentPresetID = settings?.preset ?? ''
        let currentPresetName = modelDefaultPresets[0].name
        let currentPresetDescription = modelDefaultPresets[0].description ?? ''
        let currentPresetModel = modelDefaultPresets[0].model
        let presetChanged = false
        const defaults = new StoryPreset('', selectedModel.str)
        const foundPreset = combinedPresets.find((preset) => preset.id === currentPresetID)
        presetChanged = compareStoryAndPreset(foundPreset ? foundPreset : defaults, story)

        if (foundPreset) {
            currentPresetName = foundPreset.name
            currentPresetDescription = foundPreset.description ?? ''
            currentPresetModel = foundPreset.model
        }
        const newPreset = {
            id: currentPresetID,
            name: currentPresetName,
            description: currentPresetDescription,
            changed: presetChanged,
            model: currentPresetModel,
        }
        if (JSON.stringify(newPreset) !== JSON.stringify(currentPreset)) {
            setCurrentPreset(newPreset)
        }
        if (presetChanged && meta?.id === lastStoryID && onPresetChange) {
            onPresetChange()
        }
        if (meta && foundPreset && presetChanged && !story.settingsDirty && meta.id !== lastStoryID) {
            updateStory(() => copyPresetToStory(foundPreset, story, currentPresetID), setterPackage, false)
        } else if (meta && foundPreset && presetChanged && !story.settingsDirty && meta.id === lastStoryID) {
            updateStory(() => {
                story.settingsDirty = true
            }, setterPackage)
        }
        lastStoryID = meta?.id ?? ''
        // eslint-disable-next-line react-hooks/exhaustive-deps
    }, [storyUpdate, meta, session, settings, userPresets])

    const presetSelect = (
        <PresetSelect options={options} currentPreset={currentPreset} setPreset={setPreset} />
    )

    return {
        presetSelect,
        currentPreset,
        setUserPresets,
        setPreset,
        userPresets,
        defaultPresets,
    }
}

export const OpenSelector = styled(Icon).attrs(() => ({
    size: 25,
    name: 'expand-more',
}))`
    color: ${(props) => props.theme.colors.textMain};
    border-color: ${(props) => props.theme.colors.bg0};
    border-left-width: 1px;
    padding-left: 7px;
`

export function PresetSelect(props: {
    currentPreset: PresetInfo
    options: SelectorSection[]
    setPreset: (id: string) => void
    minMenuHeight?: number
    maxMenuHeight?: number
}): JSX.Element {
    return (
        <SelectorModal
            data={props.options}
            selectedID={props.currentPreset.id}
            setID={props.setPreset}
            selectedLabel={
                <View style={{ padding: 10, flexDirection: 'row', justifyContent: 'space-between' }}>
                    <BodyLarge400>{props.currentPreset.name}</BodyLarge400>
                    <OpenSelector />
                </View>
            }
            noSeparators={true}
        />
    )
}
