﻿import React, { useMemo, useRef } from 'react'
import { SectionList, Dimensions, View } from 'react-native'
import { useRecoilValue } from 'recoil'
import styled from 'styled-components/native'
import { SiteTheme } from '../../../shared/globals/state'
import { BodyMedium400M, BodyMedium600M, SoftFade } from '../../styles/fonts'
import { noBounceProps } from '../../styles/noBounce'
import { HoverModal } from './common'

//This whole thing looks awful, remember to bother Tabloid until he makes modal designs for mobile.

export interface SelectorOption {
    value: string
    description: string
    rawLabel: string
    label: string | JSX.Element
}

export interface SelectorSection {
    title: string
    data: SelectorOption[]
}

//Taken from react-select, changed options to data
export interface Options extends SelectorOption {}
export interface GroupBase<Option> {
    readonly data: readonly Option[]
    readonly label?: string
}
export type OptionsOrGroups<Option, Group extends GroupBase<Option>> = readonly (Option | Group)[]

export default function SelectorModal(props: {
    data: SelectorSection[]
    selectedID: string
    setID: (id: string) => void
    onClose?(): void
    selectedLabel?: JSX.Element
    noSeparators?: boolean
}): JSX.Element {
    const siteTheme = useRecoilValue(SiteTheme)
    const labelHeight = useRef(50)

    const [visible, setVisible] = React.useState(false)
    const closeModal = () => {
        setVisible(false)
        if (props.onClose) props.onClose()
    }

    const combinedOptions = useMemo(() => {
        return props.data.reduce((total: SelectorOption[], section) => total.concat(section.data), [])
    }, [props.data])

    const selectedLabel: JSX.Element = React.useMemo(() => {
        if (!!props.selectedLabel) return props.selectedLabel
        const label = combinedOptions.find((option) => option.value === props.selectedID)?.label
        return (
            <OptionContainer
                onPress={() => setVisible(true)}
                selected={false}
                onLayout={({ nativeEvent }) => (labelHeight.current = nativeEvent.layout.height)}
            >
                {label ? (
                    (label as JSX.Element).props ? (
                        (label as JSX.Element)
                    ) : (
                        <OptionText>{label}</OptionText>
                    )
                ) : (
                    <></>
                )}
            </OptionContainer>
        )
    }, [props.data, props.selectedID])

    const renderSection = (props: { section: { label?: string } }) =>
        props.section.label && props.section.label !== '' ? (
            <HeaderContainer>
                <SectionText>{props.section.label}</SectionText>
            </HeaderContainer>
        ) : (
            <></>
        )

    const renderItem = (rProps: { item: SelectorOption & { style?: any } }) => (
        <OptionContainer
            style={rProps.item.style}
            selected={rProps.item.value === props.selectedID}
            onPress={() => {
                props.setID(rProps.item.value)
                closeModal()
            }}
        >
            {(rProps.item.label as JSX.Element).props ? (
                rProps.item.label
            ) : (
                <OptionText>{rProps.item.label}</OptionText>
            )}
        </OptionContainer>
    )

    return (
        <>
            <ButtonView onPress={() => setVisible(true)}>{selectedLabel}</ButtonView>
            <HoverModal
                visible={visible}
                onRequestClose={closeModal}
                innerStyle={{
                    borderColor: siteTheme.colors.bg3,
                    borderWidth: 1,
                    backgroundColor: siteTheme.colors.bg2,
                    maxHeight: '60%',
                    maxWidth: '80%',
                    minWidth: 250,
                    minHeight: 150,
                    height: (labelHeight.current + 10) * combinedOptions.length,
                }}
            >
                <SectionList
                    initialNumToRender={Math.ceil(
                        (Dimensions.get('window').height * 0.8) / labelHeight.current
                    )}
                    ItemSeparatorComponent={() =>
                        !props.noSeparators ? <View style={{ height: 10 }} /> : <></>
                    }
                    ListFooterComponent={() =>
                        !props.noSeparators ? <View style={{ height: 10 }} /> : <></>
                    }
                    sections={props.data}
                    renderItem={renderItem}
                    renderSectionHeader={renderSection}
                    extraData={props.selectedID}
                    {...noBounceProps}
                />
            </HoverModal>
        </>
    )
}

const ButtonView = styled.Pressable`
    background-color: ${(props) => props.theme.colors.bg2};
    border: solid 1px ${(props) => props.theme.colors.bg3};
    justify-content: center;
`

export const OptionContainer = styled.Pressable<{ selected: boolean }>`
    background-color: ${(props) => (props.selected ? props.theme.colors.bg1 : props.theme.colors.bg2)};
    justify-content: center;
    min-height: 50px;
`

//really tiny on web, maybe a bit bigger here
const SectionText = styled.Text`
    ${BodyMedium400M};
    ${SoftFade}
    margin-left: 5px;
`

export const OptionText = styled.Text`
    ${BodyMedium600M};
    margin-left: 10px;
`

const HeaderContainer = styled.View`
    border-color: ${(props) => props.theme.colors.bg3};
    border-bottom-width: 2px;
    min-height: 35px;
    justify-content: flex-end;
    margin-bottom: 10px;
    padding-left: 7px;
`
