﻿import React, { useState } from 'react'
import { View } from 'react-native'
import { useRecoilState, useRecoilValue, useSetRecoilState } from 'recoil'
import styled from 'styled-components/native'
import { HeadingLarge600M, BodyMedium600M } from '../../styles/fonts'
import { LargeTrashIcon } from '../common/icons'
import { HoverModal, ModalTitleRow } from './common'
import { ModalBackgroundSplash } from './common.style'
import { SubmitButtonText, SubtleButton, SubtleButtonText, WarningButton } from '../common/button.style'
import { GlobalUserContext } from '../../../shared/globals/globals'
import { StoryId } from '../../../shared/data/story/storycontainer'
import {
    GenerationRequestActive,
    SelectedShelf,
    SelectedStory,
    Session,
    Stories,
    StoryUpdate,
} from '../../../shared/globals/state'
import { deleteStory, StatePackage } from '../../../shared/component-logic/optionslogic'

export function StoryDeleteButton(props: { id: StoryId }): JSX.Element {
    const generationRequestActive = useRecoilValue(GenerationRequestActive)
    const currentStory = GlobalUserContext.stories.get(props.id)
    const [visible, setVisible] = useState(false)

    const statePackage: StatePackage = {
        session: useRecoilValue(Session),
        selectedShelf: useRecoilValue(SelectedShelf),
        setSelectedStory: useSetRecoilState(SelectedStory),
        storiesState: useRecoilState(Stories),
        storyUpdateState: useSetRecoilState(StoryUpdate('')),
    }

    return (
        <WarningModalButton
            visible={visible}
            setVisible={setVisible}
            buttonText={'Delete Story'}
            disabled={generationRequestActive}
            description={`Are you sure you want to delete "${currentStory?.title}"?\nThis cannot be reversed.`}
            onConfirm={() => setVisible(!deleteStory(props.id, statePackage))}
        />
    )
}

export function WarningModalButton(props: {
    visible: boolean
    setVisible: (visible: boolean) => void
    onConfirm: () => void
    buttonText: string
    label?: string
    description: string
    confirmText?: string
    cancelText?: string
    icon?: JSX.Element
    disabled?: boolean
}): JSX.Element {
    return (
        <>
            <WarningButton disabled={props.disabled} onPress={() => props.setVisible(true)}>
                <SubmitButtonText>{props.buttonText}</SubmitButtonText>
            </WarningButton>
            <DeleteModal
                visible={props.visible}
                closeModal={() => props.setVisible(false)}
                onConfirm={props.onConfirm}
                label={props.label}
                description={props.description}
                confirmText={props.confirmText}
                cancelText={props.cancelText}
                customIcon={props.icon}
            />
        </>
    )
}

export default function DeleteModal(props: {
    visible: boolean
    closeModal: () => void
    onConfirm: () => void
    disabled?: boolean
    label?: string
    description: string
    confirmText?: string
    cancelText?: string
    customIcon?: JSX.Element
}): JSX.Element {
    return (
        <HoverModal
            visible={props.visible}
            onRequestClose={props.closeModal}
            innerStyle={{ alignItems: 'center' }}
        >
            <ModalBackgroundSplash />
            <ModalTitleRow title={''} onClose={props.closeModal} transparent={true} />
            <View style={{ paddingHorizontal: 10, alignItems: 'center' }}>
                {props.customIcon ?? <BigTrash />}
                <DeleteText>{props.label ?? 'Delete Story?'}</DeleteText>
                <Description>{props.description}</Description>
                <WarningButton onPress={props.onConfirm} disabled={props.disabled}>
                    <SubmitButtonText>{props.confirmText ?? 'Delete it!'}</SubmitButtonText>
                </WarningButton>
                <SubtleButton onPress={props.closeModal}>
                    <SubtleButtonText>{props.cancelText ?? 'I changed my mind.'}</SubtleButtonText>
                </SubtleButton>
            </View>
        </HoverModal>
    )
}

const BigTrash = styled(LargeTrashIcon).attrs((props) => ({
    primary: props.theme.colors.warning,
}))``

const DeleteText = styled.Text`
    ${HeadingLarge600M};
    color: ${(props) => props.theme.colors.warning};
    margin-top: 10px;
`

const Description = styled.Text`
    ${BodyMedium600M};
    margin-vertical: 5px;
    text-align: center;
`
