﻿import React, { useMemo, useState } from 'react'
import { Text, View } from 'react-native'
import { useRecoilState, useRecoilValue, useSetRecoilState, useRecoilCallback } from 'recoil'
import dayjs from 'dayjs'
import { deleteStory, StatePackage } from '../../../shared/component-logic/optionslogic'
import { useRemoteSaveQueueStatus } from '../../../shared/data/storage/queue'
import { MetadataMatchResult } from '../../../shared/data/storage/search'
import { getStorage } from '../../../shared/data/storage/storage'
import { StoryContainer, StoryId, StoryMetadata } from '../../../shared/data/story/storycontainer'
import { GlobalUserContext } from '../../../shared/globals/globals'
import {
    GenerationRequestActive,
    SelectedShelf,
    SelectedStory,
    SelectedStoryId,
    Session,
    Stories,
    StorySearch,
    StoryShelves,
    StoryUpdate,
} from '../../../shared/globals/state'
import {
    Story as StyledStory,
    StoryContent,
    StoryMetadataInfo as StyledStoryMetadata,
    StoryFavorite,
    StoryTitle,
    StoryTags,
    StoryRemoteDone,
    StoryRemoteUpload,
    HeartDisabled,
    HeartEnabled,
    ParentView,
    Folder,
    ButtonsRow,
    ButtonView,
    Delete,
    Duplicate,
    Download,
    Edit,
    StyledShelfElement,
    ShelfButtonsRow,
    ShelfButtonView,
    ShelfTitleRow,
    ShelfIcon,
    BottomSpacer,
    StyledStoryListPlaceholder,
    LargeLookingGlassIcon,
    Library,
} from './elements.style'
import { TextHighlight } from '../util/texthighlight'
import DeleteModal from '../modals/deletemodal'
import { downloadTextFile } from '../../util/browser'
import StoryMetadataModal from '../modals/storymetadatamodal'
import { FocusedMenubarElement } from '../../globals/state'
import { ShelfMetadataModal } from '../modals/shelfmetadatamodal'
import { HeadingLarge400 } from '../../styles/fonts'
import { Plus } from './menubar.style'
import { GoldenButtonText, RegularButton } from '../common/button.style'
import useAddStory from '../../../shared/hooks/useAddStory'
import useDownloadStoryJson from '../../../shared/hooks/useDownloadStory'

export function StoryElement(props: {
    story: StoryMetadata
    onPress: (long?: boolean) => void
    id: string
    highlight: MetadataMatchResult
    parent?: StoryMetadata
}) {
    const [storyUpdate, setStoryUpdate] = useRecoilState(StoryUpdate(props.story.id))
    const remoteSaveQueue = useRemoteSaveQueueStatus()
    const focusedElement = useRecoilValue(FocusedMenubarElement)

    const toggleFavorite = (toggleStory: StoryMetadata) => {
        toggleStory.favorite = !toggleStory.favorite
        setStoryUpdate(toggleStory.save())
    }

    const showPreview =
        props.story.description.length === 0 ||
        (props.highlight.description.length === 0 && props.highlight.textPreview.length > 0)

    return (
        <StyledStory
            aria-label={props.story.title}
            selected={storyUpdate.selected ?? false}
            focused={focusedElement === props.id}
            onPress={() => props.onPress()}
            onLongPress={() => props.onPress(true)}
            delayLongPress={200}
            data-id={props.id}
        >
            <View>
                <StoryTitle text={props.story.title} highlight={props.highlight.title} />
                <StoryFavorite
                    onPress={(e) => {
                        toggleFavorite(props.story)
                    }}
                >
                    {props.story.favorite ? <HeartEnabled /> : <HeartDisabled />}
                </StoryFavorite>
            </View>
            <StoryContent>
                <TextHighlight
                    text={showPreview ? props.story.textPreview : props.story.description}
                    highlight={showPreview ? props.highlight.textPreview : props.highlight.description}
                />
            </StoryContent>
            <StyledStoryMetadata>
                Created at {dayjs(props.story.createdAt).format('YYYY-MM-DD @ HH:mma')}
            </StyledStoryMetadata>
            {props.story.remote ? (
                remoteSaveQueue.has(props.story.id) ? (
                    <StoryRemoteUpload />
                ) : (
                    <StoryRemoteDone />
                )
            ) : null}
            {props.highlight.tags.size > 0 ? (
                <>
                    <StoryTags>
                        {[...props.highlight.tags.entries()].map((entry, index) => (
                            <TextHighlight key={index} text={entry[0]} highlight={entry[1]} />
                        ))}
                    </StoryTags>
                    {!storyUpdate.selected && !props.parent ? <BottomSpacer /> : null}
                </>
            ) : null}
            {storyUpdate.selected && focusedElement === props.id ? (
                <StoryButtonsRow story={props.story} id={props.id} />
            ) : null}
            {props.parent ? (
                <ParentView>
                    <Folder />
                    <StyledStoryMetadata>Contained in {props.parent.title}</StyledStoryMetadata>
                </ParentView>
            ) : null}
        </StyledStory>
    )
}

export function ShelfElement(props: { id: StoryId }) {
    const shelf = useMemo(() => GlobalUserContext.shelves.get(props.id), [props.id])

    const [stories, setStories] = useRecoilState(Stories)
    const session = useRecoilValue(Session)
    const setSelectedShelf = useSetRecoilState(SelectedShelf)
    const [focusedElement, setFocusedElement] = useRecoilState(FocusedMenubarElement)
    const selectedStoryId = useRecoilValue(SelectedStoryId)

    const containedStories = useMemo(
        () => shelf?.children?.filter((child) => !!GlobalUserContext.stories.get(child.id)).length ?? 0,
        // eslint-disable-next-line react-hooks/exhaustive-deps
        [stories, shelf]
    )

    const [modalVisible, setModalVisible] = useState(false)
    const [deleteOpen, setDeleteOpen] = useState(false)

    //should think about moving this to its own file, as it is mostly shared by ShelfMetadataModal
    const deleteShelf = useRecoilCallback(({ set }) => () => {
        if (shelf) getStorage(session).deleteStoryShelf(shelf)
        set(StoryShelves, (shelves) => [...shelves].filter((shelf) => shelf !== props.id))
        setSelectedShelf('')
        setFocusedElement(selectedStoryId)
    })

    if (!shelf) return <Text>SHELF NOT FOUND</Text>

    return (
        <StyledShelfElement
            onPress={() => {
                setSelectedShelf(props.id)
                setFocusedElement(props.id)
            }}
            onLongPress={() => {
                setFocusedElement(props.id)
            }}
            delayLongPress={200}
            selected={focusedElement === props.id}
        >
            <ShelfTitleRow>
                <ShelfIcon width={20} height={20} />
                <StoryTitle text={shelf.title} />
            </ShelfTitleRow>
            <StoryContent>
                <TextHighlight text={shelf.description} />
            </StoryContent>
            <StyledStoryMetadata>
                Contains {containedStories} Stor{containedStories === 1 ? 'y' : 'ies'}
            </StyledStoryMetadata>
            {focusedElement === props.id ? (
                <ShelfButtonsRow>
                    <ShelfButtonView onPress={() => setModalVisible(true)}>
                        <Edit />
                    </ShelfButtonView>
                    <ShelfButtonView onPress={() => setDeleteOpen(true)}>
                        <Delete />
                    </ShelfButtonView>
                </ShelfButtonsRow>
            ) : null}
            <ShelfMetadataModal
                id={props.id}
                visible={modalVisible}
                closeModal={() => setModalVisible(false)}
            />
            <DeleteModal
                label={'Delete Shelf?'}
                description={`Really delete the Shelf ${shelf?.title}?\nContained stories will be moved out of the shelf.`}
                confirmText={'Delete Shelf'}
                visible={deleteOpen}
                closeModal={() => setDeleteOpen(false)}
                onConfirm={deleteShelf}
            />
        </StyledShelfElement>
    )
}

export function StoryNoResultPlaceholder() {
    const searchValue = useRecoilValue(StorySearch)
    return (
        <StyledStoryListPlaceholder>
            {!searchValue ? <ShelfIcon width={80} height={80} opacity={0.2} /> : <LargeLookingGlassIcon />}
            <HeadingLarge400>
                {!searchValue ? 'This shelf is empty.' : "We can't find what you're looking for."}
            </HeadingLarge400>
        </StyledStoryListPlaceholder>
    )
}

export function StoryListPlaceholder(props: { addStoryPress: () => void }) {
    return (
        <StyledStoryListPlaceholder>
            <Library />
            <HeadingLarge400>Looks like your Library is empty.</HeadingLarge400>
            <RegularButton
                onPress={props.addStoryPress}
                style={{ marginTop: 5, marginLeft: 10, marginRight: 10, alignSelf: 'stretch' }}
            >
                <GoldenButtonText>
                    <Plus />
                    {'  '}New Story
                </GoldenButtonText>
            </RegularButton>
        </StyledStoryListPlaceholder>
    )
}

function StoryButtonsRow(props: { story: StoryMetadata; id: string }) {
    const [deleteModalOpen, setDeleteModalOpen] = useState(false)
    const [metadataModalOpen, setMetadataModalOpen] = useState(false)
    const statePackage: StatePackage = {
        session: useRecoilValue(Session),
        selectedShelf: useRecoilValue(SelectedShelf),
        setSelectedStory: useSetRecoilState(SelectedStory),
        storiesState: useRecoilState(Stories),
        storyUpdateState: useSetRecoilState(StoryUpdate('')),
    }
    const generationRequestActive = useRecoilValue(GenerationRequestActive)

    const { duplicateStory } = useAddStory()
    const duplicateStoryInner = () => {
        if (!props.story) {
            return
        }
        const content = GlobalUserContext.storyContentCache.get(props.id)
        if (!content) {
            return
        }
        duplicateStory(StoryContainer.bundle(props.story, content))
    }

    const downloadStoryJson = useDownloadStoryJson(props.id)

    return (
        <ButtonsRow>
            <ButtonView onPress={() => setDeleteModalOpen(true)}>
                <Delete />
            </ButtonView>
            <ButtonView onPress={duplicateStoryInner}>
                <Duplicate />
            </ButtonView>
            <ButtonView onPress={() => downloadStoryJson()}>
                <Download />
            </ButtonView>
            <ButtonView onPress={() => setMetadataModalOpen(true)}>
                <Edit />
            </ButtonView>
            <DeleteModal
                description={`Are you sure you want to delete "${props.story.title}"?\nThis cannot be reversed.`}
                visible={deleteModalOpen}
                disabled={generationRequestActive}
                closeModal={() => setDeleteModalOpen(false)}
                onConfirm={() => setDeleteModalOpen(!deleteStory(props.id, statePackage))}
            />
            <StoryMetadataModal
                visible={metadataModalOpen}
                closeModal={() => setMetadataModalOpen(false)}
                id={props.id}
            />
        </ButtonsRow>
    )
}
