import React, { useCallback, useEffect, useState } from 'react'
import { Modal, View } from 'react-native'
import { useRecoilState, useRecoilValue, useSetRecoilState } from 'recoil'
import { adjustHue } from 'polished'
import { buildContext, ContextReport, ContextStageReport } from '../../../../shared/data/ai/context'
import { modelMaxContextSize } from '../../../../shared/data/ai/model'
import { EventHandler } from '../../../../shared/data/event/eventhandling'
import { TextGenerationModel } from '../../../../shared/data/request/model'
import { GlobalUserContext } from '../../../../shared/globals/globals'
import {
    ContextViewerPage,
    InputModes,
    LastContextReport,
    LastResponse,
    SelectedInputMode,
    SelectedStory,
    Session,
    SiteTheme,
} from '../../../../shared/globals/state'
import { Dark } from '../../../../shared/styles/themes/dark'
import { calcPreamble } from '../../../../shared/util/models'
import { getAccountContextLimit } from '../../../../shared/util/subscription'
import { FlexRow } from '../../common/common.style'
import { LightColorButton, RegularButtonText } from '../../common/button.style'
import Tabs, { Tab } from '../tabs'
import { FullModalView } from '../../modals/common.style'
import { ModalTitleRow } from '../../modals/common'
import { NoModule } from '../../../../shared/data/story/storysettings'
import { BodyLarge600, BodyMedium400 } from '../../../styles/fonts'
import Icon from 'react-native-vector-icons/MaterialIcons'
import {
    ContextDisplay,
    InvisibleContext,
    ShownContext,
    StageButton,
    StyledContext,
} from './contextviewer.style'
import { EphemeralContextModal } from './ephemeralcontext'

export function ContextViewerButtons(): JSX.Element {
    const setContextViewerPage = useSetRecoilState(ContextViewerPage)
    const [ephemeralContextModalVisible, setEphemeralContextModalVisible] = useState(false)

    return (
        <View>
            <FlexRow>
                <LightColorButton
                    style={{ width: '50%', flexShrink: 1, marginRight: 12 }}
                    onPress={() => setContextViewerPage(0)}
                >
                    <RegularButtonText>Last Context</RegularButtonText>
                </LightColorButton>
                <LightColorButton
                    style={{ width: '50%', flexShrink: 1 }}
                    onPress={() => setContextViewerPage(1)}
                >
                    <RegularButtonText>Current Context</RegularButtonText>
                </LightColorButton>
            </FlexRow>
            <LightColorButton style={{ marginTop: 10 }} onPress={() => setEphemeralContextModalVisible(true)}>
                <RegularButtonText>Edit Ephemeral Context</RegularButtonText>
            </LightColorButton>
            <ContextViewerModal
                setTab={(index: number) => setContextViewerPage(index)}
                onRequestClose={() => setContextViewerPage(-1)}
            />
            <EphemeralContextModal
                visible={ephemeralContextModalVisible}
                closeModal={() => setEphemeralContextModalVisible(false)}
            />
        </View>
    )
}

export default function ContextViewerModal(props: {
    setTab: (index: number) => void
    onRequestClose: () => void
}): JSX.Element {
    const lastContext = useRecoilValue(LastContextReport)
    const [selectedStory] = useRecoilState(SelectedStory)
    const inputMode = useRecoilValue(SelectedInputMode)
    const inputModes = useRecoilValue(InputModes)
    const currentTab = useRecoilValue(ContextViewerPage)
    const session = useRecoilValue(Session)
    const siteTheme = useRecoilValue(SiteTheme)
    const [currentContext, setCurrentContext] = useState(new ContextReport())

    const currentStoryContent = GlobalUserContext.storyContentCache.get(selectedStory.id)
    const currentStoryMetadata = GlobalUserContext.stories.get(selectedStory.id)
    const model = currentStoryContent?.settings.model || session.settings.defaultModel
    const [loading, setLoading] = useState(true)
    const [showMax, setShowMax] = useState(getAccountContextLimit(session) >= 2048)
    const [colors, setColors] = useState<Map<string, string>>(new Map<string, string>())
    const [lastColors, setLastColors] = useState<Map<string, string>>(new Map<string, string>())

    const assignColors = useCallback(
        (context: ContextReport): Map<string, string> => {
            const theme = siteTheme ?? Dark

            const loreColors = [
                theme.colors.textUser,
                adjustHue(15, theme.colors.textUser),
                adjustHue(30, theme.colors.textUser),
                adjustHue(-15, theme.colors.textUser),
                adjustHue(-30, theme.colors.textUser),
            ]
            const ephemeralColors = [
                theme.colors.textEdit,
                adjustHue(15, theme.colors.textEdit),
                adjustHue(30, theme.colors.textEdit),
                adjustHue(-15, theme.colors.textEdit),
                adjustHue(-30, theme.colors.textEdit),
            ]
            const newColors = new Map<string, string>()
            const loreStack: { id: string; color: string }[] = []
            const ephemeralStack: { id: string; color: string }[] = []
            const loreAdded = { count: 0 }
            const ephmeralAdded = { count: 0 }

            for (const [i, section] of context.structuredOutput.entries()) {
                let color = theme.colors.textMain
                if (session.settings.contextViewerColors === false) {
                    continue
                }
                switch (section.type) {
                    case 'ephemeral':
                    case 'lore':
                        const stack = section.type === 'lore' ? loreStack : ephemeralStack
                        const colors = section.type === 'lore' ? loreColors : ephemeralColors
                        const added = section.type === 'lore' ? loreAdded : ephmeralAdded
                        color = colors[added.count % (colors.length - 1)]
                        added.count = added.count + 1
                        if (!stack[stack.length - 1] || !context.structuredOutput[i + 1]) {
                            break
                        }
                        if (
                            color === stack[stack.length - 1].color &&
                            context.structuredOutput[i + 1].identifier === stack[stack.length - 1].id
                        ) {
                            added.count++
                            color = colors[i % (colors.length - 1)]
                        }
                        if (stack[stack.length - 1].id === section.identifier) {
                            const top = loreStack.pop()
                            if (top) color = top.color
                        }
                        if (
                            context.structuredOutput
                                .slice(i + 1)
                                .some((o) => section.identifier === o.identifier)
                        ) {
                            stack.push({ id: section.identifier, color: color })
                        }
                        break
                    case 'memory':
                        color = adjustHue(15, theme.colors.textPrompt)
                        break
                    case 'an':
                        color = adjustHue(-15, theme.colors.textPrompt)
                        break
                    case 'story':
                        color = theme.colors.textAI
                        break
                    default:
                        break
                }
                newColors.set(section.identifier, color)
            }
            return newColors
        },
        [session.settings.contextViewerColors, siteTheme]
    )

    useEffect(() => {
        let colors = assignColors(lastContext)
        for (const status of lastContext.contextStatuses) {
            if (status.subContext) {
                colors = new Map([...colors, ...assignColors(status.subContext)])
            }
        }
        setLastColors(colors)
    }, [assignColors, lastContext])
    useEffect(() => {
        const buildCurrentContext = async (b: boolean) => {
            setLoading(true)
            if (currentStoryContent && currentStoryMetadata) {
                let limit =
                    (b ? modelMaxContextSize(model) : 1024) -
                    currentStoryContent.settings.parameters.max_length
                if (session.settings.continueGenerationToSentenceEnd ?? true) {
                    limit -= 20
                }

                buildContext(
                    currentStoryContent,
                    new EventHandler(currentStoryContent, currentStoryMetadata, inputMode, inputModes), //Note to self: Look into how event handling works
                    limit,
                    session.settings.prependPreamble ?? true
                ).then((result) => {
                    setCurrentContext(result)
                    setLoading(false)
                    let colors = assignColors(result)
                    for (const status of result.contextStatuses) {
                        if (status.subContext) {
                            colors = new Map([...colors, ...assignColors(status.subContext)])
                        }
                    }
                    setColors(colors)
                })
            }
        }
        if (currentTab === 1) {
            buildCurrentContext(showMax)
        }
    }, [
        assignColors,
        currentStoryContent,
        currentStoryMetadata,
        inputMode,
        inputModes,
        currentTab,
        session.settings.continueGenerationToSentenceEnd,
        session.settings.defaultModel,
        session.settings?.prependPreamble,
        showMax,
        model,
    ])

    const lastResponse = useRecoilValue(LastResponse)

    return (
        <Modal visible={currentTab >= 0} onRequestClose={() => props.setTab(-1)}>
            <ModalTitleRow title={'Context Viewer'} onClose={() => props.setTab(-1)} />
            <FullModalView>
                <Tabs selected={currentTab} setSelected={props.setTab}>
                    <Tab title={'Last Context'}>
                        <>
                            <ContextReportDisplay
                                colors={lastColors}
                                context={lastContext}
                                usedTokens={`(max tokens [${
                                    showMax ? modelMaxContextSize(model).toString() : '1024'
                                }]${
                                    currentStoryContent
                                        ? ' - output length [' +
                                          currentStoryContent.settings.parameters.max_length +
                                          ']'
                                        : ''
                                }${
                                    (currentStoryContent &&
                                        currentStoryContent.settings.prefix !== NoModule) ??
                                    false
                                        ? ' - AI Module [20]'
                                        : ''
                                }${
                                    session.settings.continueGenerationToSentenceEnd ?? true
                                        ? ' - continue to sentence end allowance [20]'
                                        : ''
                                })`}
                            />
                            {/** Tokenizer Output component goes here */}
                        </>
                    </Tab>
                    <Tab title={'Current Context'}>
                        {/** Override Token Toggle goes here */}
                        {loading ? (
                            <BodyLarge600>Building Context...</BodyLarge600>
                        ) : (
                            <ContextReportDisplay
                                colors={colors}
                                context={currentContext}
                                usedTokens={`(max tokens [${
                                    showMax ? modelMaxContextSize(model).toString() : '1024'
                                }]${
                                    currentStoryContent
                                        ? ' - output length [' +
                                          currentStoryContent.settings.parameters.max_length +
                                          ']'
                                        : ''
                                }${
                                    (currentStoryContent &&
                                        currentStoryContent.settings.prefix !== NoModule) ??
                                    false
                                        ? ' - AI Module [20]'
                                        : ''
                                }${
                                    session.settings.continueGenerationToSentenceEnd ?? true
                                        ? ' - continue to sentence end allowance [20]'
                                        : ''
                                })`}
                            />
                        )}
                    </Tab>
                </Tabs>
            </FullModalView>
        </Modal>
    )
}
function ContextReportDisplay(props: {
    colors: Map<string, string>
    context: ContextReport
    usedTokens?: string
    showPreamble?: boolean
}) {
    const [showNoKey, setShowNoKey] = useState(false)
    const [selectedEntry, setSelectedEntry] = useState('')
    const [selectedSubcontext, setSelectedSubcontext] = useState<ContextReport | undefined>()
    const [subcontextName, setSubcontextName] = useState('')

    if (props.context.output !== props.context.structuredOutput.map((o) => o.text).join('')) {
        throw 'Structured output did not match text output. '
    }
    const session = useRecoilValue(Session)
    const [selectedStage, setSelectedStage] = useState(props.context.stageReports.length - 1)
    let includedIndex = 0
    const stage = props.context.stageReports[selectedStage] ?? new ContextStageReport()
    const nextStage = props.context.stageReports[selectedStage + 1]

    const siteTheme = useRecoilValue(SiteTheme)

    return (
        <View style={{ marginVertical: 10 }}>
            <BodyMedium400>
                {props.context.tokens.length} tokens filled out of a maximum of {props.context.maxTokens}{' '}
                {props.usedTokens}
            </BodyMedium400>
            <FlexRow style={{ marginVertical: 10 }}>
                <StageButton
                    disabled={selectedStage <= 0}
                    onPress={() => setSelectedStage(selectedStage - 1)}
                >
                    <Icon name={'arrow-left'} size={25} style={{ color: siteTheme.colors.textMain }} />
                    <RegularButtonText>Prev Stage</RegularButtonText>
                </StageButton>
                <BodyLarge600>
                    Stage {selectedStage}/{props.context.stageReports.length - 1}
                </BodyLarge600>
                <StageButton
                    disabled={selectedStage + 1 >= props.context.stageReports.length}
                    onPress={() => setSelectedStage(selectedStage + 1)}
                >
                    <RegularButtonText>Next Stage</RegularButtonText>
                    <Icon name={'arrow-right'} size={25} style={{ color: siteTheme.colors.textMain }} />
                </StageButton>
            </FlexRow>
            {stage.description !== '' || nextStage ? (
                <BodyMedium400>
                    {stage.description !== '' ? `Just: Inserted ${stage.description}` : ''}
                    {'\n'}
                    {nextStage ? `Next: Insert ${nextStage.description}` : ''}
                </BodyMedium400>
            ) : null}
            <ContextDisplay>
                <ShownContext>
                    {props.context.preamble.str}
                    {stage.structuredOutput.map((o, i) => {
                        const color = props.colors.get(o.identifier) ?? ''
                        return (
                            <StyledContext
                                selected={selectedEntry === o.identifier}
                                color={color}
                                key={i}
                                selectable={true}
                                onPress={() => {
                                    if (selectedEntry === o.identifier) {
                                        setSelectedEntry('')
                                    } else {
                                        setSelectedEntry(o.identifier)
                                    }
                                }}
                            >
                                {o.text}
                            </StyledContext>
                        )
                    })}
                </ShownContext>
                <InvisibleContext>
                    {props.context.preamble.str}
                    {props.context.structuredOutput.map((o, i) => {
                        const color = props.colors.get(o.identifier) ?? ''
                        return (
                            <StyledContext
                                selected={selectedEntry === o.identifier}
                                color={color}
                                key={i}
                                onPress={() => {
                                    if (selectedEntry === o.identifier) {
                                        setSelectedEntry('')
                                    } else {
                                        setSelectedEntry(o.identifier)
                                    }
                                }}
                            >
                                {o.text}
                            </StyledContext>
                        )
                    })}
                </InvisibleContext>
            </ContextDisplay>
        </View>
    )
}

export function EditContextSettings(): JSX.Element {
    const [modalVisible, setModalVisible] = useState(false)

    return <></>
}
