import { createModelSchema, list, object, primitive } from 'serializr'
import { DataOrigin } from '../../../shared/components/editor/glue'
import { uniqueid, UniqueId } from '../../../shared/data/document/util'

export enum SegmentDecorations {
    none = 0,
    struck = 1,
    underlined = 2,
    both = 3,
}

export class SegmentStyle {
    bold?: boolean
    italic?: boolean
    decorations?: SegmentDecorations
}

createModelSchema(SegmentStyle, {
    bold: primitive(),
    italic: primitive(),
    decorations: primitive(),
})

export class VisualSegment {
    constructor(id: number, style?: SegmentStyle, origin?: DataOrigin, data?: string) {
        this.id = id
        this.style = style ?? {}
        this.origin = origin ?? DataOrigin.user
        this.data = data ?? ''
    }

    id: UniqueId
    style: SegmentStyle
    origin: DataOrigin
    data: string
}

export class Segment {
    constructor(origin?: DataOrigin, data?: string, style?: SegmentStyle, start?: number) {
        this.id = -1
        this.origin = origin ?? DataOrigin.user
        this.data = data ?? ''
        this.style = style ?? {}
        this.start = start ?? 0
    }

    id: UniqueId
    origin: DataOrigin
    data: string
    style: SegmentStyle
    start: number

    assignID(usedIDs: UniqueId[]) {
        /*
        if (this.id !== -1) {
            //If replacing id, remove old ID from array
            const inIDs = usedIDs.indexOf(this.id)
            if (inIDs !== -1) usedIDs.splice(inIDs, 1)
        }
        */

        let id = uniqueid()
        while (usedIDs.includes(id)) id = uniqueid()
        this.id = id
        usedIDs.push(id)
        return id
    }

    getVisual(): VisualSegment {
        return new VisualSegment(this.id, this.style, this.origin, this.data)
    }
}

createModelSchema(Segment, {
    id: primitive(),
    origin: primitive(),
    data: primitive(),
    style: object(SegmentStyle),
    start: primitive(),
})

export class ParagraphDimensions {
    width: number
    height: number
    lastLine: number

    constructor(width?: number, height?: number, lastLine?: number) {
        this.width = width ?? 0
        this.height = height ?? 0
        this.lastLine = lastLine ?? 0
    }
}

createModelSchema(ParagraphDimensions, {
    width: primitive(),
    height: primitive(),
    lastLine: primitive(),
})

export class Paragraph {
    constructor(segments?: Segment[], start?: number, length?: number, dimensions?: ParagraphDimensions) {
        this.segments = segments ?? []
        this.start = start ?? 0
        this.length = length ?? segments?.reduce((total, seg) => total + seg.data.length, 1) ?? 1
        this.dimensions = dimensions ?? new ParagraphDimensions()
    }

    segments: Segment[]
    start: number
    length: number
    dimensions: ParagraphDimensions

    updateLength() {
        this.length = this.segments.reduce((total, seg) => total + seg.data.length, 1)
        return this.length
    }

    updateSegStarts(startAt?: number) {
        let counter = startAt ? this.segments[startAt].start ?? this.start : this.start
        this.segments.slice(startAt ?? 0).forEach((seg) => {
            seg.start = counter
            counter += seg.data.length
        })
    }
}

createModelSchema(Paragraph, {
    segments: list(object(Segment)),
    start: primitive(),
    length: primitive(),
    dimensions: object(ParagraphDimensions),
})
