--- Sexbound.Actor.Pregnant Class Module.
-- @classmod Sexbound.Actor.Pregnant
-- @author Locuturus
-- @license GNU General Public License v3.0
if not SXB_RUN_TESTS then
    require("/scripts/sexbound/lib/sexbound/actor/plugin.lua")
    require("/scripts/sexbound/plugins/pregnant/baby_factory.lua")
end

Sexbound.Actor.Pregnant = Sexbound.Actor.Plugin:new()
Sexbound.Actor.Pregnant_mt = {
    __index = Sexbound.Actor.Pregnant
}

--- Instantiates a new instance of Pregnant.
-- @param parent
-- @param config
function Sexbound.Actor.Pregnant:new(parent, config)
    local _self = setmetatable({
        _logPrefix = "PREG",
        _config = config
    }, Sexbound.Actor.Pregnant_mt)

    _self:validateConfig()
    _self:init(parent, _self._logPrefix)
    _self:ensurePregnanciesAreStoredAsTable()
    _self:refreshStatusForThisActor()

    return _self
end

--- Handles message events.
-- @param message
function Sexbound.Actor.Pregnant:onMessage(message)
    if message:getType() == "Sexbound:Pregnant:BecomePregnant" then
        self:handleBecomePregnant(message)
    end
end

--- Helper - Filters the pregnancies in storage and ensures it is set to a table
function Sexbound.Actor.Pregnant:ensurePregnanciesAreStoredAsTable()
    local storage = self:getParent():getStorage():getData("sexbound").pregnant

    if "table" ~= type(storage) then
        self:getParent():getStorage():getData("sexbound").pregnant = {}
    end
end

--- Handler function for 'BecomePregnant' message
-- @param message
function Sexbound.Actor.Pregnant:handleBecomePregnant(message)
    self.otherActor = message:getData()

    self:getLog():info("Actor is trying to become pregnant: " .. self:getParent():getName())

    -- All of these checks must pass or this actor will not become pregnant
    if self:thisActorIsAlreadyTooPregnant() or self:thisActorIsCloneOfOtherActor() or
        self:thisActorIsUsingContraception() or self:otherActorIsUsingContraception() or
        not self:thisActorHasRoleInPositionWhichCanBeImpregnated() or not self:thisActorCanOvulate() or
        not self:otherActorCanProduceSperm() or not self:otherActorIsCompatibleSpecies() or
        not self:thisActorHasEnoughFertility() then
        return
    end

    self:becomePregnant()
end

--- Adds a new pregnancy to the in-game entity associated with this actor
function Sexbound.Actor.Pregnant:becomePregnant()
    self:getLog():info("Actor will become pregnant: " .. self:getParent():getName())

    local babyConfig = self:makeBaby()

    self:getLog():debug(babyConfig)

    self:increaseStatisticPregnancyCountForThisActor()
    self:increaseStatisticImpregnateOtherCountForOtherActor()
    self:tryToSendRadioMessageToThisActor(babyConfig)
    self:tryToSendRadioMessageToOtherActor(babyConfig)
    self:tryToRefreshStatusTextForThisActor()
    self:tryToApplyPregnantStatusEffectForThisActor()

    self:storeBaby(babyConfig)

    self.otherActor = nil
end

--- Generates and returns a random number between 1 & 100
function Sexbound.Actor.Pregnant:generateRandomNumber()
    return util.randomIntInRange({1, 100})
end

--- Increases the 'impregnateOtherCount' statistic of the other actor by +1
function Sexbound.Actor.Pregnant:increaseStatisticImpregnateOtherCountForOtherActor()
    world.sendEntityMessage(self.otherActor:getEntityId(), "Sexbound:Statistics:Add", {
        name = "impregnateOtherCount",
        amount = 1
    })
end

--- Increases the 'pregnancyCount' statistic of the this actor by +1
function Sexbound.Actor.Pregnant:increaseStatisticPregnancyCountForThisActor()
    world.sendEntityMessage(self:getParent():getEntityId(), "Sexbound:Statistics:Add", {
        name = "pregnancyCount",
        amount = 1
    })
end

--- Returns whether or not the actor is pregnant
function Sexbound.Actor.Pregnant:isPregnant()
    return self:getCurrentPregnancyCount() > 0
end

--- Returns a reference to the last current pregnancy
function Sexbound.Actor.Pregnant:lastCurrentPregnancy()
    return self:getCurrentPregnancies(self:getCurrentPregnancyCount())
end

--- Loads notifications configuration from file
function Sexbound.Actor.Pregnant:loadNotificationDialog()
    local filePath = self:getConfig().notifications
    local config = root.assetJson(filePath)
    local plugins = config.plugins or {}

    return plugins.pregnant
end

--- Returns whether or not this actor has a role in a position which allows it to be impregnated
function Sexbound.Actor.Pregnant:thisActorHasRoleInPositionWhichCanBeImpregnated()
    local positionConfig = self:getParent():getPosition():getConfig()

    return positionConfig.possiblePregnancy[self:getParent():getActorNumber()] or false
end

--- Returns whether or not this actor is already too pregnant
function Sexbound.Actor.Pregnant:thisActorIsAlreadyTooPregnant()
    if self:getConfig().allowMultipleImpregnations == true or self:getConfig().enableMultipleImpregnations == true then
        return false
    end

    return self:isPregnant()
end

--- Returns whether or not this actor can ovulate
function Sexbound.Actor.Pregnant:thisActorCanOvulate()
    local thisActor = self:getParent()

    if self:getConfig().enableFreeForAll == true then
        return true
    end
    if thisActor:getIdentity().sxbCanOvulate == true then
        return true
    end

    local targetGender = thisActor:getSubGender() or thisActor:getGender()
    for _, gender in ipairs(self:getConfig().whichGendersCanOvulate) do
        if (gender == targetGender) then return true end
    end

    return false
end

--- Returns whether or not this actor is a clonse of the other actor
function Sexbound.Actor.Pregnant:thisActorIsCloneOfOtherActor()
    if self:getConfig().enableAsexualReproduction == true then
        return false
    end
    return self:getParent():getId() == self.otherActor:getId()
end

--- Returns whether or not this actor has enough fertility to become pregnant
function Sexbound.Actor.Pregnant:thisActorHasEnoughFertility()
    local sxbFertility = self:getParent():getIdentity().sxbFertility
    local fertility = sxbFertility or self:getConfig().fertility
    local pregnancyChance = self:generateRandomNumber() / 100;
    self:getLog():info("Pregnancy roll : " .. pregnancyChance .. " <= " .. fertility)
    return pregnancyChance <= fertility
end

--- Returns whether or not this actor is using contraception
function Sexbound.Actor.Pregnant:thisActorIsUsingContraception()
    return self:getParent():status():hasOneOf(self:getConfig().preventStatuses)
end

--- Returns whether or not the other actor can produce sperm
function Sexbound.Actor.Pregnant:otherActorCanProduceSperm()
    if self:getConfig().enableFreeForAll == true or self.otherActor:getIdentity().sxbCanProduceSperm == true then
        return true
    end

    local targetGender = self.otherActor:getSubGender() or self.otherActor:getGender()
    for _, gender in ipairs(self:getConfig().whichGendersCanProduceSperm) do
        if (gender == targetGender) then return true end
    end

    return false
end

--- Returns whether or not the other actor is using contraception
function Sexbound.Actor.Pregnant:otherActorIsUsingContraception()
    return self.otherActor:status():hasOneOf(self:getConfig().preventStatuses)
end

--- Returns whether or not the other actor is a compatible species
function Sexbound.Actor.Pregnant:otherActorIsCompatibleSpecies()
    if self:getConfig().enableFreeForAll == true or self:getConfig().enableCompatibleSpeciesOnly == false or
        self:getParent():getSpecies() == self.otherActor:getSpecies() then
        return true
    end
    local speciesList = self:getConfig().compatibleSpecies[self:getParent():getSpecies()]
    local otherActorSpecies = self.otherActor:getSpecies()
    for _, species in ipairs(speciesList or {}) do
        if otherActorSpecies == species then
            return true
        end
    end
    return false
end

--- Refreshes the status of this actor depending upon whether it is pregnant or not
function Sexbound.Actor.Pregnant:refreshStatusForThisActor()
    local status = self:getParent():status()
    if self:isPregnant() then
        status:addStatus("pregnant")
    else
        status:removeStatus("pregnant")
    end
end

--- Prepares the message to send to other actor
function Sexbound.Actor.Pregnant:prepareMessageToSendToOtherActor(baby)
    local dayCount = baby.dayCount
    local dialog = self:loadNotificationDialog() or {}
    local message = ""

    if dayCount <= 1 then
        message = message .. (dialog.impregnatedMessage1 or "")
    else
        message = message .. (dialog.impregnatedMessage2 or "")
    end

    message = util.replaceTag(message, "name", "^green;" .. self:getParent():getName() .. "^reset;")
    message = util.replaceTag(message, "daycount", "^red;" .. dayCount .. "^reset;")
    return message
end

--- Prepares the message to send to this actor
function Sexbound.Actor.Pregnant:prepareMessageToSendToThisActor(baby)
    local dayCount = baby.dayCount
    local dialog = self:loadNotificationDialog() or {}
    local message = ""

    if dayCount <= 1 then
        message = message .. (dialog.impregnatedMessage3 or "")
    else
        message = message .. (dialog.impregnatedMessage4 or "")
    end

    message = util.replaceTag(message, "name", "^green;" .. self.otherActor:getName() .. "^reset;")
    message = util.replaceTag(message, "daycount", "^red;" .. dayCount .. "^reset;")
    return message
end

--- Tries to apply 'sexbound_pregnant' status effect for this actor
function Sexbound.Actor.Pregnant:tryToApplyPregnantStatusEffectForThisActor()
    if self:getConfig().enableSilentPregnancies == true then
        return
    end
    world.sendEntityMessage(self:getParent():getEntityId(), "applyStatusEffect", "sexbound_pregnant", math.huge,
        self.otherActor:getEntityId())
end

--- Tries to refresh status text for this actor
function Sexbound.Actor.Pregnant:tryToRefreshStatusTextForThisActor()
    if self:getConfig().enableSilentPregnancies == true or "npc" ~= self:getParent():getEntityType() then
        return
    end
    world.sendEntityMessage(self:getParent():getEntityId(), "Sexbound:Pregnant:RefreshStatusText")
end

--- Tries to send radio message to this actor
function Sexbound.Actor.Pregnant:tryToSendRadioMessageToThisActor(baby)
    if self:getConfig().enableNotifyPlayers == false then
        return
    end
    if self:getConfig().enableSilentPregnancies == true then
        return
    end
    if "player" ~= self:getParent():getEntityType() then
        return
    end

    local message = self:prepareMessageToSendToThisActor(baby)

    world.sendEntityMessage(self:getParent():getEntityId(), "queueRadioMessage", {
        messageId = "Pregnant:Success",
        unique = false,
        text = message
    })

    return true
end

--- Tries to send radio message to the other actor
function Sexbound.Actor.Pregnant:tryToSendRadioMessageToOtherActor(baby)
    if self:getConfig().enableNotifyPlayers == false then
        return
    end
    if self:getConfig().enableSilentPregnancies == true then
        return
    end
    if "player" ~= self.otherActor:getEntityType() then
        return
    end

    local message = self:prepareMessageToSendToOtherActor(baby)

    world.sendEntityMessage(self.otherActor:getEntityId(), "queueRadioMessage", {
        messageId = "Pregnant:Success",
        unique = false,
        text = message
    })

    return true
end

--- Makes a new baby and stores it in this actor's storage
function Sexbound.Actor.Pregnant:makeBaby()
    local baby = BabyFactory:new(self):make()
    baby.birthEntityGroup, baby.birthSpecies = self:reconcileEntityGroups()
    baby.fatherName = self.otherActor:getName()
    return baby
end

--- Reconciles differences between this actor and the other actor
function Sexbound.Actor.Pregnant:reconcileEntityGroups()
    if self:getParent():getEntityGroup() == "humanoid" and self.otherActor:getEntityGroup() == "humanoid" then
        return "humanoid", self:generateBirthSpecies()
    end

    if self:getParent():getEntityGroup() == "monsters" and self.otherActor:getEntityGroup() == "monsters" then
        return "monsters", self:generateBirthSpecies()
    end

    if self:getParent():getEntityGroup() == "monsters" then
        return "monsters", self:getParent():getSpecies()
    end

    if self.otherActor:getEntityGroup() == "monsters" then
        return "monsters", self.otherActor:getSpecies()
    end

    return self:getParent():getEntityGroup(), self:getParent():getSpecies()
end

-- Returns a random species name based on the species of the parents
function Sexbound.Actor.Pregnant:generateBirthSpecies()
    local actor1Species = self:getParent():getSpecies() or "human"
    local actor2Species = self.otherActor:getSpecies() or actor1Species
    return util.randomChoice({actor1Species, actor2Species})
end

--- Stores a baby inside this actor
function Sexbound.Actor.Pregnant:storeBaby(baby)
    local _storage = self:getParent():getStorage()
    local _storageData = _storage:getData()
    table.insert(_storageData.sexbound.pregnant, baby)

    self:getParent():getStorage():sync(function(storageData)
        storageData.sexbound = storageData.sexbound or {}
        storageData.sexbound.pregnant = storageData.sexbound.pregnant or {}
        table.insert(storageData.sexbound.pregnant, baby)
        return storageData
    end)
end

--- Validates the loaded config and sets missing config options to be default values.
function Sexbound.Actor.Pregnant:validateConfig()
    self:validateCompatibleSpecies(self._config.compatibleSpecies)
    self:validateEnableAsexualReproduction(self._config.enableAsexualReproduction)
    self:validateEnableCompatibleSpeciesOnly(self._config.enableCompatibleSpeciesOnly)
    self:validateEnableFreeForAll(self._config.enableFreeForAll)
    self:validateEnableMultipleImpregnations(self._config.enableMultipleImpregnations)
    self:validateEnableNotifyPlayers(self._config.enableNotifyPlayers)
    self:validateEnablePregnancyFetish(self._config.enablePregnancyFetish)
    self:validateEnableSilentImpregnations(self._config.enableSilentImpregnations)
    self:validateFertility(self._config.fertility)
    self:validateNotifications(self._config.notifications)
    self:validatePreventStatuses(self._config.preventStatuses)
    self:validateWhichGendersCanProduceSperm(self._config.whichGendersCanProduceSperm)
    self:validateWhichGendersCanOvulate(self._config.whichGendersCanOvulate)
    self:validateTrimesterCount(self._config.trimesterCount)
    self:validateTrimesterLength(self._config.trimesterLength)
    self:validateUseOSTimeForPregnancies(self._config.useOSTimeForPregnancies)
end

--- Returns a reference to this actor's current pregnancies table
-- @param index
function Sexbound.Actor.Pregnant:getCurrentPregnancies(index)
    local pregnancies = self:getParent():getStorage():getData("sexbound").pregnant
    if index then
        return pregnancies[index]
    end
    return pregnancies
end

--- Returns the count of current pregnancies for this actor
function Sexbound.Actor.Pregnant:getCurrentPregnancyCount()
    local pregnancies = self:getParent():getStorage():getData("sexbound").pregnant or {}
    local count = 0
    for k, v in pairs(pregnancies) do
        count = count + 1
    end
    return count
end

--- Ensures compatibleSpecies is set to an allowed value
-- @param value
function Sexbound.Actor.Pregnant:validateCompatibleSpecies(value)
    if type(value) ~= "table" then
        self._config.compatibleSpecies = {}
        return
    end

    self._config.compatibleSpecies = value
end

--- Ensures enableAsexualReproduction is set to an allowed value
-- @param value
function Sexbound.Actor.Pregnant:validateEnableAsexualReproduction(value)
    if type(value) ~= "boolean" then
        self._config.enableAsexualReproduction = false
        return
    end

    self._config.enableAsexualReproduction = value
end

--- Ensures enableCompatibleSpeciesOnly is set to an allowed value
-- @param value
function Sexbound.Actor.Pregnant:validateEnableCompatibleSpeciesOnly(value)
    if type(value) ~= "boolean" then
        self._config.enableCompatibleSpeciesOnly = false
        return
    end

    self._config.enableCompatibleSpeciesOnly = value
end

--- Ensures enableFreeForAll is set to an allowed value
-- @param value
function Sexbound.Actor.Pregnant:validateEnableFreeForAll(value)
    if type(value) ~= "boolean" then
        self._config.enableFreeForAll = false
        return
    end

    self._config.enableFreeForAll = value
end

--- Ensures enableMultipleImpregnations is set to an allowed value
-- @param value
function Sexbound.Actor.Pregnant:validateEnableMultipleImpregnations(value)
    if type(value) ~= "boolean" then
        self._config.enableMultipleImpregnations = false
        return
    end

    self._config.enableMultipleImpregnations = value
end

--- Ensures enableNotifyPlayers is set to an allowed value
-- @param value
function Sexbound.Actor.Pregnant:validateEnableNotifyPlayers(value)
    if type(value) ~= "boolean" then
        self._config.enableNotifyPlayers = true
        return
    end
    self._config.enableNotifyPlayers = value
end

--- Ensures enablePregnancyFetish is set to an allowed value
-- @param value
function Sexbound.Actor.Pregnant:validateEnablePregnancyFetish(value)
    if type(value) ~= "boolean" then
        self._config.enablePregnancyFetish = false
        return
    end
    self._config.enablePregnancyFetish = value
end

--- Ensures enableSilentImpregnations is set to an allowed value
function Sexbound.Actor.Pregnant:validateEnableSilentImpregnations(value)
    if type(value) ~= "boolean" then
        self._config.enableSilentImpregnations = false
        return
    end
    self._config.enableSilentImpregnations = value
end

--- Ensures fertility is set to an allowed value
-- @param value
function Sexbound.Actor.Pregnant:validateFertility(value)
    if type(value) ~= "number" then
        self._config.fertility = 0.75
        return
    end
    self._config.fertility = util.clamp(value, 0, 1)
end

--- Ensures notifications is set to an allowed value
-- @param value
function Sexbound.Actor.Pregnant:validateNotifications(value)
    if type(value) ~= "string" then
        self._config.notifications = "/dialog/sexbound/en/notifications.config"
        return
    end
    self._config.notifications = value
end

--- Ensures preventStatuses are set to an allowed value
-- @param value
function Sexbound.Actor.Pregnant:validatePreventStatuses(value)
    local defaultPreventStatuses = {"birthcontrol", "equipped_nopregnant", "equipped_iud", "infertile", "sterile"}

    if type(value) ~= "table" then
        self._config.preventStatuses = defaultPreventStatuses
        return
    end

    self._config.preventStatuses = {}
    for _, v in ipairs(value) do
        if type(v) == "string" then
            table.insert(self._config.preventStatuses, v)
        end
    end
end

--- Ensures trimesterCount is set to an allowed value
-- @param value
function Sexbound.Actor.Pregnant:validateTrimesterCount(value)
    if type(value) ~= "number" then
        self._config.trimesterCount = 3
        return
    end
    self._config.trimesterCount = util.clamp(value, 1, value)
end

--- Ensures trimesterLength is set to an allowed value
-- @param value
function Sexbound.Actor.Pregnant:validateTrimesterLength(value)
    if type(value) == "number" then
        self._config.trimesterLength = util.clamp(value, 1, value)
        return
    end
    if type(value) == "table" then
        local lo = value[1]
        local hi = value[2]
        if type(lo) ~= "number" then
            lo = 2
        end
        if type(hi) ~= "number" then
            hi = 3
        end
        self._config.trimesterLength = {}
        self._config.trimesterLength[1] = util.clamp(lo, 1, lo)
        self._config.trimesterLength[2] = util.clamp(hi, 1, hi)
        return
    end
    self._config.trimesterLength = {}
    self._config.trimesterLength[1] = 2
    self._config.trimesterLength[2] = 3
end

--- Ensures whichGendersCanOvulate is set to an allowed value
-- @param value
function Sexbound.Actor.Pregnant:validateWhichGendersCanOvulate(value)
    if type(value) ~= "table" then
        self._config.whichGendersCanOvulate = {"female"}
        return
    end
    self._config.whichGendersCanOvulate = {}
    for _, v in ipairs(value) do
        if type(v) == "string" then
            table.insert(self._config.whichGendersCanOvulate, v)
        end
    end
end

--- Ensures whichGendersCanProduceSperm is set to an allowed value
-- @param value
function Sexbound.Actor.Pregnant:validateWhichGendersCanProduceSperm(value)
    if type(value) ~= "table" then
        self._config.whichGendersCanProduceSperm = {"male"}
        return
    end
    self._config.whichGendersCanProduceSperm = {}
    for _, v in ipairs(value) do
        if type(v) == "string" then
            table.insert(self._config.whichGendersCanProduceSperm, v)
        end
    end
end

--- Ensures useOSTimeForPregnancies is set to an allowed value
-- @param value
function Sexbound.Actor.Pregnant:validateUseOSTimeForPregnancies(value)
    if type(value) ~= "boolean" then
        self._config.useOSTimeForPregnancies = true
        return
    end
    self._config.useOSTimeForPregnancies = value
end
