--- Sexbound.Player Module.
-- @module Sexbound.Player
require "/scripts/sexbound/override/common.lua"

Sexbound.Player = Sexbound.Common:new()
Sexbound.Player_mt = {
    __index = Sexbound.Player
}

local SexboundErrorCounter = 0

--- Override Hook (init)
local Sexbound_Old_Init = init
function init()
    xpcall(function()
        Sexbound_Old_Init()

        self.sb_player = Sexbound.Player.new()
    end, sb.logError)
end

--- Override Hook (update)
local Sexbound_Old_Update = update
function update(dt)
    xpcall(function()
        Sexbound_Old_Update(dt)
    end, sb.logError)

    if SexboundErrorCounter < 5 then
        xpcall(function()
            self.sb_player:update(dt)
        end, function(err)
            SexboundErrorCounter = SexboundErrorCounter + 1

            sb.logError(err)
        end)
    end
end

function Sexbound.Player.new()
    local self = setmetatable({
        _controllerId = nil,
        _isHavingSex = false,
        _startItemsList = {"sexbound1-codex", "sexboundcustomizer"},
        _loungeId = nil,
        _states = {"defaultState", "havingSexState"}
    }, Sexbound.Player_mt)

    self:init(self, "player")

    -- Init. the sexboundConfig in the storage
    storage.sexboundConfig = storage.sexboundConfig or {
        hasGivenStartItems = false
    }

    self._legal = Sexbound.Player.Legal:new(self)
    self._legal:verify()

    self._identity = Sexbound.Player.Identity:new(self)

    self:initMessageHandlers()

    -- Remove Sexbound specific status effects when the player begins the game
    status.setStatusProperty("sexbound_sex", false)

    status.setStatusProperty("sexbound_abortion", false)

    -- Instantiate player override classes manually; Need to change to be dynamic
    self._apparel = Sexbound.Player.Apparel:new(self)
    self._arousal = Sexbound.Player.Arousal:new(self)
    self._climax = Sexbound.Player.Climax:new(self)
    self._subGender = Sexbound.Player.SubGender:new(self)
    self._pregnant = Sexbound.Player.Pregnant:new(self)
    self._statistics = Sexbound.Player.Statistics:new(self)
    self._transform = Sexbound.Player.Transform:new(self)

    -- Restore the Previous Storage
    self:restorePreviousStorage()

    -- Do not check if false because current players won't have this property set
    if not storage.sexboundConfig.hasGivenStartItems then
        storage.sexboundConfig.hasGivenStartItems = self:giveStartItems()
    end

    -- Set Universe Flags
    player.setUniverseFlag("sexbound_installed")

    -- Load StateMachine
    self._stateMachine = self:helper_loadStateMachine()

    return self
end

-- [Helper] Loads the stateMachine
-- See state definitions at the bottom of the script
function Sexbound.Player:helper_loadStateMachine()
    self._stateDefinitions = {
        defaultState = self:defineStateDefaultState(),
        havingSexState = self:defineStateHavingSexState()
    }

    return stateMachine.create(self._states, self._stateDefinitions)
end

--- Updates the Player
-- @param dt
function Sexbound.Player:update(dt)
    -- Update the legal verification system
    self._legal:update(dt)

    -- Update the Player's Statemachine
    self._stateMachine.update(dt)

    -- Only the Player needs to update pregnancy via script. NPCs and Monsters update via AI.
    self._pregnant:update(dt)
end

function Sexbound.Player:handleEnterClimaxState(args)
    return
end

function Sexbound.Player:handleEnterIdleState(args)
    -- Nothing
end

function Sexbound.Player:handleEnterSexState(args)
    -- Nothing
end

function Sexbound.Player:handleLounge(args)
    self._loungeId = args.loungeId

    -- Lounge the player in the object's first anchor
    player.lounge(self._loungeId, 0)
end

function Sexbound.Player:giveStartItems()
    -- Attempt to cycle through and give the player one of each start item
    for _, _item in ipairs(self._startItemsList) do
        if not player.hasItem(_item) then
            player.giveItem(_item)
        end
    end

    -- Check that the player actually received all of the start items
    for _, _item in ipairs(self._startItemsList) do
        if not player.hasItem(_item) then
            return false
        end
    end

    return true
end

function Sexbound.Player:handleShowCustomizerUI(args)
    return self:showCustomizerUI(args)
end

function Sexbound.Player:showCustomizerUI(args)
    -- Check if player.interact is a function because some previous version of starbound did not implement it
    if "function" ~= type(player.interact) then return end
    xpcall(function()
        local _loadedConfig = root.assetJson("/interface/sexbound/customizer/customizer.config")
        _loadedConfig.config.statistics = self:getStatistics():getStatistics()
        player.interact("ScriptPane", _loadedConfig, player.id())
    end, function(err)
        sb.logError("Unable to load config Sexbound Settings.")
    end)
end

function Sexbound.Player:handleDismissUI(args)
    return not status.statusProperty("sexbound_sex")
end

function Sexbound.Player:handleShowUI(args)
    return self:showUI(args.config)
end

function Sexbound.Player:showUI(config)
    -- Check if player.interact is a function because some previous version of starbound did not implement it
    if "function" ~= type(player.interact) then return end
    player.interact("ScriptPane", config, config.config.controllerId or player.id())
end

function Sexbound.Player:handleRestore(args)
    if args then
        self:mergeStorage(args)
    end

    self:restore()
end

function Sexbound.Player:restore()
    self._isHavingSex = false

    if storage and storage.sexbound then
        self:getClimax():setCurrentValue(storage.sexbound.climax)
    end

    -- Remove specific Sexbound status effects and unstun the player
    status.removeEphemeralEffect("sexbound_sex")
    status.removeEphemeralEffect("sexbound_invisible")
    status.removeEphemeralEffect("sexbound_stun")
    status.setStatusProperty("sexbound_stun", false)
end

function Sexbound.Player:handleRetrieveConfig(args)
    return nil
end

function Sexbound.Player:handleRetrieveStorage(args)
    if args and args.name then
        return storage[args.name]
    end

    return storage
end

function Sexbound.Player:handleSyncStorage(newData)
    storage = util.mergeTable(storage, newData or {})
end

function Sexbound.Player:initMessageHandlers()
    message.setHandler("Sexbound:Actor:Restore", function(_, _, args)
        return self:handleRestore(args)
    end)
    message.setHandler("Sexbound:Actor:Respawn", function(_, _, args)
        return self:handleRespawn(args)
    end)
    message.setHandler("Sexbound:Config:Retrieve", function(_, _, args)
        return self:handleRetrieveConfig(args)
    end)
    message.setHandler("Sexbound:ClimaxState:Enter", function(_, _, args)
        return self:handleEnterClimaxState(args)
    end)
    message.setHandler("Sexbound:CustomizerUI:Show", function(_, _, args)
        return self:handleShowCustomizerUI(args)
    end)
    message.setHandler("Sexbound:Reward:Currency", function(_, _, args)
        return self:handleRewardCurrency(args)
    end)
    message.setHandler("Sexbound:IdleState:Enter", function(_, _, args)
        return self:handleEnterIdleState(args)
    end)
    message.setHandler("Sexbound:Node:Lounge", function(_, _, args)
        return self:handleLounge(args)
    end)
    message.setHandler("Sexbound:SexState:Enter", function(_, _, args)
        return self:handleEnterSexState(args)
    end)
    message.setHandler("Sexbound:Storage:Retrieve", function(_, _, args)
        return self:handleRetrieveStorage(args)
    end)
    message.setHandler("Sexbound:Storage:Sync", function(_, _, args)
        return self:handleSyncStorage(args)
    end)
    message.setHandler("Sexbound:UI:Dismiss", function(_, _, args)
        return self:handleDismissUI(args)
    end)
    message.setHandler("Sexbound:UI:Show", function(_, _, args)
        return self:handleShowUI(args)
    end)
end

function Sexbound.Player:mergeStorage(newData)
    storage = util.mergeTable(storage, newData or {})
end

--- Setup the Player's actor data and send it to the lounge entity.
-- @param controllerId
function Sexbound.Player:setup(controllerId, params)
    self._isHavingSex = true

    if controllerId then
        local actorData = util.mergeTable(self:getActorData(), params or {})

        promises:add(world.sendEntityMessage(controllerId, "Sexbound:Actor:Setup", actorData), function(result)
            self._isHavingSex = result or false

            -- Handle the case that the Player has a quest to have sex
            world.sendEntityMessage(player.id(), "Sexbound:Quest:HaveSex")
        end)

        return true
    end

    return false
end

-- Getters / Setters

function Sexbound.Player:getActorData()
    local gender = player.gender()

    storage.sexbound.climax = self:getClimax():getCurrentValue()

    local identity = self:getIdentity():build()
    identity.sxbSubGender = self:getSubGender():getSxbSubGender()

    return {
        entityId = player.id(),
        forceRole = math.floor(status.stat("forceRole")),
        uniqueId = player.uniqueId(),
        entityType = "player",
        identity = identity,
        backwear = self:getApparel():prepareBackwear(gender),
        chestwear = self:getApparel():prepareChestwear(gender),
        groinwear = self:getApparel():prepareGroinwear(gender),
        headwear = self:getApparel():prepareHeadwear(gender),
        legswear = self:getApparel():prepareLegswear(gender),
        nippleswear = self:getApparel():prepareNippleswear(gender),
        storage = storage
    }
end

-- Getters/Setters

function Sexbound.Player:getApparel()
    return self._apparel
end
function Sexbound.Player:getArousal()
    return self._arousal
end
function Sexbound.Player:getClimax()
    return self._climax
end
function Sexbound.Player:getControllerId()
    return self._controllerId
end
function Sexbound.Player:getSubGender()
    return self._subGender
end
function Sexbound.Player:getIdentity()
    return self._identity
end
function Sexbound.Player:getLegal()
    return self._legal
end
function Sexbound.Player:getPregnant()
    return self._pregnant
end
function Sexbound.Player:getStatistics()
    return self._statistics
end
function Sexbound.Player:getTransform()
    return self._transform
end

--- Legacy functions

function Sexbound.Player:handleRespawn(args)
    if args then
        self:mergeStorage(args)
    end

    self:respawn()
end

function Sexbound.Player:handleRewardCurrency(args)
    player.addCurrency(args.currencyName, args.amount)
end

function Sexbound.Player:respawn()
    self:restore()
end

--- State Definitions

-- State Definition : defaultState
function Sexbound.Player:defineStateDefaultState()
    return {
        enter = function()
            if not self._isHavingSex then
                return {}
            end
        end,

        enteringState = function(stateData)
            self:getArousal():setRegenRate("default")
            self:getClimax():setRegenRate("default")
        end,

        update = function(dt, stateData)
            if player.isLounging() and status.statusProperty("sexbound_sex") == true then
                self._isHavingSex = true
            end

            -- Exit condition #1
            if self._isHavingSex then
                return true
            end

            self:getClimax():update(dt)
        end
    }
end

-- State Definition : havingSexState
function Sexbound.Player:defineStateHavingSexState()
    return {
        enter = function()
            if self._isHavingSex then
                return {
                    isLounging = player.isLounging(),
                    loungeId = player.loungingIn()
                }
            end
        end,

        enteringState = function(stateData)
            self:getArousal():setRegenRate("havingSex")
            self:getClimax():setRegenRate("havingSex")

            if player.isLounging() then
                self._loungeId = stateData.loungeId

                promises:add(world.sendEntityMessage(self._loungeId, "Sexbound:Retrieve:ControllerId"),
                    function(controllerId)
                        self._controllerId = controllerId

                        self:setup(self._controllerId)
                    end, function() -- Failed
                        self._isHavingSex = false
                    end)
            end
        end,

        update = function(dt, stateData)
            -- When isLounging is true then the player is probably lounging in a sexnode
            if stateData.isLounging and status.statusProperty("sexbound_sex") ~= true then
                self._isHavingSex = false
            end

            -- Exit condition #1
            if not self._isHavingSex then
                return true
            end

            -- Update the worn apparel
            self:getApparel():update(self._controllerId or self._loungeId, player.gender())
        end,

        leavingState = function(stateData)
            local controllerId = self._controllerId or self._loungeId

            if controllerId then
                world.sendEntityMessage(controllerId, "Sexbound:Actor:Remove", entity.id())
            end

            self._loungeId = nil

            self._controllerId = nil

            self:restore()
        end
    }
end
