Sexbound.Common.Pregnant = {}
Sexbound.Common.Pregnant_mt = {
    __index = Sexbound.Common.Pregnant
}

function Sexbound.Common.Pregnant:new()
    local _self = setmetatable({
        _isGivingBirth = false
    }, Sexbound.Common.Pregnant_mt)

    _self:loadPluginConfig()
    _self:updateIsTimeToGiveBirthToFunction()

    return _self
end

function Sexbound.Common.Pregnant:updateIsTimeToGiveBirthToFunction()
    if self:getConfig().useOSTimeForPregnancies then
        self.isTimeToGiveBirthTo = self.checkTimeToGiveBirthBasedOnOSTime
    else
        self.isTimeToGiveBirthTo = self.checkTimeToGiveBirthBasedOnWorldTime
    end
end

function Sexbound.Common.Pregnant:loadPluginConfig()
    local oldConfig = root.assetJson("/scripts/sexbound/plugins/pregnant/pregnant.config")
    local newConfig = root.assetJson("/sxb_plugin.pregnant.config")

    self._config = util.mergeTable(oldConfig, newConfig)
end

function Sexbound.Common.Pregnant:init(parent)
    self._parent = parent

    -- Ensure this is always a table
    if type(storage.sexbound.pregnant) ~= "table" then
        storage.sexbound.pregnant = {}
    end

    -- Migrate previous pregnant data into new storage location
    self:dataMigrate()

    -- Always filters erroneous data after data migration
    self:dataFilter()

    -- Always refresh status effects after validating data
    self:refreshStatusEffects()

    self:initMessageHandlers()
end

function Sexbound.Common.Pregnant:initMessageHandlers()
    message.setHandler("Sexbound:Pregnant:Abortion", function(_, _, args)
        return self:abortPregnancy(args)
    end)
    message.setHandler("Sexbound:Pregnant:GetData", function(_, _, args)
        return self:getData(args)
    end)
    message.setHandler("Sexbound:Pregnant:GiveBirth", function(_, _, args)
        return self:handleGiveBirth(args)
    end)
end

--- Performs data migration to preserve pregnant data from previous versions.
function Sexbound.Common.Pregnant:dataMigrate()
    if type(storage.pregnant) ~= "table" then
        return
    end
    if isEmpty(storage.pregnant) then
        storage.pregnant = nil
        return
    end

    if storage.pregnant[1] ~= nil then -- Is Multiple Pregnancies
        util.each(storage.pregnant, function(_, pregnancy)
            table.insert(storage.sexbound.pregnant, pregnancy)
        end)
    else -- Is single pregnancy
        table.insert(storage.sexbound.pregnant, storage.pregnant)
    end

    storage.pregnant = nil
end

function Sexbound.Common.Pregnant:dataFilter()
    if type(storage.sexbound.pregnant) ~= "table" then return end
    if isEmpty(storage.sexbound.pregnant) then return end

    local filteredData = {}
    local count = 0
    for k, v in pairs(storage.sexbound.pregnant) do
        if type(v) == "table" and type(v.birthDate) == "number" and type(v.birthTime) == "number" then
            v.birthWorldTime = v.birthDate + v.birthTime
        end

        if type(v) == "table" then
            v.birthOSTime = v.birthOSTime or os.time()
        end

        count = count + 1

        table.insert(filteredData, count, v)
    end

    storage.sexbound.pregnant = filteredData
end

-- Aborts all pregnancies by reseting storage to empty table
function Sexbound.Common.Pregnant:abortPregnancy()
    storage.sexbound.pregnant = {}

    self:refreshStatusEffects()
end

function Sexbound.Common.Pregnant:findReadyBabyIndex()
    if isEmpty(storage.sexbound.pregnant or {}) then
        return
    end

    self:updateWorldTime()

    for index, baby in pairs(storage.sexbound.pregnant) do
        if self:isTimeToGiveBirthTo(baby) == true then
            return index
        end
    end

    return
end

function Sexbound.Common.Pregnant:checkTimeToGiveBirthBasedOnOSTime(baby)
    return os.time() >= baby.birthOSTime
end

function Sexbound.Common.Pregnant:checkTimeToGiveBirthBasedOnWorldTime(baby)
    return self:updateWorldTime() >= baby.birthWorldTime
end

function Sexbound.Common.Pregnant:giveBirth(babyConfig)
    babyConfig.birthEntityGroup = babyConfig.birthEntityGroup or "humanoid"

    if babyConfig.birthEntityGroup == "monsters" then
        return self:_giveBirthToMonster(babyConfig)
    else
        return self:_giveBirthToHumanoid(babyConfig)
    end
end

function Sexbound.Common.Pregnant:refreshStatusEffects()
    if self:getIsPregnant() then
        self:addStatusEffects()
        return
    end

    self:removeStatusEffects()
end

function Sexbound.Common.Pregnant:addStatusEffects()
    status.addEphemeralEffect("sexbound_pregnant", math.huge)
end

function Sexbound.Common.Pregnant:removeStatusEffects()
    status.removeEphemeralEffect("sexbound_pregnant")
end

function Sexbound.Common.Pregnant:updateWorldTime()
    self._worldTime = world.day() + world.timeOfDay()
    return self._worldTime
end

function Sexbound.Common.Pregnant:_convertBabyConfigToSpawnableMonster(babyConfig)
    local params = {}
    params.baseParameters = {}
    params.baseParameters.uniqueId = sb.makeUuid()
    params.baseParameters.statusSettings = {}
    params.baseParameters.statusSettings.statusProperties = {
        sexbound_birthday = babyConfig
    }
    params = util.mergeTable(params, babyConfig.birthParams or {})
    return {
        params   = params,
        position = babyConfig.birthPosition or entity.position(),
        type     = babyConfig.birthSpecies  or "gleap"
    }
end

function Sexbound.Common.Pregnant:_convertBabyConfigToSpawnableNPC(babyConfig)
    local params = {}
    params.scriptConfig = {}
    params.scriptConfig.uniqueId = sb.makeUuid()
    params.statusControllerSettings = {}
    params.statusControllerSettings.statusProperties = {
        sexbound_birthday = babyConfig,
        sexbound_previous_storage = {
            previousDamageTeam = storage.previousDamageTeam
        }
    }
    util.mergeTable(params, babyConfig.birthParams or {})
    local spawnableNPC = {
        level    = babyConfig.birthLevel or 1,
        npcType  = "crewmembersexbound",
        params   = params,
        position = babyConfig.birthPosition or entity.position(),
        seed     = babyConfig.birthSeed,
        species  = babyConfig.birthSpecies or "human"
    }

    if npc then spawnableNPC.npcType = npc.npcType() end
    if monster then spawnableNPC.npcType = "villager" end

    return spawnableNPC
end

function Sexbound.Common.Pregnant:_giveBirthToHumanoid(babyConfig)
    local spawnableNPC = self:_convertBabyConfigToSpawnableNPC(babyConfig)
    return world.spawnNpc(
        spawnableNPC.position,
        spawnableNPC.species,
        spawnableNPC.npcType,
        spawnableNPC.level,
        spawnableNPC.seed,
        spawnableNPC.params
    )
end

function Sexbound.Common.Pregnant:_giveBirthToMonster(babyConfig)
    local spawnableMonster = self:_convertBabyConfigToSpawnableMonster(babyConfig)
    return world.spawnMonster(
        spawnableMonster.type,
        spawnableMonster.position,
        spawnableMonster.params
    )
end

function Sexbound.Common.Pregnant:getConfig()
    return self._config
end

function Sexbound.Common.Pregnant:getData(index)
    if index and storage.sexbound.pregnant then
        return storage.sexbound.pregnant[index]
    end
    return storage.sexbound.pregnant
end

function Sexbound.Common.Pregnant:getIsPregnant()
    return not isEmpty(storage.sexbound.pregnant)
end

function Sexbound.Common.Pregnant:isGivingBirth()
    return self._isGivingBirth
end

function Sexbound.Common.Pregnant:setIsGivingBirth(isGivingBirth)
    self._isGivingBirth = isGivingBirth
end

function Sexbound.Common.Pregnant:getParent()
    return self._parent
end

function Sexbound.Common.Pregnant:getWorldTime()
    return self._worldTime or self:updateWorldTime()
end
