if SXB_RUN_TESTS ~= true then
  require "/scripts/util.lua"
  require "/scripts/messageutil.lua"
  require "/scripts/sexbound/util.lua"
end

local PregnancyTest = {}
local PregnancyTest_mt = { __index = PregnancyTest }

function PregnancyTest:new()
  return setmetatable({
    entityId = activeItem.ownerEntityId()
  }, PregnancyTest_mt)
end

function PregnancyTest:update(dt) promises:update() end

function PregnancyTest:activate(fireMode, shiftHeld)
  promises:add(
    world.sendEntityMessage(self.entityId, "Sexbound:Pregnant:GetData"),
    function(pregnancies)
      return self:handlePregnancyData(pregnancies)
    end
  )
end

function PregnancyTest:handlePregnancyData(pregnancies)
  if not self:validatePregnanciesIsTable(pregnancies)
  or not self:thisEntityHasAtLeastOnePregnancy(pregnancies)
  then self:notifyThisEntityTheyAreNotPregnant()
  else self:notifyThisEntityTheyArePregnant(pregnancies) end
end

function PregnancyTest:buildTimingMeridian(hour)
  if hour <= 12 then return "AM" else return "PM" end
end

function PregnancyTest:buildFormattedBirthTimeString(hour, minutes)
  local meridian = self:buildTimingMeridian(hour)
  local time = string.format("%s:%s %s", hour, minutes, meridian)
  return "^red;" .. time .. "^reset;"
end

function PregnancyTest:buildBirthTimeBasedOnOSTime(birthOSTime)
  if birthOSTime == nil then return "" end
  local birthDateTimestamp = math.floor(birthOSTime + 0) -- convert to number
  local todayDateTimestamp = os.time()
  local diffTime = math.max(0, os.difftime(birthDateTimestamp, todayDateTimestamp))

  if (diffTime <= 86400) then
    local totalMinutes = diffTime / 60
    local hours   = math.floor(totalMinutes / 60)
    hours = "^red;" .. hours .. "^reset;"

    local minutes = math.floor(totalMinutes % 60)
    minutes = "^red;" .. minutes .. "^reset;"

    local text = config.getParameter("radioMessages.birthTodayBasedOnOSTime")
    text = util.replaceTag(text, "hours", hours)
    text = util.replaceTag(text, "minutes", minutes)
    text = text .. " - or - "
    return text
  else
    local days = math.floor((birthDateTimestamp - todayDateTimestamp) / 86400)
    days = "^red;" .. days .. "^reset;"
    local text = config.getParameter("radioMessages.birthLaterBasedOnOSTime")
    text = util.replaceTag(text, "days", days)
    text = text .. " - or - "
    return text
  end
end

function PregnancyTest:buildBirthTimeBasedOnWorldTime(birthTime)
  local days = math.floor(birthTime)

  birthTime = birthTime - days

  days = math.max(0, days - world.day())

  local hours = birthTime
  hours = math.floor(24 * hours)

  -- local meridian = self:buildTimingMeridian(hours)
  local minutes = math.floor(60 * (24 * birthTime - hours))
  if minutes <= 9 then minutes = minutes .. '0' end

  if hours > 12 then hours = hours - 12 end

  local time = self:buildFormattedBirthTimeString(hours, minutes)

  if days > 0 then
    local text = config.getParameter("radioMessages.birthLaterBasedOnWorldTime")
    days = "^red;" .. days .. "^reset;"
    text = util.replaceTag(text, "days", days)
    return text
  else
    local text = config.getParameter("radioMessages.birthTodayBasedOnWorldTime")
    text = util.replaceTag(text, "time", time)
    return text
  end
end

function PregnancyTest:notifyThisEntityTheyArePregnant(pregnancies)
  local baby = pregnancies[1]
  local birthWorldTime = baby.birthWorldTime
  local birthOSTime = baby.birthOSTime

  animator.setAnimationState("result", "positive", true)
  local text = "You're pregnant!\n\n"
  text = text .. self:buildBirthTimeBasedOnOSTime(birthOSTime)
  text = text .. self:buildBirthTimeBasedOnWorldTime(birthWorldTime)
  local symbol = "(^green;+^reset;) "
  text = symbol .. text
  self:queueRadioMessageForThisEntity("PregnancyTest", text)
end

function PregnancyTest:notifyThisEntityTheyAreNotPregnant()
  local text = config.getParameter("radioMessages.notPregnant")
  local symbol = "(^red;-^reset;) "
  text = symbol .. text
  animator.setAnimationState("result", "negative", true)
  self:queueRadioMessageForThisEntity("PregnancyTest", text)
end

function PregnancyTest:queueRadioMessageForThisEntity(messageId, text)
  local entityId = activeItem.ownerEntityId()
  local sendEntityMessageOptions = { messageId = messageId, unique = false, text = text }
  return world.sendEntityMessage(entityId, "queueRadioMessage", sendEntityMessageOptions)
end

function PregnancyTest:validatePregnanciesIsTable(pregnancies)
  return type(pregnancies) == "table"
end

function PregnancyTest:thisEntityHasAtLeastOnePregnancy(pregnancies)
  return #pregnancies >= 1
end

-- Active Item Hooks --

function init()
  self.pregnancyTest = PregnancyTest:new()
end

function update(dt)
  self.pregnancyTest:update(dt)
end

function activate(fireMode, shiftHeld)
  self.pregnancyTest:activate(fireMode, shiftHeld)
end