require "/scripts/util.lua"

require "/objects/aphrodite/aphroditesdancetable/scripts/flamethrower.lua"

--- Aphrodite Class Module.
-- @classmod Aphrodite
-- @author Loxodon
Aphrodite = {}
Aphrodite_mt = {__index = Aphrodite}

function Aphrodite:new( parent )
  local self = setmetatable({
    _actualAngle  = 0,
    _currentAngle = 0,
    _direction = "down",
    _parent = parent,
    _rotationCenter = {0.25, -0.75},
    _rotationSpeed = 45,
    _states = { "rockState", "flamethrowerUpState", "flamethrowerDownState" },
    _status = "rock",
    _timers = {}
  }, Aphrodite_mt)
  
  self._flamethrower = Flamethrower:new( self )
  
  return self
end

function Aphrodite:init()
  self._flamethrower:init()
  
  self._stateDefinitions = {
    rockState = {
      enter = function()
        if self._status == "rock" then
          return {}
        end
      end,
      
      enteringState = function(stateData)
        animator.resetTransformationGroup("fauxActorArms")

        self._flamethrower:setStatus("off")
      end,
      
      update = function(dt, stateData)
        -- Exit condidtion
        if self._status ~= "rock" then
          return true
        end
      end
    },
    
    flamethrowerUpState = {
      enter = function()
        if self._status == "flamethrower" and self._direction == "up" then
          return {}
        end
      end,
      
      enteringState = function(stateData)
        self:resetArmsUp()
        
        self:tryTurningOnFlamethrower()
      end,
      
      update = function(dt, stateData)
        -- Exit condidtion
        if self._status ~= "flamethrower" or self._direction ~= "up" then
          return true
        end
        
        self:rotateArmsUp(dt)
      end
    },
    
    flamethrowerDownState = {
      enter = function()
        if self._status == "flamethrower" and self._direction == "down" then
          return {}
        end
      end,
      
      enteringState = function(stateData)
        self:resetArmsDown()
        
        self:tryTurningOnFlamethrower()
      end,
      
      update = function(dt, stateData)
        -- Exit condition
        if self._status ~= "flamethrower" or self._direction ~= "down" then return true end

        self:rotateArmsDown(dt)
      end
    },
  }
  
  self._stateMachine = stateMachine.create(self._states, self._stateDefinitions)
end

function Aphrodite:update(dt)
  self._stateMachine.update(dt)
  self._flamethrower:update(dt)
end

function Aphrodite:aimVector(inaccuracy)
  local aimVector = vec2.rotate({1, 0}, self._actualAngle + sb.nrand(inaccuracy, 0))
  aimVector[1] = aimVector[1] * self._parent._facingDir
  return aimVector
end

function Aphrodite:addTimer(name, amount) self._timers[name] = self._timers[name] + amount end
function Aphrodite:getTimer(name)         return self._timers[name] end
function Aphrodite:setTimer(name, value)  self._timers[name] = value end
function Aphrodite:setStatus(name)        self._status = name end

function Aphrodite:tryTurningOnFlamethrower()
  if self._flamethrower:getStatus() == "off" then
    self._flamethrower:setStatus("on")
  end
end

function Aphrodite:resetArmsDown()
  self._currentAngle = 90
  self._actualAngle  = util.toRadians(self._currentAngle)
  animator.resetTransformationGroup("fauxActorArms")
  animator.rotateTransformationGroup("fauxActorArms", self._actualAngle, self._rotationCenter)
end

function Aphrodite:resetArmsUp()
  self._currentAngle = -45
  self._actualAngle  = util.toRadians(360 + self._currentAngle)
  animator.resetTransformationGroup("fauxActorArms")
  animator.rotateTransformationGroup("fauxActorArms", self._actualAngle, self._rotationCenter)
end

function Aphrodite:rotateArmsDown(dt)
  local rotAmount = self._rotationSpeed * dt  
  local tempArmsRotation = self._currentAngle - rotAmount 
  
  if tempArmsRotation < -45 then
     rotAmount = rotAmount + tempArmsRotation
     
     self._direction = "up"
  end

  self._currentAngle = self._currentAngle - rotAmount
  
  local changeAmount = -util.toRadians(util.wrapAngle(rotAmount))
  
  self._actualAngle  = util.wrapAngle(self._actualAngle + changeAmount)
  
  if self._currentAngle >= -45 then
    animator.rotateTransformationGroup("fauxActorArms", changeAmount, self._rotationCenter)
  end
end

function Aphrodite:rotateArmsUp(dt)
  local rotAmount = self._rotationSpeed * dt
  local tempArmsRotation = self._currentAngle + rotAmount 
  
  if tempArmsRotation > 45 then
     local diff = tempArmsRotation - 45
     
     rotAmount = rotAmount - diff
     
     self._direction = "down"
  end
  
  self._currentAngle = self._currentAngle + rotAmount
  
  local changeAmount = util.toRadians(util.wrapAngle(rotAmount))
  
  self._actualAngle  = util.wrapAngle(self._actualAngle + changeAmount)
  
  if self._currentAngle <= 45 then
    animator.rotateTransformationGroup("fauxActorArms", changeAmount, self._rotationCenter)
  end
end