require "/scripts/sexbound/v2/api.lua"
require "/scripts/util.lua"
require "/scripts/stateMachine.lua"

require "/objects/aphrodite/aphroditesdancetable/scripts/aphrodite.lua"
require "/objects/aphrodite/aphroditesdancetable/scripts/arena.lua"
require "/objects/aphrodite/aphroditesdancetable/scripts/platform.lua"

--- AphroditesDanceTable Class Module.
-- @classmod AphroditesDanceTable
-- @author Loxodon
AphroditesDanceTable = {}
AphroditesDanceTable_mt = {__index = AphroditesDanceTable}

function AphroditesDanceTable:new()
  local self = setmetatable({
    _destroyPowerCrystalCount = 0,
    _canFuck   = false,
    _facingDir = 1,
    _fauxActorConfig = config.getParameter("actorConfig.aphroditeGogo"),
    _cooldowns = config.getParameter("cooldowns"),
    _currentOccupant = nil,
    _mouthOffsetY = config.getParameter("mouthPosition")[2],
    _outputNodeIndex = 0,
    _platform = Platform:new(),
    _powerCrystals = config.getParameter("powerCrystals"),
    _stage  = 0,
    _states = { "idleState", "introState", "vulnerableState", "haveSexState", "battleState", "completeState" },
    _status = "idle",
    _timers = config.getParameter("timers")
  }, AphroditesDanceTable_mt)
  
  self._aphrodite = Aphrodite:new( self )
  
  self._arena = Arena:new( self )
  
  self:refreshCooldowns()
  
  return self
end

function AphroditesDanceTable:setDialogSequence(dialogList, wait)
  self._dialogList = dialogList
  
  self._dialogWait = wait
  
  self._nextDialog = 1
  
  self:setTimer("dialog", 0)
end

-- Returns true when finished.
function AphroditesDanceTable:sayNext()
  animator.setAnimationState("fauxActorEmote", "blabber", true)

  self:setTimer("blabber", 0)
  
  object.say(self._dialogList[self._nextDialog])
  
  if self._nextDialog < #self._dialogList then
    self._nextDialog = self._nextDialog + 1
  else
    return false
  end
  
  return true
end

function AphroditesDanceTable:sayRandom(dialogList)
  animator.setAnimationState("fauxActorEmote", "blabber", true)

  self:setTimer("blabber", 0)
  
  object.say(util.randomChoice(dialogList))
end

function AphroditesDanceTable:init()
  self._aphrodite:init() -- Initialize Aphrodite's Controller

  self._arena:init()     -- Initialize the Arena Controller
  
  self._platform:init()  -- Initialise the Platform Controller
  
  self._stateDefinitions = {
    -- IDLE STATE 
    idleState = {
      enter = function()
        if self._status == "idle" then
          return {}
        end
      end,
      
      enteringState = function(stateData)
        self:reset()
      end,
      
      update = function(dt, stateData)
        -- Exit condition : When status is not set to 'idle'
        if self._status ~= "idle" then
          return true
        end
      end
    },
    
    -- INTRO STATE
    introState = {
      enter = function()
        if self._status == "intro" then
          return {}
        end
      end,
      
      enteringState = function(stateData)
        object.setOutputNodeLevel(self._outputNodeIndex, true)

        self:beamInActor()

        self._platform:setStatus("raising")
        
        self:setDialogSequence(config.getParameter("dialog.intro"), 4)
      end,
      
      update = function(dt, stateData)
        -- Exit condition : When status is not set to 'battle' or stage is not set to 0
        if self._status ~= "intro" then
          return true
        end
        
        self:updateMouthPositon(0)
        
        self:tryChangeFacingDirection(dt)
        
        self:addToTimer("dialog", dt)
        
        if self:getTimer("dialog") >= self._dialogWait then
            if not self:sayNext() then self._status = "battle" end
            
            self:setTimer("dialog", 0)
        end
      end
    },
    
    -- BATTLE STATE
    battleState = {
      enter = function()
        if self._status == "battle" then
          return {}
        end
      end,
      
      enteringState = function(dt, stateData)
        self._stage = self._stage + 1
      
        self:showFauxActor()

        animator.setGlobalTag("beamColor", "?hueshift=0")
        
        self:updateMouthPositon(0)
        
        self:sayRandom(config.getParameter("dialog.battle")[self._stage])
        
        if self._stage == 1 then
          self:spawnPowerCrystal(self._powerCrystals[1])
        elseif self._stage == 2 then
          self:spawnPowerCrystal(self._powerCrystals[1])
          self:spawnPowerCrystal(self._powerCrystals[2])
        else
          self._aphrodite:setStatus("flamethrower")
          self:spawnPowerCrystal(self._powerCrystals[1])
          self:spawnPowerCrystal(self._powerCrystals[2])
          self:spawnPowerCrystal(self._powerCrystals[3])
          self:spawnPowerCrystal(self._powerCrystals[4])
        end
        
        self:setTimer("taunt", 0)
        
        self._platform:setStatus("raising")
        
        self._arena:setStatus("attack")
      end,
      
      update = function(dt, stateData)
        -- Exit condition
        if self._status ~= "battle" then
          return true
        end
        
        self._platform:update(dt)
        
        self._arena:update(dt)
        
        self:updateMouthPositon(0)
        
        self:tryTaunt(dt)
        
        self:tryChangeFacingDirection(dt)
      end,
      
      leavingState = function(stateData)
        self._aphrodite:setStatus("rock")
      
        self._arena:setStatus("wait")
      end
    },
    
    -- HAVE SEX STATE
    haveSexState = {
      enter = function()
        if self._status == "havingSex" then
          return {}
        end
      end,
      
      enteringState = function(stateData)
        self:hideFauxActor()
      
        self:applyStatusToOccupant("aphrodite_regen", config.getParameter("regenTimeout"))
      end,
      
      update = function(dt, stateData)
        if self._status ~= "havingSex" or not world.loungeableOccupied(entity.id()) then return true end

        if self._stage < 3 then self:processVulnerable(dt) end
        
        self:flashPlatformBeam(dt)
        
        self._platform:update(dt)
        
        self:updateMouthPositon()
      end,
      
      leavingState = function(stateData)
        self:applyStatusToOccupant("aphrodite_regen_expire", 1)
        
        if     self._canFuck and self._stage < 3 then
          self._status = "vulnerable"
        elseif self._canFuck and self._stage >= 3 then
          self._status = "complete"
        else
          self._status = "battle"
        end
      
        Sexbound.API.Actors.uninitActors()
      end
    },
    
    -- VULNERABLE STATE
    vulnerableState = {
      enter = function()
        if self._status == "vulnerable" then
          return {}
        end
      end,
      
      enteringState = function(stateData)
        self._canFuck = true

        self:updateMouthPositon(0)
        
        self:sayRandom(config.getParameter("dialog.vulnerable")[self._stage])

        object.setInteractive(self._canFuck)
        
        self:showFauxActor()
      end,
      
      update = function(dt, stateData)
        if self._status ~= "vulnerable" then
          return true
        end
        
        if world.loungeableOccupied(entity.id()) then
          self._status = "havingSex"
          return true
        end
        
        self:flashPlatformBeam(dt)
        
        if self._stage < 3 then self:processVulnerable(dt) end
        
        self._platform:update(dt)
        
        self:updateMouthPositon(0)
        
        self:tryChangeFacingDirection(dt)
      end
    },
    
    -- COMPLETE STATE
    completeState = {
      enter = function()
        if self._status == "complete" then
          return {}
        end
      end,
      
      enteringState = function(stateData)
        object.setOutputNodeLevel(1, true)  -- Open the exit door
      
        self:setDialogSequence(config.getParameter("dialog.complete"), 4)
        
        object.setInteractive(self._canFuck)
        
        self:showFauxActor()
      end,
      
      update = function(dt, stateData)
        if world.loungeableOccupied(entity.id()) then
          self._status = "havingSex"
          return true
        end
        
        self._platform:update(dt)
        
        self:flashPlatformBeam(dt)
        
        self:updateMouthPositon(0)
        
        self:tryChangeFacingDirection(dt)
        
        if not self._canFuck then
          self:addToTimer("dialog", dt)
          
          if self:getTimer("dialog") >= self._dialogWait then
              if not self:sayNext() then 
                self._canFuck = true
                
                object.setInteractive(self._canFuck)
              end
              
              self:setTimer("dialog", 0)
          end
        end
      end
    }
  }
  
  self._stateMachine = stateMachine.create(self._states, self._stateDefinitions)
  
  animator.setGlobalTag("fauxActorDirectives", self._fauxActorConfig.identity.bodyDirectives .. self._fauxActorConfig.identity.hairDirectives)
  
  self:initMessageHandlers()
end

function AphroditesDanceTable:update(dt)
  Sexbound.API.update(dt)       -- Update Sexbound

  self._stateMachine.update(dt) -- Update this object's state machine
  
  self._aphrodite:update(dt)
  
  self:tryStopBlabber(dt)
end

function AphroditesDanceTable:tryStopBlabber(dt)
  if animator.animationState("fauxActorEmote") == "blabber" then
    self:addToTimer("blabber", dt)
    
    if self:getTimer("blabber") >= self:getCooldown("blabber") then
      animator.setAnimationState("fauxActorEmote", "none", true)
      
      self:setTimer("blabber", 0)
    end
  end
end

function AphroditesDanceTable:updateMouthPositon(mouthXPos)
  if mouthXPos == nil then
    mouthXPos = config.getParameter("mouthPosition")[1]
  end
  
  object.setConfigParameter("mouthPosition", {mouthXPos, self._mouthOffsetY + self._platform:getPosition()})
end

--- Used for applying Aphrodite's regen and removing it to lounging players.
function AphroditesDanceTable:applyStatusToOccupant(name, duration, sourceEntityId)
  if self._currentOccupant then
    world.sendEntityMessage(self._currentOccupant, "applyStatusEffect", name, duration, sourceEntityId)
  end
end

function AphroditesDanceTable:flashPlatformBeam(dt)
  self:addToTimer("beamFlash", dt)

  animator.setGlobalTag("beamColor", "?hueshift=" .. util.lerp(self:getTimer("beamFlash") / self:getCooldown("beamFlash"), 0, 360))
  
  if self:getTimer("beamFlash") >= self:getCooldown("beamFlash") then
    self:setTimer("beamFlash", 0)
  end
end

--- Attempt to fill boss chest with treasure.
function AphroditesDanceTable:fillTreasureChest()
  world.sendEntityMessage("aphroditeschest", "fill", root.createTreasure(config.getParameter("bossTreasure"), 1))
end

function AphroditesDanceTable:hideFauxActor()
  animator.setAnimationState("fauxActor", "off", false)
end

function AphroditesDanceTable:beamInActor()
  animator.setAnimationState("fauxActor", "teleport", true)
end

function AphroditesDanceTable:showFauxActor()
  animator.setAnimationState("fauxActor", "on", false)
end

function AphroditesDanceTable:initMessageHandlers()
  message.setHandler("destroypowercrystal", function(_,_,args)
    self:handleDestroyPowerCrystal(args)
  end)
end

function AphroditesDanceTable:handleDestroyPowerCrystal(args)
  self:incrementPowerCrystalCount()
  
  local result = false
  
  if     self._stage == 1 and self._destroyPowerCrystalCount == 1 then
    result = true
  elseif self._stage == 2 and self._destroyPowerCrystalCount == 3 then
    result = true
  elseif self._stage == 3 and self._destroyPowerCrystalCount == 7 then
    result = true
  end
  
  if result then
    if self._stage < 3 then 
      self._status = "vulnerable"
    else
      self._status = "complete"
      
      self:fillTreasureChest() -- Called here because the complete state can be entered more than once.
    end
    
    self._platform:setStatus("lowering") -- Lower the platform
  end
end

function AphroditesDanceTable:incrementPowerCrystalCount()
  self._destroyPowerCrystalCount = self._destroyPowerCrystalCount + 1
end

function AphroditesDanceTable:handleInteraction(args)
  if self._status == "vulnerable" or self._status == "complete" then
    self._currentOccupant = args.sourceId
    
    Sexbound.API.Actors.addActor(config.getParameter("actorConfig.aphroditeGogo"))
    
    return Sexbound.API.handleInteract(args)
  end
end

function AphroditesDanceTable:handleOnInputNodeChange(args)
  if self._status == "idle" then
    self._status = "intro" -- Set status to battle to kick off the boss fight
  end
end

function AphroditesDanceTable:processVulnerable(dt)
  self:addToTimer("vulnerable", dt)
  
  if self:getTimer("vulnerable") >= self:getCooldown("vulnerable") then
    Sexbound.API.Actors.uninitActors()
  
    self._canFuck = false
  
    object.setInteractive(self._canFuck)
  
    self._status = "battle"
    
    self:setTimer("vulnerable", 0)
  end
end

function AphroditesDanceTable:reset()
  object.setOutputNodeLevel(self._outputNodeIndex, false)

  self._canFuck = false
  
  self._destroyPowerCrystalCount = 0
  
  self._stage = 0
  
  self:hideFauxActor()
  
  animator.setGlobalTag("beamColor", "?hueshift=0")
  
  self:refreshCooldowns()
end

function AphroditesDanceTable:spawnPowerCrystal(position)
  local entities = world.entityQuery(object.toAbsolutePosition(position), 1)

  for _,entityId in ipairs(entities) do
    world.sendEntityMessage(entityId, "open")
  end
end

function AphroditesDanceTable:tryTaunt(dt)
  self:addToTimer("taunt", dt)
  
  if self:getTimer("taunt") >= self:getCooldown("taunt") then
    self:sayRandom(config.getParameter("dialog.taunt")[self._stage])
    
    self:refreshCooldown("taunt")
    
    self:setTimer("taunt", 0)
  end
end

function AphroditesDanceTable:tryChangeFacingDirection(dt)
  self:addToTimer("changeFacingDirection", dt)

  if self:getTimer("changeFacingDirection") >= self:getCooldown("changeFacingDirection") then
    self._facingDir = self._facingDir * -1 -- This value is referenced within other sub-classes
  
    animator.scaleTransformationGroup("fauxActor", {-1, 1})
  
    self:refreshCooldown("changeFacingDirection")
  
    self:setTimer("changeFacingDirection", 0)
  end
end

--- Getters / Setters / Helpers

function AphroditesDanceTable:addToTimer(name, amount)
  self._timers[name] = self._timers[name] + amount
end

function AphroditesDanceTable:getCooldown(name)
  return self._cooldowns[name]
end

function AphroditesDanceTable:getTimer(name)
  return self._timers[name]
end

function AphroditesDanceTable:setTimer(name, value)
  self._timers[name] = value
end

function AphroditesDanceTable:refreshCooldown(name)
  self._cooldowns[name] = util.randomInRange(config.getParameter("cooldowns." .. name))
end

function AphroditesDanceTable:refreshCooldowns()
  local cooldowns = config.getParameter("cooldowns")
  
  for name,cooldown in pairs(cooldowns) do
    self:refreshCooldown(name)
  end
end

--- OBJECT HOOKS

function init()
  Sexbound.API.init()
  
  local sitPositions  = config.getParameter("sexboundConfig.sitPositions",  {{0,0},{0,0}})
  
  local nodePositions = config.getParameter("sexboundConfig.nodePositions", {{0,0},{0,0}})
  
  Sexbound.API.Nodes.becomeNode(nodePositions[1], sitPositions[1])
  
  self.A = AphroditesDanceTable:new()
  
  self.A:init()
end

function update(dt)
  self.A:update(dt)
end

function onInteraction(args)
  return self.A:handleInteraction(args)
end

function onInputNodeChange(args)
  self.A:handleOnInputNodeChange(args)
end

function uninit()
  Sexbound.API.uninit()
end