/* post-processing shader. draws an outline on edges determined by
 * the difference in the surface normals around the current fragment.
 */


uniform sampler2D uTexDif;
uniform sampler2D uTexDepth;
uniform sampler2D uTexRT1;

uniform vec2 uResolution;


in vec2 fsInUV;

out vec4 fsOut0;


// return range [-1,1]
vec2 sampleNrm2D(in sampler2D tex, in vec2 d)
{
	vec3 nrm = texture(tex, fsInUV + d).xyz;
	// from [0,1] to [-1,1]
	nrm = (nrm * 2.0) - vec3(1.0);

	// range [-1,1] for each
	float hDot = dot( nrm, vec3(1.0, 0.0, 0.0) );
	float vDot = dot( nrm, vec3(0.0, 1.0, 0.0) );

	return vec2(hDot, vDot);
}

vec3 sampleNrm(in sampler2D tex, in vec2 d)
{
	d = clamp( fsInUV + d, 0.001, 0.999 );
	vec3 nrm = texture(tex, d).xyz;
	// from [0,1] to [-1,1]
	return (nrm * 2.0) - vec3(1.0);
}

void getNrmSamples(inout vec3 samples[9], in sampler2D tex)
{
	float dx = 1.0 / uResolution.x;
	float dy = 1.0 / uResolution.y;

	samples[0] = sampleNrm( tex, vec2(-dx, dy) );
	samples[1] = sampleNrm( tex, vec2(0.0, dy) );
	samples[2] = sampleNrm( tex, vec2(dx, dy) );

	samples[3] = sampleNrm( tex, vec2(-dx, 0.0) );
	samples[4] = sampleNrm( tex, vec2(0.0, 0.0) );
	samples[5] = sampleNrm( tex, vec2(dx, 0.0) );

	samples[6] = sampleNrm( tex, vec2(-dx, -dy) );
	samples[7] = sampleNrm( tex, vec2(0.0, -dy) );
	samples[8] = sampleNrm( tex, vec2(dx, -dy) );
}


float sampleDepth( in sampler2D tex, in vec2 d )
{
	d = clamp( fsInUV + d, 0.001, 0.999 );
	float depth = texture(tex, d).x;
	float n = 0.1;
	float f = 1000.0;
//	depth = (2 * n) / ( f + n - depth * (f - n) );
//	depth = clamp( depth * 150.0, 0.0, 1.0 );
//	depth = pow( clamp(depth * 1.0, 0.0, 1.0), 2.0 );
	return depth;
}

void getDepthSamples( inout float samples[9], in sampler2D tex )
{
	float dx = 1.0 / uResolution.x;
	float dy = 1.0 / uResolution.y;

	samples[0] = sampleDepth( tex, vec2(-dx, dy) );
	samples[1] = sampleDepth( tex, vec2(0.0, dy) );
	samples[2] = sampleDepth( tex, vec2(dx, dy) );

	samples[3] = sampleDepth( tex, vec2(-dx, 0.0) );
	samples[4] = sampleDepth( tex, vec2(0.0, 0.0) );
	samples[5] = sampleDepth( tex, vec2(dx, 0.0) );

	samples[6] = sampleDepth( tex, vec2(-dx, -dy) );
	samples[7] = sampleDepth( tex, vec2(0.0, -dy) );
	samples[8] = sampleDepth( tex, vec2(dx, -dy) );
}


float sobelNrm3D_3x3()
{
	float f1 = 1.0;
	float f2 = 2.0;

	vec3 samples[9];
	getNrmSamples(samples, uTexRT1);

	vec3 vHoriz =
		f1*samples[6] + f2*samples[7] + f1*samples[8] -
		(f1*samples[0] + f2*samples[1] + f1*samples[2]);
	vec3 vVert =
		f1*samples[0] + f2*samples[3] + f1*samples[6] -
		(f1*samples[2] + f2*samples[5] + f1*samples[8]);

	float sobel = length(vHoriz) + length(vVert) * 1.0;

	return 1.0 - clamp(sobel, 0.0, 1.0);
}


float sobelNrm3D_2x2()
{
	vec3 samples[9];
	getNrmSamples(samples, uTexRT1);

	vec3 vHoriz = (samples[4] - samples[5]) + (samples[1] - samples[2]);
	vec3 vVert = (samples[4] - samples[1]) + (samples[5] - samples[2]);

	float sobel = length(vHoriz) + length(vVert) * 1.0;

	return 1.0 - clamp(sobel, 0.0, 1.0);
}


float nrmDeriv()
{
	vec3 samples[9];
	getNrmSamples(samples, uTexRT1);

	vec3 vdx = dFdx(samples[4]);
	vec3 vdy = dFdy(samples[4]);

	float d = max( length(vdx), length(vdy) );
	return d;
}


float nrmDiff2()
{
	vec3 samples[9];
	getNrmSamples(samples, uTexRT1);

	vec3 vdx = samples[4] - samples[1];
	vec3 vdy = samples[4] - samples[5];

	float d = length(vdx) + length(vdy);
	return d / 2.0;
}

float nrmDiff3()
{
	vec3 samples[9];
	getNrmSamples(samples, uTexRT1);

	vec3 vd1 = samples[4] - samples[1];
	vec3 vd2 = samples[4] - samples[2];
	vec3 vd3 = samples[4] - samples[5];

	float d = length(vd1) + length(vd2) + length(vd3);
	return 1.0 - clamp((d / 3.0) - 0.2, 0.0, 1.0);
}

float nrmDiff4()
{
	vec3 samples[9];
	getNrmSamples(samples, uTexRT1);

	vec3 vd1 = samples[4] - samples[1];
	vec3 vd2 = samples[4] - samples[3];
	vec3 vd3 = samples[4] - samples[5];
	vec3 vd4 = samples[4] - samples[7];

	float d = length(vd1) + length(vd2) + length(vd3) + length(vd4);
	return 1.0 - clamp((d / 4.0) - 0.2, 0.0, 1.0);
}

float nrmDiff4Diag()
{
	vec3 samples[9];
	getNrmSamples(samples, uTexRT1);

	vec3 vd1 = samples[4] - samples[0];
	vec3 vd2 = samples[4] - samples[2];
	vec3 vd3 = samples[4] - samples[6];
	vec3 vd4 = samples[4] - samples[8];

	float d = length(vd1) + length(vd2) + length(vd3) + length(vd4);
	return 1.0 - clamp((d / 4.0) - 0.2, 0.0, 1.0);
}

float nrmDiff8()
{
	vec3 samples[9];
	getNrmSamples(samples, uTexRT1);

	vec3 vd1 = samples[4] - samples[0];
	vec3 vd2 = samples[4] - samples[1];
	vec3 vd3 = samples[4] - samples[2];
	vec3 vd4 = samples[4] - samples[3];
	vec3 vd5 = samples[4] - samples[5];
	vec3 vd6 = samples[4] - samples[6];
	vec3 vd7 = samples[4] - samples[7];
	vec3 vd8 = samples[4] - samples[8];

	float d = length(vd1) + length(vd2) + length(vd3) + length(vd4) + length(vd5) + length(vd6) + length(vd7) + length(vd8);
	return 1.0 - clamp((d / 8.0) - 0.2, 0.0, 1.0);
}


float dotDiff(int i1, int i2)
{
	vec3 samples[9];
	getNrmSamples(samples, uTexRT1);

	return abs( dot(samples[4], samples[i1]) - dot(samples[4], samples[i2]) );
}

float dotDiff2()
{
	float d = dotDiff(3, 5) + dotDiff(1, 7);
	return d;
}

float dotDiff4()
{
	float d = dotDiff(3, 5) + dotDiff(1, 7) + dotDiff(0, 8) + dotDiff(2, 6);
	return 1.0 - clamp((d / 2.0) - 0.2, 0.0, 1.0);
}


float lapl8()
{
	vec3 samples[9];
	getNrmSamples(samples, uTexRT1);

	vec3 lapl = 8*samples[4] - (samples[0] + samples[1] + samples[2] + samples[3] + samples[5] + samples[6] + samples[7] + samples[8] );

	return 1.0 - clamp( length(lapl) / 4.0 - 0.3, 0.0, 1.0 );
}


float lapl3()
{
	vec3 samples[9];
	getNrmSamples(samples, uTexRT1);

	vec3 lapl = 3*samples[4] - (samples[1] + samples[2] + samples[5]);

	return 1.0 - clamp( length(lapl) / 2.5 - 0.3, 0.0, 1.0 );
}


float dot8()
{
	vec3 samples[9];
	getNrmSamples(samples, uTexRT1);

	float d =
		(1.0 - abs(dot(samples[4], samples[0]))) +
		(1.0 - abs(dot(samples[4], samples[1]))) +
		(1.0 - abs(dot(samples[4], samples[2]))) +
		(1.0 - abs(dot(samples[4], samples[3]))) +
		(1.0 - abs(dot(samples[4], samples[5]))) +
		(1.0 - abs(dot(samples[4], samples[6]))) +
		(1.0 - abs(dot(samples[4], samples[7]))) +
		(1.0 - abs(dot(samples[4], samples[8])));
	return 1.0 - clamp(pow(d / 8.0, 3.0) * 10.0, 0.0, 1.0);
}

float nrmDiff4C()
{
	vec3 samples[9];
	getNrmSamples(samples, uTexRT1);

	vec3 vd1 = samples[4] - samples[1];
	vec3 vd2 = samples[4] - samples[3];
	vec3 vd3 = samples[4] - samples[5];
	vec3 vd4 = samples[4] - samples[7];

//	float d = abs(vd1.x) + abs(vd4.x) + abs(vd2.y) + abs(vd3.y);
	float d = abs(vd1.y) + abs(vd4.y) + abs(vd2.x) + abs(vd3.x);
	d += abs(vd1.z) + abs(vd4.z) + abs(vd2.z) + abs(vd3.z);
	return 1.0 - clamp((d / 2.0) - 0.3, 0.0, 1.0);
}

float nrmDiff2C()
{
	vec3 samples[9];
	getNrmSamples(samples, uTexRT1);

	vec3 vd1 = samples[4] - samples[1];
	vec3 vd2 = samples[4] - samples[5];

	float d = abs(vd1.x) + abs(vd2.y) + abs(vd1.z) + abs(vd2.z);
//	float d = abs(vd1.x) + abs(vd2.x) + abs(vd1.y) + abs(vd2.y) + abs(vd1.z) + abs(vd2.z);
	return 1.0 - clamp((d / 2.0) - 0.3, 0.0, 1.0);
}


float nrmDiffDirPos(vec3 samples[9], int i)
{
	vec3 v = samples[4] - samples[i];
	return max(0.0, v.x + v.y + v.z);
}

float nrmDiffDirNeg(vec3 samples[9], int i)
{
	vec3 v = samples[i] - samples[4];
	return max(0.0, -1.0 * (v.x + v.y + v.z));
}

float nrmDiffDir1()
{
	vec3 samples[9];
	getNrmSamples(samples, uTexRT1);

	float dPos = nrmDiffDirPos(samples, 2);
	float dNeg = nrmDiffDirNeg(samples, 2);
//	float d = max(dPos, dNeg);
	float d = dPos;
	return 1.0 - clamp((d / 1.0), 0.0, 1.0);
}

float nrmDiffDir2()
{
	vec3 samples[9];
	getNrmSamples(samples, uTexRT1);

	float dPos = nrmDiffDirPos(samples, 1) + nrmDiffDirPos(samples, 5);
	float dNeg = nrmDiffDirNeg(samples, 1) + nrmDiffDirNeg(samples, 5);
//	float d = max(dPos, dNeg);
	float d = dPos;
	return 1.0 - clamp((d / 2.0), 0.0, 1.0);
}

float nrmDiffDir3()
{
	vec3 samples[9];
	getNrmSamples(samples, uTexRT1);

	float dPos = nrmDiffDirPos(samples, 1) + nrmDiffDirPos(samples, 2) + nrmDiffDirPos(samples, 5);
	float dNeg = nrmDiffDirNeg(samples, 1) + nrmDiffDirNeg(samples, 2) + nrmDiffDirNeg(samples, 5);
//	float d = max(dPos, dNeg);
	float d = dPos;
	return 1.0 - clamp((d / 3.0), 0.0, 1.0);
}

float nrmDiffDir4()
{
	vec3 samples[9];
	getNrmSamples(samples, uTexRT1);

	float d = nrmDiffDirPos(samples, 1) + nrmDiffDirNeg(samples, 3) + nrmDiffDirPos(samples, 5) + nrmDiffDirNeg(samples, 7);
	return 1.0 - clamp((d / 4.0), 0.0, 1.0);
}

float nrmDiffDir2Diag()
{
	vec3 samples[9];
	getNrmSamples(samples, uTexRT1);

	float d = 0.0;

	// top left
	d += clamp( samples[0].x - samples[4].x, 0.0, 1.0 );
	d += clamp( samples[4].y - samples[0].y, 0.0, 1.0 );
	// top right
	d += clamp( samples[4].x - samples[2].x, 0.0, 1.0 );
	d += clamp( samples[4].y - samples[2].y, 0.0, 1.0 );

	return 1.0 - clamp((d / 2.0), 0.0, 1.0);
}

float nrmDiffDir4Diag()
{
	vec3 samples[9];
	getNrmSamples(samples, uTexRT1);

	float d = 0.0;

	// top left
	d += clamp( samples[0].x - samples[4].x, 0.0, 1.0 );
	d += clamp( samples[4].y - samples[0].y, 0.0, 1.0 );
	// top right
	d += clamp( samples[4].x - samples[2].x, 0.0, 1.0 );
	d += clamp( samples[4].y - samples[2].y, 0.0, 1.0 );
	// bottom left
	d += clamp( samples[6].x - samples[4].x, 0.0, 1.0 );
	d += clamp( samples[6].y - samples[4].y, 0.0, 1.0 );
	// bottom right
	d += clamp( samples[4].x - samples[8].x, 0.0, 1.0 );
	d += clamp( samples[8].y - samples[4].y, 0.0, 1.0 );

	return 1.0 - clamp((d / 4.0), 0.0, 1.0);
}

float nrmDiffDir8()
{
	vec3 samples[9];
	getNrmSamples(samples, uTexRT1);

	float d = 0.0;

	// top
	d += max( samples[4].y - samples[1].y, 0.0 );
	// left
	d += max( samples[3].x - samples[4].x, 0.0 );
	// right
	d += max( samples[4].x - samples[5].x, 0.0 );
	// bottom
	d += max( samples[7].y - samples[4].y, 0.0 );

	// top left
	d += max( samples[0].x - samples[4].x, 0.0 );
	d += max( samples[4].y - samples[0].y, 0.0 );
	// top right
	d += max( samples[4].x - samples[2].x, 0.0 );
	d += max( samples[4].y - samples[2].y, 0.0 );
	// bottom left
	d += max( samples[6].x - samples[4].x, 0.0 );
	d += max( samples[6].y - samples[4].y, 0.0 );
	// bottom right
	d += max( samples[4].x - samples[8].x, 0.0 );
	d += max( samples[8].y - samples[4].y, 0.0 );
/*
	// //
	float d = 0.0;

	// top
	d += samples[4].y - samples[1].y;
	// left
	d += samples[3].x - samples[4].x;
	// right
	d += samples[4].x - samples[5].x;
	// bottom
	d += samples[7].y - samples[4].y;

	// top left
	d += samples[0].x - samples[4].x;
	d += samples[4].y - samples[0].y;
	// top right
	d += samples[4].x - samples[2].x;
	d += samples[4].y - samples[2].y;
	// bottom left
	d += samples[6].x - samples[4].x;
	d += samples[6].y - samples[4].y;
	// bottom right
	d += samples[4].x - samples[8].x;
	d += samples[8].y - samples[4].y;
	// //
*/
	// //
	// top
	float t = max( samples[4].y - samples[1].y, 0.0 );
	// left
	float l = max( samples[3].x - samples[4].x, 0.0 );
	// right
	float r = max( samples[4].x - samples[5].x, 0.0 );
	// bottom
	float b = max( samples[7].y - samples[4].y, 0.0 );

	// top left
	float  tlx = max( samples[0].x - samples[4].x, 0.0 );
	float tly = max( samples[4].y - samples[0].y, 0.0 );
	// top right
	float trx = max( samples[4].x - samples[2].x, 0.0 );
	float try = max( samples[4].y - samples[2].y, 0.0 );
	// bottom left
	float blx = max( samples[6].x - samples[4].x, 0.0 );
	float bly = max( samples[6].y - samples[4].y, 0.0 );
	// bottom right
	float brx = max( samples[4].x - samples[8].x, 0.0 );
	float bry = max( samples[8].y - samples[4].y, 0.0 );

	float tz = max( samples[4].z - samples[1].z, 0.0 );
	float trz = max( samples[4].z - samples[2].z, 0.0 );
	float rz = max( samples[4].z - samples[5].z, 0.0 );
/*
	d = t*2.0 + l*2.0 + r*2.0 + b*2.0 + tlx + tly + trx + try + blx + bly + brx + bry;
	return 1.0 - clamp((d / 2.0) - 0.3, 0.0, 1.0);
*/
	// //
/*
//	d = t*2.0 + r*2.0 + trx + try + tz*2.0 + trz + rz*2.0;
	d = t*2.0 + r*2.0 + trx + try;
	return 1.0 - clamp((d / 1.0) - 0.3, 0.0, 1.0);
*/
	// //
/*
	d = 0.0;
	d = max(d, t*2.0);
	d = max(d, l*2.0);
	d = max(d, r*2.0);
	d = max(d, b*2.0);
	d = max(d, tlx);
	d = max(d, tly);
	d = max(d, trx);
	d = max(d, try);
	d = max(d, blx);
	d = max(d, bly);
	d = max(d, brx);
	d = max(d, bry);
*/
	d = 0.0;
	d = max(d, t*2.0);
	d = max(d, r*2.0);
	d = max(d, trx);
	d = max(d, try);
	d = max(d, tz);
	d = max(d, trz);
	d = max(d, rz);

	return 1.0 - clamp( clamp((d / 1.0) - 0.3, 0.0, 1.0) * 2.0, 0.0, 1.0 );
}


float nrmDiffDir8_depth()
{
	vec3 samples[9];
	getNrmSamples(samples, uTexRT1);

	float depthSamples[9];
	getDepthSamples( depthSamples, uTexDepth );

	// top
	float t = max( samples[4].y - samples[1].y, 0.0 );
	// left
	float l = max( samples[3].x - samples[4].x, 0.0 );
	// right
	float r = max( samples[4].x - samples[5].x, 0.0 );
	// bottom
	float b = max( samples[7].y - samples[4].y, 0.0 );

	// top left
	float  tlx = max( samples[0].x - samples[4].x, 0.0 );
	float tly = max( samples[4].y - samples[0].y, 0.0 );
	// top right
	float trx = max( samples[4].x - samples[2].x, 0.0 );
	float try = max( samples[4].y - samples[2].y, 0.0 );
	// bottom left
	float blx = max( samples[6].x - samples[4].x, 0.0 );
	float bly = max( samples[6].y - samples[4].y, 0.0 );
	// bottom right
	float brx = max( samples[4].x - samples[8].x, 0.0 );
	float bry = max( samples[8].y - samples[4].y, 0.0 );

	float tz = max( samples[4].z - samples[1].z, 0.0 );
	float trz = max( samples[4].z - samples[2].z, 0.0 );
	float rz = max( samples[4].z - samples[5].z, 0.0 );

	float d = 0.0;
	d = max(d, t*2.0);
	d = max(d, r*2.0);
	d = max(d, trx);
	d = max(d, try);
	d = max(d, tz);
	d = max(d, trz);
	d = max(d, rz);

	float facing = 1.0;
	facing = min( facing, dot( samples[0], vec3(0.0, 0.0, 1.0) ) );
	facing = min( facing, dot( samples[1], vec3(0.0, 0.0, 1.0) ) );
	facing = min( facing, dot( samples[2], vec3(0.0, 0.0, 1.0) ) );
	facing = min( facing, dot( samples[3], vec3(0.0, 0.0, 1.0) ) );
	facing = min( facing, dot( samples[4], vec3(0.0, 0.0, 1.0) ) );
	facing = min( facing, dot( samples[5], vec3(0.0, 0.0, 1.0) ) );
	facing = min( facing, dot( samples[6], vec3(0.0, 0.0, 1.0) ) );
	facing = min( facing, dot( samples[7], vec3(0.0, 0.0, 1.0) ) );
	facing = min( facing, dot( samples[8], vec3(0.0, 0.0, 1.0) ) );
	facing = max( 0.0, facing );

	float dt = abs( depthSamples[4] - depthSamples[1] );
	float dtr = abs( depthSamples[4] - depthSamples[2] );
	float dr = abs( depthSamples[4] - depthSamples[5] );

	float d2 = (dt + dtr + dr);
	if( d2 > 0.00004 && facing < 0.1 )
		return 0.0;
	else
		return 1.0;

	d = max( d, d2 );

//	return 1.0 - clamp( clamp((d / 1.0) - 0.4, 0.0, 1.0) * 2.0, 0.0, 1.0 );
	return 1.0 - clamp( d2, 0.0, 1.0 );
}


float stalker()
{
	// normals filter
	vec3 normals[9];
	getNrmSamples(normals, uTexRT1);

	// top
	float t = max( normals[4].y - normals[1].y, 0.0 );
	// left
	float l = max( normals[3].x - normals[4].x, 0.0 );
	// right
	float r = max( normals[4].x - normals[5].x, 0.0 );
	// bottom
	float b = max( normals[7].y - normals[4].y, 0.0 );

	// top left
	float  tlx = max( normals[0].x - normals[4].x, 0.0 );
	float tly = max( normals[4].y - normals[0].y, 0.0 );
	// top right
	float trx = max( normals[4].x - normals[2].x, 0.0 );
	float try = max( normals[4].y - normals[2].y, 0.0 );
	// bottom left
	float blx = max( normals[6].x - normals[4].x, 0.0 );
	float bly = max( normals[6].y - normals[4].y, 0.0 );
	// bottom right
	float brx = max( normals[4].x - normals[8].x, 0.0 );
	float bry = max( normals[8].y - normals[4].y, 0.0 );

	float tz = max( normals[4].z - normals[1].z, 0.0 );
	float trz = max( normals[4].z - normals[2].z, 0.0 );
	float rz = max( normals[4].z - normals[5].z, 0.0 );

	float nf = 0.0;
	nf = max(nf, t*2.0);
	nf = max(nf, r*2.0);
	nf = max(nf, trx);
	nf = max(nf, try);
	nf = max(nf, tz);
	nf = max(nf, trz);
	nf = max(nf, rz);

	nf = clamp( clamp((nf / 1.0) - 0.3, 0.0, 1.0) * 2.0, 0.0, 1.0 );

	// depth filter
	float depth[9];
	getDepthSamples( depth, uTexDepth );

	vec4 dd;
	dd.x = depth[0] + depth[8];
	dd.y = depth[2] + depth[6];
	dd.z = depth[3] + depth[5];
	dd.w = depth[1] + depth[7];

	dd = 2 * depth[4] - dd;
	dd = max( vec4(0.0), dd );
	dd = dd - 0.00001;
	dd = step( dd, vec4(0.0) );
	float df = 1.0 - dot( dd, vec4(0.25) );

	float f = 1.0 - clamp( nf + df, 0.0, 1.0 );
	return clamp( f, 0.0, 1.0 );
}


float outlineFinal()
{
	vec3 samples[9];
	getNrmSamples(samples, uTexRT1);

	float depthSamples[9];
	getDepthSamples( depthSamples, uTexDepth );

	// top
	vec3 t = samples[4] - samples[1];
	// right
	vec3 r = samples[4] - samples[5];
	// top right
	vec3 tr = samples[4] - samples[2];
/*
	t = max( t, vec3(0.0) );
	r = max( r, vec3(0.0) );
	tr = max( tr, vec3(0.0) );
*/
	t = abs( t );
	r = abs( r );
	tr = abs( tr );
	
	float n = 0.0;
/*
	n = max(n, ty*2.0);
	n = max(n, rx*2.0);
	n = max(n, trx);
	n = max(n, try);
	n = max(n, tz);
	n = max(n, trz);
	n = max(n, rz);
	n = 1.0 - clamp( clamp((n * 1.0) - 0.5, 0.0, 1.0) * 2.0, 0.0, 1.0 );
*/
	// //

	n = max( n, t.x );
	n = max( n, t.y );
	n = max( n, t.z );
	n = max( n, r.x );
	n = max( n, r.y );
	n = max( n, r.z );
	n = max( n, tr.x );
	n = max( n, tr.y );
	n = max( n, tr.z );
	n = 1.0 - clamp( clamp((n * 2.0) - 0.8, 0.0, 1.0) * 1.5, 0.0, 1.0 );

	// //
/*
	n = length(t) + length(r) + length(tr) * 2.0;
	n = 1.0 - clamp( clamp((n * 0.6) - 0.9, 0.0, 1.0) * 2.0, 0.0, 1.0 );
*/
	// //
/*
	n = max(max( length(t), length(r) ), length(tr) );
	n = 1.0 - clamp( clamp((n * 1.5) - 0.9, 0.0, 1.0) * 1.0, 0.0, 1.0 );
*/

	float d = 0.0;
	d = max( d, step( 1.0, depthSamples[0] ) - step( 1.0, depthSamples[4] ) );
	d = max( d, step( 1.0, depthSamples[1] ) - step( 1.0, depthSamples[4] ) );
	d = max( d, step( 1.0, depthSamples[2] ) - step( 1.0, depthSamples[4] ) );
	d = max( d, step( 1.0, depthSamples[3] ) - step( 1.0, depthSamples[4] ) );
	d = max( d, step( 1.0, depthSamples[4] ) - step( 1.0, depthSamples[4] ) );
	d = max( d, step( 1.0, depthSamples[5] ) - step( 1.0, depthSamples[4] ) );
	d = max( d, step( 1.0, depthSamples[6] ) - step( 1.0, depthSamples[4] ) );
	d = max( d, step( 1.0, depthSamples[7] ) - step( 1.0, depthSamples[4] ) );

	d = 1.0 - d;

	return min( n, d );
}


void main(void)
{
//	float outline = nrmDiff3();
//	float outline = nrmDiff8();
//	float outline = lapl3();
//	float outline = dot8();
//	float outline = nrmDiff2C();

//	float outline = nrmDiffDir8();
//	float outline = nrmDiffDir8_depth();
//	float outline = stalker();
	float outline = outlineFinal();

	fsOut0.rgb = texture(uTexDif, fsInUV).rgb * (0.1 + 0.9*outline);

//	fsOut0.rgb = vec3(outline);
//	fsOut0.rgb = texture( uTexRT1, fsInUV ).rgb;
//	fsOut0.rgb = vec3(fsInUV.x, fsInUV.y, 0.0);

	float depth = sampleDepth( uTexDepth, vec2(0.0) );
//	fsOut0.rgb = vec3(depth);
}