/* dynamic lighting stuff.
 */
 
 
#pragma define MAX_LIGHTS


uniform vec3 uAmbLightColor;
// x = number of dir lights
// y = number of dir lights + point lights (total lights)
uniform vec2 uNumLights;
// vec is position or direction, depending on the light type.
// for directional lights, it's the direction to the light.
uniform vec4 uLightVec[MAX_LIGHTS];
uniform vec3 uLightColor[MAX_LIGHTS];
// for point lights:
// x = max radius squared
uniform vec4 uLightParams[MAX_LIGHTS];

uniform float uSpecExp;


float pointLightAtten( in int li, in float distSq )
{
	float atten = 1.0 - clamp( distSq / uLightParams[li].x, 0.0, 1.0 );
	return atten * atten;
}

void pointLightAttenClrAndLightVec
	(	in int li, in vec3 vCPos,
		out vec3 clr, out vec3 toLight )
{
	toLight = uLightVec[li].xyz - vCPos;
	float distSq = length(toLight);
	distSq *= distSq;
	float atten = pointLightAtten( li, distSq );
	toLight /= sqrt(distSq);
	clr = uLightColor[li]* atten;
}


void lightDiffLambert
	(	in vec3 clr, in vec3 nrm, in vec3 toLight,
		inout vec3 diffuse )
{
	float lambert = dot( nrm, toLight );
	lambert = max( 0.0, lambert );
	
	diffuse += vec3( lambert * clr );
}

void lightDiffHalfLambert
	(	in vec3 clr, in vec3 nrm, in vec3 toLight,
		inout vec3 diffuse )
{
	float lambert = dot( nrm, toLight );
	lambert = lambert * 0.5 + 0.5;
	
	diffuse += vec3( lambert * clr );
}

void lightDiffWaxLambert
	(	in vec3 clr, in vec3 nrm, in vec3 toLight, in float waxiness,
		inout vec3 diffuse )
{
	float lambert = dot( nrm, toLight );
	lambert = max( 0.0, lambert );
	lambert = waxiness + (1.0 - waxiness) * lambert;
	
	diffuse += vec3( lambert * clr );
}


void lightSpec
	(	in vec3 clr, in vec3 toCam, in vec3 nrm, in vec3 toLight,
		inout vec3 specular )
{
	vec3 rVec = reflect( -normalize(toLight), nrm );
	float s = max( 0.0, dot(toCam, rVec) );
	s = pow( s, uSpecExp );
	
	specular += clr * s;
}


// backface lighting scattered forward.
// http://en.wikibooks.org/wiki/GLSL_Programming/Unity/Translucent_Surfaces
void lightFwdScatter
	(	in vec3 clr, in vec3 toCam, in vec3 toLight, in float sharpnessExp,
		inout vec3 scatter )
{
	float s = max( 0.0, dot(-toLight, toCam) );
	s = pow( s, sharpnessExp );
	
	scatter += clr * s;
}


// accumulate all lighting
void lightAccum
	(	in vec3 vCPos, in vec3 nrm,
		inout vec3 diffuse, inout vec3 specular )
{
	vec3 toCam = normalize( -vCPos );
	
	int li;
	for( li = 0; li < int(uNumLights.x); li++ )
	{
		lightDiffLambert( uLightColor[li], nrm, uLightVec[li].xyz, diffuse );
		lightSpec( uLightColor[li], toCam, nrm, uLightVec[li].xyz, specular );
	}

	// point lights come after directional lights
	for( ; li < int(uNumLights.y); li++ )
	{
		vec3 clr;
		vec3 toLight;
		pointLightAttenClrAndLightVec( li, vCPos, clr, toLight );
		
		lightDiffLambert( clr, nrm, toLight, diffuse );
		lightSpec( clr, toCam, nrm, toLight, specular );
	}

	diffuse += uAmbLightColor;
}


void lightAccumXlucFront
	(	in vec3 vCPos, in vec3 nrm,
		inout vec3 diffuse, inout vec3 specular )
{
	vec3 toCam = normalize( -vCPos );
	
	int li;
	for( li = 0; li < int(uNumLights.x); li++ )
	{
		lightDiffHalfLambert( uLightColor[li], nrm, uLightVec[li].xyz, diffuse );
		lightSpec( uLightColor[li], toCam, nrm, uLightVec[li].xyz, specular );
	}

	// point lights come after directional lights
	for( ; li < int(uNumLights.y); li++ )
	{
		vec3 clr;
		vec3 toLight;
		pointLightAttenClrAndLightVec( li, vCPos, clr, toLight );
		
		lightDiffHalfLambert( clr, nrm, toLight, diffuse );
		lightSpec( clr, toCam, nrm, toLight, specular );
	}

	diffuse += uAmbLightColor;
}


// http://www.gamedev.net/topic/481494-sub-surface-scattering-hack--comments-wanted-/
void lightAccumXlucBack
	(	in vec3 vCPos, in vec3 nrm,
		inout vec3 diffuse, inout vec3 specular )
{
	vec3 toCamNeg = normalize( vCPos );
	
	int li;
	for( li = 0; li < int(uNumLights.x); li++ )
	{
		lightDiffLambert( uLightColor[li], -nrm, uLightVec[li].xyz, diffuse );
		lightDiffHalfLambert( uLightColor[li], toCamNeg, uLightVec[li].xyz, specular );
	}

	// point lights come after directional lights
	for( ; li < int(uNumLights.y); li++ )
	{
		vec3 clr;
		vec3 toLight;
		pointLightAttenClrAndLightVec( li, vCPos, clr, toLight );
		
		lightDiffLambert( clr, -nrm, toLight, diffuse );
		lightDiffHalfLambert( clr, toCamNeg, toLight, specular );
	}

	diffuse += uAmbLightColor;
}