#version 120
#ifdef GL_ES
  #define LOWP lowp
  precision mediump float;
#else
  #define LOWP
#endif

varying vec4 v_color;
varying vec2 v_texCoords;
uniform sampler2D u_texture;

// scaling:
uniform int scale;
uniform vec2 output_resolution;
uniform vec2 input_resolution;
uniform float distortion;
uniform float base_frame_size;
uniform float input_gamma;
uniform float output_gamma;

float rect_dist(vec2 uv, vec2 tl, vec2 br) {
    float dx = max(tl.x - uv.x, uv.x - br.x);
    float dy = max(tl.y - uv.y, uv.y - br.y);
    dx = max(0.0, dx);
    dy = max(0.0, dy);
    return sqrt(dx*dx + dy*dy);
}

vec3 reflect(vec2 pxy, float frame_dist) {
    // amount to contract reflection
	float refl_contract = 0.3;

	// amount to offset reflection *towards* screen
	float refl_offset = 0.02;

	float dist = (frame_dist * (refl_contract * frame_dist)) + refl_offset;

    float xcast = mix(pxy.x, 0.5, dist);
    vec2 adj_xy = vec2(xcast, pxy.y);

    return vec3(adj_xy, dist);
}

vec2 curvature(vec2 coord) {
    coord *= input_resolution / input_resolution;
    vec2 cc = coord - 0.5;
    float dist = dot(cc, cc) * distortion;
    return (coord + cc * (1.0 + dist) * dist) * input_resolution / input_resolution;
}

vec4 scanline(float distance, vec4 color) {
    vec4 wid = 2.0 + 2.0 * pow(color, vec4(4.0));

    vec4 weights = vec4(distance / 0.3);
    return 2.6 * exp(-pow(weights * inversesqrt(0.5 * wid), wid)) / (0.6 + 0.2 * wid);
}

vec4 crt(vec2 xy) {
    vec2 ratio_scale = xy * input_resolution - vec2(0.5);
    vec2 uv_ratio = fract(ratio_scale);

    xy.y = (floor(ratio_scale.y) + 0.5) / input_resolution.y;

    vec2 one = 1.0 / input_resolution;

    vec4 a = pow(texture2D(u_texture, (xy)), vec4(input_gamma));
    vec4 b = pow(texture2D(u_texture, (xy + vec2(0.0, one.y))), vec4(input_gamma));

    vec4 scan_a  = a * scanline(uv_ratio.y, a);
    vec4 scan_b = b * scanline(1.0 - uv_ratio.y, b);
    vec3 scan  = (scan_a + scan_b).rgb;

    float mask_factor = v_texCoords.x * input_resolution.x * output_resolution.x / input_resolution.x;
    vec3 mask = mix(vec3(1.0, 0.7, 1.0), vec3(0.7, 1.0, 0.7), floor(mod(mask_factor, 2.0)));

    scan *= mask;

    return vec4(pow(scan, vec3(1.0 / output_gamma)), 1.0);
}

vec4 blur5CRT(vec2 uv, vec2 resolution, vec2 direction) {
  vec4 color = vec4(0.0);
  vec2 off1 = vec2(1.3333333333333333) * direction;
  color += crt(uv) * 0.29411764705882354;
  color += crt(uv + (off1 / resolution)) * 0.35294117647058826;
  color += crt(uv - (off1 / resolution)) * 0.35294117647058826;
  return color;
}

void main() {
    vec2 xy = curvature(v_texCoords * (1.0 + base_frame_size * 2) - base_frame_size);

    float frame_dist = rect_dist(xy, vec2(0.0, 0.0), vec2(1.0, 1.0)) * 30;

    if (frame_dist > 0) {
        vec3 lecu;
        float corner_fade;
        float edge_fade;

        if (xy.x > 1.0 - base_frame_size / 40 || xy.x < 0 + base_frame_size / 40) {
            lecu = reflect(xy, frame_dist);
            corner_fade = 3;
            edge_fade = 1.25;
        } else {
            lecu = reflect(xy.yx, frame_dist).yxz;
            corner_fade = 3;
            edge_fade = 2;
        }

        xy = lecu.xy;

        float adj_dist = lecu.z;
        vec2 flipped_xy = abs(0.5 - xy);
        float corner_dist = (flipped_xy.x * flipped_xy.y) * corner_fade;

        vec4 black = vec4(0.0, 0.0, 0.0, 1.0);

        vec4 blurred = blur5CRT(xy, output_resolution / 4, vec2(0.0, 1.0));
        vec4 faded = mix(blurred, black, (adj_dist * edge_fade) + corner_dist);
        vec4 darkened = mix(faded, black, 0.5);

        gl_FragColor = darkened;
    } else {
        gl_FragColor = crt(xy);
    }
}
