import vs

#==================================================================================================
def AddValidObjectToList( objectList, obj ):
    if ( obj != None ): objectList.append( obj )
    

#==================================================================================================
def HideControlGroups( rig, rootGroup, *groupNames ):
    for name in groupNames:    
        group = rootGroup.FindChildByName( name, False )
        if ( group != None ):
            rig.HideControlGroup( group )

    
#==================================================================================================
# Create the reverse foot control and operators for the foot on the specified side
#==================================================================================================
def CreateReverseFoot( controlName, sideName, gameModel, animSet, shot, helperControlGroup, footControlGroup ) :
    
    # Cannot create foot controls without heel position, so check for that first
    heelAttachName = "pvt_heel_" + sideName
    if ( gameModel.FindAttachment( heelAttachName ) == 0 ):
        print "Could not create foot control " + controlName + ", model is missing heel attachment point: " + heelAttachName;
        return None
    
    footRollDefault = 0.5
    rotationAxis = vs.Vector( 1, 0, 0 )
        
    # Construct the name of the dag nodes of the foot and toe for the specified side
    footName = "rig_foot_" + sideName
    toeName = "rig_toe_" + sideName    
    
    # Get the world space position and orientation of the foot and toe
    footPos = sfm.GetPosition( footName )
    footRot = sfm.GetRotation( footName )
    toePos = sfm.GetPosition( toeName )
    
    # Setup the reverse foot hierarchy such that the foot is the parent of all the foot transforms, the 
    # reverse heel is the parent of the heel, so it can be used for rotations around the ball of the 
    # foot that will move the heel, the heel is the parent of the foot IK handle so that it can perform
    # rotations around the heel and move the foot IK handle, resulting in moving all the foot bones.
    # root
    #   + rig_foot_R
    #       + rig_knee_R
    #       + rig_reverseHeel_R
    #           + rig_heel_R
    #               + rig_footIK_R
    
  
    # Construct the reverse heel joint this will be used to rotate the heel around the toe, and as
    # such is positioned at the toe, but using the rotation of the foot which will be its parent, 
    # so that it has no local rotation once parented to the foot.
    reverseHeelName = "rig_reverseHeel_" + sideName
    reverseHeelDag = sfm.CreateRigHandle( reverseHeelName, pos=toePos, rot=footRot, rotControl=False )
    sfmUtils.Parent( reverseHeelName, footName, vs.REPARENT_LOGS_OVERWRITE )
    
    
    
    # Construct the heel joint, this will be used to rotate the foot around the back of the heel so it 
    # is created at the heel location (offset from the foot) and also given the rotation of its parent.
    heelName = "rig_heel_" + sideName
    vecHeelPos = gameModel.ComputeAttachmentPosition( heelAttachName )
    heelPos = [ vecHeelPos.x, vecHeelPos.y, vecHeelPos.z ]     
    heelRot = sfm.GetRotation( reverseHeelName )
    heelDag = sfm.CreateRigHandle( heelName, pos=heelPos, rot=heelRot, posControl=True, rotControl=False )
    sfmUtils.Parent( heelName, reverseHeelName, vs.REPARENT_LOGS_OVERWRITE )
    
    # Create the ik handle which will be used as the target for the ik chain for the leg
    ikHandleName = "rig_footIK_" + sideName   
    ikHandleDag = sfmUtils.CreateHandleAt( ikHandleName, footName )
    sfmUtils.Parent( ikHandleName, heelName, vs.REPARENT_LOGS_OVERWRITE )
                    
    # Create an orient constraint which causes the toe's orientation to match the foot's orientation
    footRollControlName = controlName + "_" + sideName
    toeOrientTarget = sfm.OrientConstraint( footName, toeName, mo=True, controls=False )
    footRollControl, footRollValue = sfmUtils.CreateControlledValue( footRollControlName, "value", vs.AT_FLOAT, footRollDefault, animSet, shot )
    
    # Create the expressions to re-map the footroll slider value for use in the constraint and rotation operators
    toeOrientExprName = "expr_toeOrientEnable_" + sideName    
    toeOrientExpr = sfmUtils.CreateExpression( toeOrientExprName, "inrange( footRoll, 0.5001, 1.0 )", animSet )
    toeOrientExpr.SetValue( "footRoll", footRollDefault )
    
    toeRotateExprName = "expr_toeRotation_" + sideName
    toeRotateExpr = sfmUtils.CreateExpression( toeRotateExprName, "max( 0, (footRoll - 0.5) ) * 140", animSet )
    toeRotateExpr.SetValue( "footRoll", footRollDefault )
                            
    heelRotateExprName = "expr_heelRotation_" + sideName
    heelRotateExpr = sfmUtils.CreateExpression( heelRotateExprName, "max( 0, (0.5 - footRoll) ) * -100", animSet )
    heelRotateExpr.SetValue( "footRoll", footRollDefault )
        
    # Create a connection from the footroll value to all of the expressions that require it
    footRollConnName = "conn_footRoll_" + sideName
    footRollConn = sfmUtils.CreateConnection( footRollConnName, footRollValue, "value", animSet )
    footRollConn.AddOutput( toeOrientExpr, "footRoll" )
    footRollConn.AddOutput( toeRotateExpr, "footRoll" )
    footRollConn.AddOutput( heelRotateExpr, "footRoll" )
    
    # Create the connection from the toe orientation enable expression to the target weight of the 
    # toe orientation constraint, this will turn the constraint on an off based on the footRoll value
    toeOrientConnName = "conn_toeOrientExpr_" + sideName;
    toeOrientConn = sfmUtils.CreateConnection( toeOrientConnName, toeOrientExpr, "result", animSet )
    toeOrientConn.AddOutput( toeOrientTarget, "targetWeight" )
    
    # Create a rotation constraint to drive the toe rotation and connect its input to the 
    # toe rotation expression and connect its output to the reverse heel dag's orientation
    toeRotateConstraintName = "rotationConstraint_toe_" + sideName
    toeRotateConstraint = sfmUtils.CreateRotationConstraint( toeRotateConstraintName, rotationAxis, reverseHeelDag, animSet )
    
    toeRotateExprConnName = "conn_toeRotateExpr_" + sideName
    toeRotateExprConn = sfmUtils.CreateConnection( toeRotateExprConnName, toeRotateExpr, "result", animSet )
    toeRotateExprConn.AddOutput( toeRotateConstraint, "rotations", 0 );

    # Create a rotation constraint to drive the heel rotation and connect its input to the 
    # heel rotation expression and connect its output to the heel dag's orientation 
    heelRotateConstraintName = "rotationConstraint_heel_" + sideName
    heelRotateConstraint = sfmUtils.CreateRotationConstraint( heelRotateConstraintName, rotationAxis, heelDag, animSet )
    
    heelRotateExprConnName = "conn_heelRotateExpr_" + sideName
    heelRotateExprConn = sfmUtils.CreateConnection( heelRotateExprConnName, heelRotateExpr, "result", animSet )
    heelRotateExprConn.AddOutput( heelRotateConstraint, "rotations", 0 )
    
    if ( helperControlGroup != None ):
        sfmUtils.AddDagControlsToGroup( helperControlGroup, reverseHeelDag, ikHandleDag, heelDag )       
    
    if ( footControlGroup != None ):
        footControlGroup.AddControl( footRollControl )
        
    return ikHandleDag


#==================================================================================================
# Compute the direction from boneA to boneB
#==================================================================================================
def ComputeVectorBetweenBones( boneA, boneB, scaleFactor ):
    
    vPosA = vs.Vector( 0, 0, 0 )
    boneA.GetAbsPosition( vPosA )
    
    vPosB = vs.Vector( 0, 0, 0 )
    boneB.GetAbsPosition( vPosB )
    
    vDir = vs.Vector( 0, 0, 0 )
    vs.mathlib.VectorSubtract( vPosB, vPosA, vDir )
    vDir.NormalizeInPlace()
    
    vScaledDir = vs.Vector( 0, 0, 0 )
    vs.mathlib.VectorScale( vDir, scaleFactor, vScaledDir )    
    
    return vScaledDir
    
   
#==================================================================================================
# Build a simple ik rig for the currently selected animation set
#==================================================================================================
def BuildRig():
    
    # Get the currently selected animation set and shot
    shot = sfm.GetCurrentShot()
    animSet = sfm.GetCurrentAnimationSet()
    gameModel = animSet.gameModel
    rootGroup = animSet.GetRootControlGroup()
    
    # Start the biped rig to which all of the controls and constraints will be added
    rig = sfm.BeginRig( "rig_biped_" + animSet.GetName() );
    if ( rig == None ):
        return
    
    # Change the operation mode to passthrough so changes chan be made temporarily
    sfm.SetOperationMode( "Pass" )
    
    # Move everything into the reference pose
    sfm.SelectAll()
    sfm.SetReferencePose()
    
    #==============================================================================================
    # Find the dag nodes for all of the bones in the model which will be used by the script
    #==============================================================================================
    boneRoot      = sfmUtils.FindFirstDag( [ "RootTransform" ], True )
    bonePelvis    = sfmUtils.FindFirstDag( [ "spine",		"Bip01_Pelvis",		"bip_pelvis" ], True )
    boneSpine0    = sfmUtils.FindFirstDag( [ "spine.001",		"Bip01_Spine",		"bip_spine_0" ], True )
    boneSpine1    = sfmUtils.FindFirstDag( [ "spine.002",		"Bip01_Spine1",		"bip_spine_1" ], True )
    boneSpine2    = sfmUtils.FindFirstDag( [ "spine.003",		"Bip01_Spine2",		"bip_spine_2" ], True ) 
    boneSpine3    = sfmUtils.FindFirstDag( [ "spine.004",		"Bip01_Spine3",		"bip_spine_3",	"ValveBiped.Bip01_Spine4" ], True )
    boneNeck      = sfmUtils.FindFirstDag( [ "spine.005",		"Bip01_Neck",		"bip_neck_0",	"bip_neck", "ValveBiped.Bip01_Neck1" ], True )
    boneHead      = sfmUtils.FindFirstDag( [ "spine.006",		"Bip01_Head",		"bip_head",		"ValveBiped.Bip01_Head1" ], True )
    
    boneUpperLegR = sfmUtils.FindFirstDag( [ "thigh.R",    "Bip01_R_Thigh",    "bip_hip_R" ], True )
    boneLowerLegR = sfmUtils.FindFirstDag( [ "shin.R",     "Bip01_R_Calf",     "bip_knee_R" ], True )
    boneFootR     = sfmUtils.FindFirstDag( [ "foot.R",     "Bip01_R_Foot",     "bip_foot_R" ], True )
    boneToeR      = sfmUtils.FindFirstDag( [ "toe.R",     "Bip01_R_Toe0",     "bip_toe_R" ], True ) 
    boneCollarR   = sfmUtils.FindFirstDag( [ "shoulder.R", "Bip01_R_Clavicle", "bip_collar_R" ], True )   
    boneUpperArmR = sfmUtils.FindFirstDag( [ "upper_arm.R", "Bip01_R_UpperArm", "bip_upperArm_R" ], True ) 
    boneLowerArmR = sfmUtils.FindFirstDag( [ "forearm.R",  "Bip01_R_Forearm",  "bip_lowerArm_R" ], True )
    boneHandR     = sfmUtils.FindFirstDag( [ "hand.R",     "Bip01_R_Hand",     "bip_hand_R" ], True )
   
    boneUpperLegL = sfmUtils.FindFirstDag( [ "thigh.L",    "Bip01_L_Thigh",    "bip_hip_L" ], True )
    boneLowerLegL = sfmUtils.FindFirstDag( [ "shin.L",     "Bip01_L_Calf",     "bip_knee_L" ], True )
    boneFootL     = sfmUtils.FindFirstDag( [ "foot.L",     "Bip01_L_Foot",     "bip_foot_L" ], True )
    boneToeL      = sfmUtils.FindFirstDag( [ "toe.L",     "Bip01_L_Toe0",     "bip_toe_L" ], True ) 
    boneCollarL   = sfmUtils.FindFirstDag( [ "shoulder.L", "Bip01_L_Clavicle", "bip_collar_L" ], True )        
    boneUpperArmL = sfmUtils.FindFirstDag( [ "upper_arm.L", "Bip01_L_UpperArm", "bip_upperArm_L" ], True ) 
    boneLowerArmL = sfmUtils.FindFirstDag( [ "forearm.L",  "Bip01_L_Forearm",  "bip_lowerArm_L" ], True ) 
    boneHandL     = sfmUtils.FindFirstDag( [ "hand.L",     "Bip01_L_Hand",     "bip_hand_L" ], True )
    
    boneFinger00r = sfmUtils.FindFirstDag( [ "thumb.01.R" ], True )
    boneFinger01r = sfmUtils.FindFirstDag( [ "thumb.02.R" ], True )
    boneFinger02r = sfmUtils.FindFirstDag( [ "thumb.03.R" ], True )
    boneFinger10r = sfmUtils.FindFirstDag( [ "f_index.01.R" ], True )
    boneFinger11r = sfmUtils.FindFirstDag( [ "f_index.02.R" ], True )
    boneFinger12r = sfmUtils.FindFirstDag( [ "f_index.03.R" ], True )
    boneFinger20r = sfmUtils.FindFirstDag( [ "f_middle.01.R" ], True )
    boneFinger21r = sfmUtils.FindFirstDag( [ "f_middle.02.R" ], True )
    boneFinger22r = sfmUtils.FindFirstDag( [ "f_middle.03.R" ], True )
    boneFinger30r = sfmUtils.FindFirstDag( [ "f_ring.01.R" ], True )
    boneFinger31r = sfmUtils.FindFirstDag( [ "f_ring.02.R" ], True )
    boneFinger32r = sfmUtils.FindFirstDag( [ "f_ring.03.R" ], True )
    boneFinger40r = sfmUtils.FindFirstDag( [ "f_pinky.01.R" ], True )
    boneFinger41r = sfmUtils.FindFirstDag( [ "f_pinky.02.R" ], True )
    boneFinger42r = sfmUtils.FindFirstDag( [ "f_pinky.03.R" ], True )
    
    boneFinger00l = sfmUtils.FindFirstDag( [ "thumb.01.L" ], True )
    boneFinger01l = sfmUtils.FindFirstDag( [ "thumb.02.L" ], True )
    boneFinger02l = sfmUtils.FindFirstDag( [ "thumb.03.L" ], True )
    boneFinger10l = sfmUtils.FindFirstDag( [ "f_index.01.L" ], True )
    boneFinger11l = sfmUtils.FindFirstDag( [ "f_index.02.L" ], True )
    boneFinger12l = sfmUtils.FindFirstDag( [ "f_index.03.L" ], True )
    boneFinger20l = sfmUtils.FindFirstDag( [ "f_middle.01.L" ], True )
    boneFinger21l = sfmUtils.FindFirstDag( [ "f_middle.02.L" ], True )
    boneFinger22l = sfmUtils.FindFirstDag( [ "f_middle.03.L" ], True )
    boneFinger30l = sfmUtils.FindFirstDag( [ "f_ring.01.L" ], True )
    boneFinger31l = sfmUtils.FindFirstDag( [ "f_ring.02.L" ], True )
    boneFinger32l = sfmUtils.FindFirstDag( [ "f_ring.03.L" ], True )
    boneFinger40l = sfmUtils.FindFirstDag( [ "f_pinky.01.L" ], True )
    boneFinger41l = sfmUtils.FindFirstDag( [ "f_pinky.02.L" ], True )
    boneFinger42l = sfmUtils.FindFirstDag( [ "f_pinky.03.L" ], True )
	
    #==============================================================================================
    # Create the rig handles and constrain them to existing bones
    #==============================================================================================
    rigRoot    = sfmUtils.CreateConstrainedHandle( "rig_root",     boneRoot,    bCreateControls=False )
    rigPelvis  = sfmUtils.CreateConstrainedHandle( "rig_pelvis",   bonePelvis,  bCreateControls=False )
    rigSpine0  = sfmUtils.CreateConstrainedHandle( "rig_spine_0",  boneSpine0,  bCreateControls=False )
    rigSpine1  = sfmUtils.CreateConstrainedHandle( "rig_spine_1",  boneSpine1,  bCreateControls=False )
    rigSpine2  = sfmUtils.CreateConstrainedHandle( "rig_spine_2",  boneSpine2,  bCreateControls=False )
    rigChest   = sfmUtils.CreateConstrainedHandle( "rig_chest",    boneSpine3,  bCreateControls=False )
    rigNeck    = sfmUtils.CreateConstrainedHandle( "rig_neck",     boneNeck,    bCreateControls=False )
    rigHead    = sfmUtils.CreateConstrainedHandle( "rig_head",     boneHead,    bCreateControls=False )
    
    rigFootR   = sfmUtils.CreateConstrainedHandle( "rig_foot_R",   boneFootR,   bCreateControls=False )
    rigToeR    = sfmUtils.CreateConstrainedHandle( "rig_toe_R",    boneToeR,    bCreateControls=False )
    rigCollarR = sfmUtils.CreateConstrainedHandle( "rig_collar_R", boneCollarR, bCreateControls=False )
    rigHandR   = sfmUtils.CreateConstrainedHandle( "rig_hand_R",   boneHandR,   bCreateControls=False )
    rigFootL   = sfmUtils.CreateConstrainedHandle( "rig_foot_L",   boneFootL,   bCreateControls=False )
    rigToeL    = sfmUtils.CreateConstrainedHandle( "rig_toe_L",    boneToeL,    bCreateControls=False )
    rigCollarL = sfmUtils.CreateConstrainedHandle( "rig_collar_L", boneCollarL, bCreateControls=False )
    rigHandL   = sfmUtils.CreateConstrainedHandle( "rig_hand_L",   boneHandL,   bCreateControls=False )
        
    rigFinger00r   = sfmUtils.CreateConstrainedHandle( "rig_finger00_r",   boneFinger00r,   bCreateControls=False )
    rigFinger01r   = sfmUtils.CreateConstrainedHandle( "rig_finger01_r",   boneFinger01r,   bCreateControls=False )
    rigFinger02r   = sfmUtils.CreateConstrainedHandle( "rig_finger02_r",   boneFinger02r,   bCreateControls=False )
    rigFinger10r   = sfmUtils.CreateConstrainedHandle( "rig_finger10_r",   boneFinger10r,   bCreateControls=False )
    rigFinger11r   = sfmUtils.CreateConstrainedHandle( "rig_finger11_r",   boneFinger11r,   bCreateControls=False )
    rigFinger12r   = sfmUtils.CreateConstrainedHandle( "rig_finger12_r",   boneFinger12r,   bCreateControls=False )
    rigFinger20r   = sfmUtils.CreateConstrainedHandle( "rig_finger20_r",   boneFinger20r,   bCreateControls=False )
    rigFinger21r   = sfmUtils.CreateConstrainedHandle( "rig_finger21_r",   boneFinger21r,   bCreateControls=False )
    rigFinger22r   = sfmUtils.CreateConstrainedHandle( "rig_finger22_r",   boneFinger22r,   bCreateControls=False )
    rigFinger30r   = sfmUtils.CreateConstrainedHandle( "rig_finger30_r",   boneFinger30r,   bCreateControls=False )
    rigFinger31r   = sfmUtils.CreateConstrainedHandle( "rig_finger31_r",   boneFinger31r,   bCreateControls=False )
    rigFinger32r   = sfmUtils.CreateConstrainedHandle( "rig_finger32_r",   boneFinger32r,   bCreateControls=False )
    rigFinger40r   = sfmUtils.CreateConstrainedHandle( "rig_finger40_r",   boneFinger40r,   bCreateControls=False )
    rigFinger41r   = sfmUtils.CreateConstrainedHandle( "rig_finger41_r",   boneFinger41r,   bCreateControls=False )
    rigFinger42r   = sfmUtils.CreateConstrainedHandle( "rig_finger42_r",   boneFinger42r,   bCreateControls=False )
    
    rigFinger00l   = sfmUtils.CreateConstrainedHandle( "rig_finger00_l",   boneFinger00l,   bCreateControls=False )
    rigFinger01l   = sfmUtils.CreateConstrainedHandle( "rig_finger01_l",   boneFinger01l,   bCreateControls=False )
    rigFinger02l   = sfmUtils.CreateConstrainedHandle( "rig_finger02_l",   boneFinger02l,   bCreateControls=False )
    rigFinger10l   = sfmUtils.CreateConstrainedHandle( "rig_finger10_l",   boneFinger10l,   bCreateControls=False )
    rigFinger11l   = sfmUtils.CreateConstrainedHandle( "rig_finger11_l",   boneFinger11l,   bCreateControls=False )
    rigFinger12l   = sfmUtils.CreateConstrainedHandle( "rig_finger12_l",   boneFinger12l,   bCreateControls=False )
    rigFinger20l   = sfmUtils.CreateConstrainedHandle( "rig_finger20_l",   boneFinger20l,   bCreateControls=False )
    rigFinger21l   = sfmUtils.CreateConstrainedHandle( "rig_finger21_l",   boneFinger21l,   bCreateControls=False )
    rigFinger22l   = sfmUtils.CreateConstrainedHandle( "rig_finger22_l",   boneFinger22l,   bCreateControls=False )
    rigFinger30l   = sfmUtils.CreateConstrainedHandle( "rig_finger30_l",   boneFinger30l,   bCreateControls=False )
    rigFinger31l   = sfmUtils.CreateConstrainedHandle( "rig_finger31_l",   boneFinger31l,   bCreateControls=False )
    rigFinger32l   = sfmUtils.CreateConstrainedHandle( "rig_finger32_l",   boneFinger32l,   bCreateControls=False )
    rigFinger40l   = sfmUtils.CreateConstrainedHandle( "rig_finger40_l",   boneFinger40l,   bCreateControls=False )
    rigFinger41l   = sfmUtils.CreateConstrainedHandle( "rig_finger41_l",   boneFinger41l,   bCreateControls=False )
    rigFinger42l   = sfmUtils.CreateConstrainedHandle( "rig_finger42_l",   boneFinger42l,   bCreateControls=False )
    
    # Use the direction from the heel to the toe to compute the knee offsets, 
    # this makes the knee offset indpendent of the inital orientation of the model.
    vKneeOffsetR = ComputeVectorBetweenBones( boneFootR, boneToeR, 10 )
    vKneeOffsetL = ComputeVectorBetweenBones( boneFootL, boneToeL, 10 )
    
    
    rigKneeR   = sfmUtils.CreateOffsetHandle( "rig_knee_R",  boneLowerLegR, vKneeOffsetR,  bCreateControls=False )   
    rigKneeL   = sfmUtils.CreateOffsetHandle( "rig_knee_L",  boneLowerLegL, vKneeOffsetL,  bCreateControls=False )
    rigElbowR  = sfmUtils.CreateOffsetHandle( "rig_elbow_R", boneLowerArmR, -vKneeOffsetR,  bCreateControls=False )
    rigElbowL  = sfmUtils.CreateOffsetHandle( "rig_elbow_L", boneLowerArmL, -vKneeOffsetL,  bCreateControls=False )
    
    # Create a helper handle which will remain constrained to the each foot position that can be used for parenting.
    rigFootHelperR = sfmUtils.CreateConstrainedHandle( "rig_footHelper_R", boneFootR, bCreateControls=False )
    rigFootHelperL = sfmUtils.CreateConstrainedHandle( "rig_footHelper_L", boneFootL, bCreateControls=False )
    
    # Create a list of all of the rig dags
    allRigHandles = [ rigRoot, rigPelvis, rigSpine0, rigSpine1, rigSpine2, rigChest, rigNeck, rigHead,
                      rigCollarR, rigElbowR, rigHandR, rigKneeR, rigFootR, rigToeR,
                      rigCollarL, rigElbowL, rigHandL, rigKneeL, rigFootL, rigToeL,
                      rigFinger00r, rigFinger01r, rigFinger02r,
                      rigFinger10r, rigFinger11r, rigFinger12r,
                      rigFinger20r, rigFinger21r, rigFinger22r,
                      rigFinger30r, rigFinger31r, rigFinger32r,
                      rigFinger40r, rigFinger41r, rigFinger42r,
                      rigFinger00l, rigFinger01l, rigFinger02l,
                      rigFinger10l, rigFinger11l, rigFinger12l,
                      rigFinger20l, rigFinger21l, rigFinger22l,
                      rigFinger30l, rigFinger31l, rigFinger32l,
                      rigFinger40l, rigFinger41l, rigFinger42l					  ];
    
    
    #==============================================================================================
    # Generate the world space logs for the rig handles and remove the constraints
    #==============================================================================================
    sfm.ClearSelection()
    sfmUtils.SelectDagList( allRigHandles )
    sfm.GenerateSamples()
    sfm.RemoveConstraints()    
    
    
    #==============================================================================================
    # Build the rig handle hierarchy
    #==============================================================================================
    sfmUtils.ParentMaintainWorld( rigPelvis,        rigRoot )
    sfmUtils.ParentMaintainWorld( rigSpine0,        rigPelvis )
    sfmUtils.ParentMaintainWorld( rigSpine1,        rigSpine0 )
    sfmUtils.ParentMaintainWorld( rigSpine2,        rigSpine1 )
    sfmUtils.ParentMaintainWorld( rigChest,         rigSpine2 )
    sfmUtils.ParentMaintainWorld( rigNeck,          rigChest )
    sfmUtils.ParentMaintainWorld( rigHead,          rigNeck )
    
    sfmUtils.ParentMaintainWorld( rigFootHelperR,   rigRoot )
    sfmUtils.ParentMaintainWorld( rigFootHelperL,   rigRoot )
    sfmUtils.ParentMaintainWorld( rigFootR,         rigRoot )
    sfmUtils.ParentMaintainWorld( rigFootL,         rigRoot )
    sfmUtils.ParentMaintainWorld( rigKneeR,         rigFootR )
    sfmUtils.ParentMaintainWorld( rigKneeL,         rigFootL )
    sfmUtils.ParentMaintainWorld( rigToeR,          rigFootHelperR )
    sfmUtils.ParentMaintainWorld( rigToeL,          rigFootHelperL )
    
    sfmUtils.ParentMaintainWorld( rigCollarR,       rigChest )
    sfmUtils.ParentMaintainWorld( rigElbowR,        rigCollarR )
    sfmUtils.ParentMaintainWorld( rigHandR,	    rigRoot )
    sfmUtils.ParentMaintainWorld( rigCollarL,       rigChest )
    sfmUtils.ParentMaintainWorld( rigElbowL,        rigCollarL )
    sfmUtils.ParentMaintainWorld( rigHandL,	    rigRoot )
	
    sfmUtils.ParentMaintainWorld( rigFinger00r,	    boneHandR )
    sfmUtils.ParentMaintainWorld( rigFinger01r,	    rigFinger00r )
    sfmUtils.ParentMaintainWorld( rigFinger02r,	    rigFinger01r )
    sfmUtils.ParentMaintainWorld( rigFinger10r,	    boneHandR )
    sfmUtils.ParentMaintainWorld( rigFinger11r,	    rigFinger10r )
    sfmUtils.ParentMaintainWorld( rigFinger12r,	    rigFinger11r )
    sfmUtils.ParentMaintainWorld( rigFinger20r,	    boneHandR )
    sfmUtils.ParentMaintainWorld( rigFinger21r,	    rigFinger20r )
    sfmUtils.ParentMaintainWorld( rigFinger22r,	    rigFinger21r )
    sfmUtils.ParentMaintainWorld( rigFinger30r,	    boneHandR )
    sfmUtils.ParentMaintainWorld( rigFinger31r,	    rigFinger30r )
    sfmUtils.ParentMaintainWorld( rigFinger32r,	    rigFinger31r )
    sfmUtils.ParentMaintainWorld( rigFinger40r,	    boneHandR )
    sfmUtils.ParentMaintainWorld( rigFinger41r,	    rigFinger40r )
    sfmUtils.ParentMaintainWorld( rigFinger42r,	    rigFinger41r )
    
    sfmUtils.ParentMaintainWorld( rigFinger00l,	    boneHandL )
    sfmUtils.ParentMaintainWorld( rigFinger01l,	    rigFinger00l )
    sfmUtils.ParentMaintainWorld( rigFinger02l,	    rigFinger01l )
    sfmUtils.ParentMaintainWorld( rigFinger10l,	    boneHandL )
    sfmUtils.ParentMaintainWorld( rigFinger11l,	    rigFinger10l )
    sfmUtils.ParentMaintainWorld( rigFinger12l,	    rigFinger11l )
    sfmUtils.ParentMaintainWorld( rigFinger20l,	    boneHandL )
    sfmUtils.ParentMaintainWorld( rigFinger21l,	    rigFinger20l )
    sfmUtils.ParentMaintainWorld( rigFinger22l,	    rigFinger21l )
    sfmUtils.ParentMaintainWorld( rigFinger30l,	    boneHandL )
    sfmUtils.ParentMaintainWorld( rigFinger31l,	    rigFinger30l )
    sfmUtils.ParentMaintainWorld( rigFinger32l,	    rigFinger31l )
    sfmUtils.ParentMaintainWorld( rigFinger40l,	    boneHandL )
    sfmUtils.ParentMaintainWorld( rigFinger41l,	    rigFinger40l )
    sfmUtils.ParentMaintainWorld( rigFinger42l,	    rigFinger41l )
    
    # Create the hips control, this allows a pelvis rotation that does not effect the spine,
    # it is only used for rotation so a position control is not created. Additionally add the
    # new control to the selection so the that set default call operates on it too.
    rigHips = sfmUtils.CreateHandleAt( "rig_hips", rigPelvis, False, True )
    sfmUtils.Parent( rigHips, rigPelvis, vs.REPARENT_LOGS_OVERWRITE )
    sfm.SelectDag( rigHips )

    # Set the defaults of the rig transforms to the current locations. Defaults are stored in local
    # space, so while the parent operation tries to preserve default values it is cleaner to just
    # set them once the final hierarchy is constructed.
    sfm.SetDefault()
    
    
    #==============================================================================================
    # Create the reverse foot controls for both the left and right foot
    #==============================================================================================
    rigLegsGroup = rootGroup.CreateControlGroup( "RigLegs" )
    rigHelpersGroup = rootGroup.CreateControlGroup( "RigHelpers" )
    rigHelpersGroup.SetVisible( False )
    rigHelpersGroup.SetSnappable( False )
    
    footIKTargetR = rigFootR
    footIkTargetL = rigFootL
    
    if ( gameModel != None ) :
        footRollIkTargetR = CreateReverseFoot( "rig_footRoll", "R", gameModel, animSet, shot, rigHelpersGroup, rigLegsGroup )
        footRollIkTargetL = CreateReverseFoot( "rig_footRoll", "L", gameModel, animSet, shot, rigHelpersGroup, rigLegsGroup )
        if ( footRollIkTargetR != None ) :
            footIKTargetR = footRollIkTargetR
        if ( footRollIkTargetL != None ) :
            footIkTargetL = footRollIkTargetL
    
    
    #==============================================================================================
    # Create constraints to drive the bone transforms using the rig handles
    #==============================================================================================
    
    # The following bones are simply constrained directly to a rig handle
    sfmUtils.CreatePointOrientConstraint( rigRoot,      boneRoot        )
    sfmUtils.CreatePointOrientConstraint( rigHips,      bonePelvis      )
    sfmUtils.CreatePointOrientConstraint( rigSpine0,    boneSpine0      )
    sfmUtils.CreatePointOrientConstraint( rigSpine1,    boneSpine1      )
    sfmUtils.CreatePointOrientConstraint( rigSpine2,    boneSpine2      )
    sfmUtils.CreatePointOrientConstraint( rigChest,     boneSpine3      )
    sfmUtils.CreatePointOrientConstraint( rigNeck,      boneNeck        )
    sfmUtils.CreatePointOrientConstraint( rigHead,      boneHead        )
    sfmUtils.CreatePointOrientConstraint( rigCollarR,   boneCollarR     )
    sfmUtils.CreatePointOrientConstraint( rigCollarL,   boneCollarL     )
    sfmUtils.CreatePointOrientConstraint( rigToeR,      boneToeR        )
    sfmUtils.CreatePointOrientConstraint( rigToeL,      boneToeL        )
	
    sfmUtils.CreatePointOrientConstraint( rigFinger00r,      boneFinger00r        )
    sfmUtils.CreatePointOrientConstraint( rigFinger01r,      boneFinger01r        )
    sfmUtils.CreatePointOrientConstraint( rigFinger02r,      boneFinger02r        )
    sfmUtils.CreatePointOrientConstraint( rigFinger10r,      boneFinger10r        )
    sfmUtils.CreatePointOrientConstraint( rigFinger11r,      boneFinger11r        )
    sfmUtils.CreatePointOrientConstraint( rigFinger12r,      boneFinger12r        )
    sfmUtils.CreatePointOrientConstraint( rigFinger20r,      boneFinger20r        )
    sfmUtils.CreatePointOrientConstraint( rigFinger21r,      boneFinger21r        )
    sfmUtils.CreatePointOrientConstraint( rigFinger22r,      boneFinger22r        )
    sfmUtils.CreatePointOrientConstraint( rigFinger30r,      boneFinger30r        )
    sfmUtils.CreatePointOrientConstraint( rigFinger31r,      boneFinger31r        )
    sfmUtils.CreatePointOrientConstraint( rigFinger32r,      boneFinger32r        )
    sfmUtils.CreatePointOrientConstraint( rigFinger40r,      boneFinger40r        )
    sfmUtils.CreatePointOrientConstraint( rigFinger41r,      boneFinger41r        )
    sfmUtils.CreatePointOrientConstraint( rigFinger42r,      boneFinger42r        )
    
    sfmUtils.CreatePointOrientConstraint( rigFinger00l,      boneFinger00l        )
    sfmUtils.CreatePointOrientConstraint( rigFinger01l,      boneFinger01l        )
    sfmUtils.CreatePointOrientConstraint( rigFinger02l,      boneFinger02l        )
    sfmUtils.CreatePointOrientConstraint( rigFinger10l,      boneFinger10l        )
    sfmUtils.CreatePointOrientConstraint( rigFinger11l,      boneFinger11l        )
    sfmUtils.CreatePointOrientConstraint( rigFinger12l,      boneFinger12l        )
    sfmUtils.CreatePointOrientConstraint( rigFinger20l,      boneFinger20l        )
    sfmUtils.CreatePointOrientConstraint( rigFinger21l,      boneFinger21l        )
    sfmUtils.CreatePointOrientConstraint( rigFinger22l,      boneFinger22l        )
    sfmUtils.CreatePointOrientConstraint( rigFinger30l,      boneFinger30l        )
    sfmUtils.CreatePointOrientConstraint( rigFinger31l,      boneFinger31l        )
    sfmUtils.CreatePointOrientConstraint( rigFinger32l,      boneFinger32l        )
    sfmUtils.CreatePointOrientConstraint( rigFinger40l,      boneFinger40l        )
    sfmUtils.CreatePointOrientConstraint( rigFinger41l,      boneFinger41l        )
    sfmUtils.CreatePointOrientConstraint( rigFinger42l,      boneFinger42l        )
    
    # Create ik constraints for the arms and legs that will control the rotation of the hip / knee and 
    # upper arm / elbow joints based on the position of the foot and hand respectively.
    sfmUtils.BuildArmLeg( rigKneeR,  footIKTargetR, boneUpperLegR,  boneFootR, True )
    sfmUtils.BuildArmLeg( rigKneeL,  footIkTargetL, boneUpperLegL,  boneFootL, True )
    sfmUtils.BuildArmLeg( rigElbowR, rigHandR,      boneUpperArmR,  boneHandR, True )
    sfmUtils.BuildArmLeg( rigElbowL, rigHandL,      boneUpperArmL,  boneHandL, True )
    
    
    #==============================================================================================
    # Create handles for the important attachment points 
    #==============================================================================================    
    attachmentGroup = rootGroup.CreateControlGroup( "Attachments" )  
    attachmentGroup.SetVisible( False )
    
    sfmUtils.CreateAttachmentHandleInGroup( "pvt_heel_R",       attachmentGroup )
    sfmUtils.CreateAttachmentHandleInGroup( "pvt_toe_R",        attachmentGroup )
    sfmUtils.CreateAttachmentHandleInGroup( "pvt_outerFoot_R",  attachmentGroup )
    sfmUtils.CreateAttachmentHandleInGroup( "pvt_innerFoot_R",  attachmentGroup )
    
    sfmUtils.CreateAttachmentHandleInGroup( "pvt_heel_L",       attachmentGroup )
    sfmUtils.CreateAttachmentHandleInGroup( "pvt_toe_L",        attachmentGroup )
    sfmUtils.CreateAttachmentHandleInGroup( "pvt_outerFoot_L",  attachmentGroup )
    sfmUtils.CreateAttachmentHandleInGroup( "pvt_innerFoot_L",  attachmentGroup )
    
    
    
    #==============================================================================================
    # Re-organize the selection groups
    #==============================================================================================  
    rigBodyGroup = rootGroup.CreateControlGroup( "RigBody" )
    rigArmsGroup = rootGroup.CreateControlGroup( "RigArms" )
    
    RightArmGroup = rootGroup.CreateControlGroup( "RightArm" )
    LeftArmGroup = rootGroup.CreateControlGroup( "LeftArm" )
    RightLegGroup = rootGroup.CreateControlGroup( "RightLeg" )
    LeftLegGroup = rootGroup.CreateControlGroup( "LeftLeg" )   
    
    RightFingersGroup = rootGroup.CreateControlGroup( "RightFingers" )
    LeftFingersGroup = rootGroup.CreateControlGroup( "LeftFingers" )
	
    sfmUtils.AddDagControlsToGroup( rigBodyGroup, rigRoot, rigPelvis, rigHips, rigSpine0, rigSpine1, rigSpine2, rigChest, rigNeck, rigHead )  
      
    rigArmsGroup.AddChild( RightArmGroup )
    rigArmsGroup.AddChild( LeftArmGroup )
    sfmUtils.AddDagControlsToGroup( RightArmGroup,  rigHandR, rigElbowR, rigCollarR )
    sfmUtils.AddDagControlsToGroup( LeftArmGroup, rigHandL, rigElbowL, rigCollarL )
    
    RightArmGroup.AddChild( RightFingersGroup )
    sfmUtils.AddDagControlsToGroup( RightFingersGroup, rigFinger00r, rigFinger01r, rigFinger02r, rigFinger10r, rigFinger11r, rigFinger12r, rigFinger20r, rigFinger21r, rigFinger22r, rigFinger30r, rigFinger31r, rigFinger32r, rigFinger40r, rigFinger41r, rigFinger42r )
    
    LeftArmGroup.AddChild( LeftFingersGroup )
    sfmUtils.AddDagControlsToGroup( LeftFingersGroup, rigFinger00l, rigFinger01l, rigFinger02l, rigFinger10l, rigFinger11l, rigFinger12l, rigFinger20l, rigFinger21l, rigFinger22l, rigFinger30l, rigFinger31l, rigFinger32l, rigFinger40l, rigFinger41l, rigFinger42l )
	
    rigLegsGroup.AddChild( RightLegGroup )
    rigLegsGroup.AddChild( LeftLegGroup )
    sfmUtils.AddDagControlsToGroup( RightLegGroup, rigKneeR, rigFootR, rigToeR )
    sfmUtils.AddDagControlsToGroup( LeftLegGroup, rigKneeL, rigFootL, rigToeL )
    
    sfmUtils.MoveControlGroup( "rig_footRoll_L", rigLegsGroup, LeftLegGroup )
    sfmUtils.MoveControlGroup( "rig_footRoll_R", rigLegsGroup, RightLegGroup )
    

 
    sfmUtils.AddDagControlsToGroup( rigHelpersGroup, rigFootHelperR, rigFootHelperL )

    # Set the control group visiblity, this is done through the rig so it can track which
    # groups it hid, so they can be set back to being visible when the rig is detached.
    HideControlGroups( rig, rootGroup, "Body", "Arms", "Legs", "Unknown", "Other", "Root" )      
        
    #Re-order the groups
    fingersGroup = rootGroup.FindChildByName( "Fingers", False ) 
    rootGroup.MoveChildToBottom( rigBodyGroup )
    rootGroup.MoveChildToBottom( rigLegsGroup )    
    rootGroup.MoveChildToBottom( rigArmsGroup )      
    rootGroup.MoveChildToBottom( fingersGroup )  
       
    rightFingersGroup = rootGroup.FindChildByName( "RightFingers", True ) 
    if ( rightFingersGroup != None ):
        RightArmGroup.AddChild( rightFingersGroup )
        rightFingersGroup.SetSelectable( False )
                                
    leftFingersGroup = rootGroup.FindChildByName( "LeftFingers", True ) 
    if ( leftFingersGroup != None ):
        LeftArmGroup.AddChild( leftFingersGroup )
        leftFingersGroup.SetSelectable( False )
        

    #==============================================================================================
    # Set the selection groups colors
    #==============================================================================================
    topLevelColor = vs.Color( 0, 128, 255, 255 )
    RightColor = vs.Color( 255, 0, 0, 255 )
    LeftColor = vs.Color( 0, 255, 0, 255 )
    RightColorSoft = vs.Color( 200, 60, 60, 255 )
    LeftColorSoft = vs.Color( 60, 200, 60, 255 )
    
    rigBodyGroup.SetGroupColor( topLevelColor, False )
    rigArmsGroup.SetGroupColor( topLevelColor, False )
    rigLegsGroup.SetGroupColor( topLevelColor, False )
    attachmentGroup.SetGroupColor( topLevelColor, False )
    rigHelpersGroup.SetGroupColor( topLevelColor, False )
    
    RightArmGroup.SetGroupColor( RightColor, False )
    LeftArmGroup.SetGroupColor( LeftColor, False )
    RightLegGroup.SetGroupColor( RightColor, False )
    LeftLegGroup.SetGroupColor( LeftColor, False )
    
    RightFingersGroup.SetGroupColor( RightColorSoft, False )
    LeftFingersGroup.SetGroupColor( LeftColorSoft, False )
    
    # End the rig definition
    sfm.EndRig()
    return
    
#==================================================================================================
# Script entry
#==================================================================================================

# Construct the rig for the selected animation set
BuildRig();


    
    




